/*
 * Written by Kinsen Choy
 */

package kinsen.melee.Guns;
import kinsen.melee.Details;
import kinsen.melee.Utils;
import java.awt.Point;

/**
 * Gun - An abstract class for a gun and a class that handles the guns.
 */
public abstract class Gun
{
	// Data needed to store after calculations
	protected Point fireAt = null;
	protected double firePower = 0;

	// Data stored for overloaded predictPosition
	private Point predictedEnemyLocation;
	private double predictedEnemyHeading;
	private double predictedBulletDistance;

	/*
	 * calculateGun: Method to calculate where to fire and fire power;
	 *					requires initial fire power which might be decreased.
	 */
	public abstract void calculateGun(double initialPower);

	/*
	 * Overloaded calculateGun: Specifies whether firepower can change.
	 */
	public abstract void calculateGun(double initialPower, boolean canChangePower);

	/*
	 * recordData: Mehtod to allow data collection by gun
	 */
	public abstract void recordData(Details enemyDetails, Details myDetails, long time);

	/*
	 * getFireAt: Returns the point to fire at
	 */
	public Point getFireAt()
	{
		return fireAt;
	}

	/*
	 * getFirePower: Returns the fire power to fire
	 */
	public double getFirePower()
	{
		return firePower;
	}

	/*
	 * predictPosition: Predicts future location based on current enemy location and velocity,
	 * 					allows for enemy turning and uses iteration to find when bullet might hit a moving enemy
	 */
	protected Point predictPosition(double enemyStartX, double enemyStartY, double myX, double myY,
									double enemyStartHeading, double velocity, double turn, double bulletPower, double bulletDistance, int predictTime)
	{
		// Enemy has not moved yet
		double enemyX = enemyStartX, enemyY = enemyStartY;
		// Heading of enemy has not changed
		double enemyHeading = enemyStartHeading;
		// Velocity of bullet
		double bulletVelocity = 20 - 3 * bulletPower;
		// Number of ticks
		int time = 0;
		// If bullet distance to enemy is not decreasing then iteration is finished
		while (Utils.distance(myX, myY, enemyX, enemyY) > bulletDistance && (predictTime == -1 || time < predictTime))
		{
			bulletDistance += bulletVelocity;
			enemyX += Math.sin(Math.toRadians(enemyHeading)) * velocity;
			enemyY += Math.cos(Math.toRadians(enemyHeading)) * velocity;
			enemyHeading += turn;
			time++;
		}

		predictedEnemyLocation = new Point((int) enemyX, (int) enemyY);
		predictedEnemyHeading = enemyHeading;
		predictedBulletDistance = bulletDistance;

		return new Point((int) enemyX, (int) enemyY);
	}

	/*
	 * predictPostion: An overloaded method of predictPosition that allows for enemy direction and turn to oscillate
	 */
	protected Point predictPosition(double enemyStartX, double enemyStartY, double myX, double myY,
									double enemyStartHeading, double velocity, double turn, double bulletPower,
									int initialTime, int defaultTime, boolean useOscillator)
	{
		int direction = 1;
		// Initial prediction
		predictPosition(enemyStartX, enemyStartY, myX, myY, enemyStartHeading, velocity, turn, bulletPower, 0, initialTime);
		// While prediction does not reach enemy
		while (Utils.distance(myX, myY, predictedEnemyLocation.getX(), predictedEnemyLocation.getY()) > predictedBulletDistance)
		{
			direction *= -1;
			predictPosition(predictedEnemyLocation.getX(), predictedEnemyLocation.getY(), myX, myY, predictedEnemyHeading,
							velocity * direction, turn * direction, bulletPower, predictedBulletDistance, defaultTime);
		}

		return predictedEnemyLocation;
	}

	protected Point predictPosition(double enemyStartX, double enemyStartY, double myX, double myY,
									double enemyStartHeading, double velocity, double turn, double bulletPower)
	{
		return predictPosition(enemyStartX, enemyStartY, myX, myY, enemyStartHeading, velocity, turn, bulletPower, 0, -1);
	}
}