package nat.ether.wavesurfing;

import nat.*;
import nat.ether.stats.*;
import robocode.*;
import nat.ether.utils.*;
import java.awt.*;
import java.awt.geom.*;
import java.util.*;
import java.util.List;
import nat.ether.wavesurfing.MovementPredictor.PredictionStatus;

public class WaveSurfingEngine {
	public final StatsManager stats;
	public final WaveManager waves;

	public Hikari r;
	
	// Dynamic factor (though not dynamic yet)
	public int WALL_STICK = 160;
	public double desiredDistance = 550d;

	private final Rectangle2D realField, wallSmoothField;
	private final double N;
	private final double S;
	private final double E;
	private final double W;

	public Wave surfingWave = null;
	public List<Danger> dangerList;
	public float[] currentStats;
	public Point2D currentTarget = null;
	public boolean statsUpdated = false;
	
	public int movingDirection = 1;

	private class Danger implements Comparable<Danger> {
		Point2D gotoLocation, reachedPosition;
		double danger;
		
		public int compareTo(Danger d) {
			return (int) Math.signum(danger - d.danger);
		}
	}

	public WaveSurfingEngine(StatsManager stats, WaveManager manager, Hikari r) {
		this.stats = stats;
		this.waves = manager;
		this.r = r;

		realField = new Rectangle2D.Double(18d, 18d, r.getBattleFieldWidth() - 36d, r.getBattleFieldHeight() - 36d);
		wallSmoothField = new Rectangle2D.Double(25d, 25d, r.getBattleFieldWidth() - 50d, r.getBattleFieldHeight() - 50d);
		N = wallSmoothField.getMaxY();
		S = wallSmoothField.getMinY();
		E = wallSmoothField.getMaxX();
		W = wallSmoothField.getMinX();
	}

	public void onScannedRobot(ScannedRobotEvent e) {
		Wave surfWave = findFirstSurfableWave();
		
		if (surfWave != null) {
			if (surfWave != surfingWave || statsUpdated)  {
				currentTarget = getSurfingLocation(surfWave);
				
				statsUpdated = false;
				surfingWave = surfWave;
			}
			
			goToLocation(currentTarget);
		} else {
//			int movingDirection = (int) Math.signum(r.getVelocity()*Math.sin(e.getBearingRadians()));
//			if (movingDirection == 0)
//				movingDirection = 1;
//			double goAngle = M.getAngle(r.myLocation, r.enemyLocation) + M.HALF_PI*movingDirection;
			r.setTurnRightRadians(Math.cos(Math.PI + e.getBearingRadians()));
		}
	}
	
	public void onHitByBullet(HitByBulletEvent e) {
		Wave w = waves.findAndRemoveWave(e.getBullet());
		
		if (w == null) {
			r.out.println("Bullet detect on non-existant wave!");
			return;
		}

		Point2D loc = new Point2D.Double(e.getBullet().getX(), e.getBullet().getY());
		stats.registerHit(w.scan, w.getGuessFactor(M.getAngle(w.fireLocation, loc)));
		
		statsUpdated = true;
	}
	
	public void onBulletHitBullet(BulletHitBulletEvent e) {
		Wave w = waves.findAndRemoveWave(e.getHitBullet());
		
		if (w == null) {
			r.out.println("Bullet detect on non-existant wave!");
			return;
		}
		
		Point2D loc = new Point2D.Double(e.getBullet().getX(), e.getBullet().getY());
		stats.registerHit(w.scan, w.getGuessFactor(M.getAngle(w.fireLocation, loc)));
		
		statsUpdated = true;
	}
	
	private final Point2D getSurfingLocation(Wave surfWave) {
		List<Point2D> points = generatePoints(surfWave);

		PredictionStatus status = new PredictionStatus(r.myLocation.getX(), r.myLocation.getY(), r.getHeadingRadians(), r.getVelocity(), r.getTime());
		currentStats = getStats(surfWave);

		dangerList = new ArrayList<Danger>(points.size());
		for (Point2D pt : points) {
			dangerList.add(evaluateDanger(surfWave, pt, status, currentStats));
		}
		Collections.sort(dangerList);
		// TODO Surf more than one wave
		Point2D target = dangerList.get(0).gotoLocation;
		
		return target;
	}

	private final float[] getStats(Wave w) {
		return stats.getStatsBin(w.scan);
	}

	private final Danger evaluateDanger(Wave wave, Point2D pt, PredictionStatus begin, float[] stats) {
		Danger d = new Danger();
		Wave w = wave.clone();

		PredictionStatus current = begin;

		while (w.status != Wave.PASSED) {
			current = MovementPredictor.predict(current, M.getAngle(current, pt), getMaxVelocity(MovementPredictor.getHeading(M.getAngle(current, pt), current.heading)), current.distance(pt), realField);
			w.update(current);
		}
		

		d.reachedPosition = current;
		d.gotoLocation = pt;
		d.danger = BinUtils.getTotalInRect(stats, w.hitRange);
		return d;
	}

	private final Wave findFirstSurfableWave() {
		List<Wave> w = waves.getSurfableWaves(r.myLocation);
		if (w.size() > 0) {
			Wave min = w.get(0);
			double dist = Double.POSITIVE_INFINITY;
			for (Wave wave : w) {
				double travelDist = (wave.fireLocation.distance(r.myLocation) - wave.distanceTraveled) / wave.bulletVelocity;
				if (travelDist < dist) {
					dist = travelDist;
					min = wave;
				}
			}
			return min;
		}
		return null;
	}
	
	private final Wave findSecondSurfableWave() {
		Wave fw = findFirstSurfableWave();
		List<Wave> w = waves.getSurfableWaves(r.myLocation);
		if (w.size() > 0) {
			Wave min = w.get(0);
			double dist = Double.POSITIVE_INFINITY;
			for (Wave wave : w) {
				double travelDist = (wave.fireLocation.distance(r.myLocation) - wave.distanceTraveled) / wave.bulletVelocity;
				if (travelDist < dist && wave != fw) {
					dist = travelDist;
					min = wave;
				}
			}
			return min;
		}
		return null;
	}

	private final List<Point2D> generatePoints(Wave wave) {
		ArrayList<Point2D> result = new ArrayList<Point2D>();
		result.addAll(generatePoints(wave, 1));
		result.addAll(generatePoints(wave, -1));
		return result;
	}

	private final ArrayList<Point2D> generatePoints(Wave surfWave, final int direction) {
		PredictionStatus status = new PredictionStatus(r.myLocation.getX(), r.myLocation.getY(), r.getHeadingRadians(), r.getVelocity(), r.getTime());
		ArrayList<Point2D> traveledPoints = new ArrayList<Point2D>();
		PredictionStatus last = status;
		
		// Add current location too
		traveledPoints.add(status);
		
		int counter = 0;
		boolean intercepted = false;
		
		Point2D waveLocation = surfWave.fireLocation;

		do {
			double distance = status.distance(waveLocation);
			double offset = Math.PI / 2 - 1 + distance / 400;

			double moveAngle = wallSmoothing(status, M.getAngle(waveLocation, status) + (direction * (offset)), direction);

			status = MovementPredictor.predict(status, moveAngle, getMaxVelocity(MovementPredictor.getHeading(moveAngle, status.heading)), Double.POSITIVE_INFINITY, realField);
			if (last.distanceSq(status) > 50) {
				traveledPoints.add(status);
				last = status;
			}
			counter++;

			if (status.distance(surfWave.fireLocation) + 80 < surfWave.distanceTraveled + (counter * surfWave.bulletVelocity)) {
				intercepted = true;
			}
			
			waveLocation = M.project(waveLocation, surfWave.fireHeading, surfWave.fireVelocity);
		} while (!intercepted && counter < 500);

		return traveledPoints;
	}
	
	private final double getMaxVelocity(double turnRemaining) {
		//turnRemaining = Math.max(0, Math.abs(turnRemaining) - M.PI/4d);
		//return M.limit(1d, 2d + 6d * ((M.PI-turnRemaining) / M.PI), 8d);
		return 8d;
	}

	private final void goToLocation(Point2D pt) {
		double distance = r.myLocation.distance(pt);
		double angle = M.normalRelativeAngle(M.getAngle(r.myLocation, pt) - r.getHeadingRadians());
		if (Math.abs(angle) > Math.PI / 2) {
			distance = -distance;
			if (angle > 0) {
				angle -= Math.PI;
			} else {
				angle += Math.PI;
			}
		}

		r.setMaxVelocity(getMaxVelocity(MovementPredictor.getHeading(angle, r.getHeadingRadians())));
		r.setTurnRightRadians(angle * Math.signum(Math.abs((int) distance)));
		r.setAhead(distance);
	}

	// CREDIT: Simonton
	private final double wallSmoothing(Point2D l, double angle, int oDir) {
		angle = smoothWest(N - l.getY(), angle - M.HALF_PI, oDir) + M.HALF_PI;
		angle = smoothWest(E - l.getX(), angle + Math.PI, oDir) - Math.PI;
		angle = smoothWest(l.getY() - S, angle + M.HALF_PI, oDir) - M.HALF_PI;
		angle = smoothWest(l.getX() - W, angle, oDir);
		angle = smoothWest(l.getY() - S, angle + M.HALF_PI, oDir) - M.HALF_PI;
		angle = smoothWest(E - l.getX(), angle + Math.PI, oDir) - Math.PI;
		angle = smoothWest(N - l.getY(), angle - M.HALF_PI, oDir) + M.HALF_PI;

		return angle;
	}

	private final double smoothWest(double dist, double angle, int oDir) {
		if (dist < -WALL_STICK * Math.sin(angle)) {
			return Math.acos(oDir * dist / WALL_STICK) - oDir * M.HALF_PI;
		}
		return angle;
	}

	public void onPaint(Graphics2D g) {
		Wave w = findFirstSurfableWave();
		if (w != null) {
			g.setColor(Color.white);
			g.fill(M.point(currentTarget, 5));
			if (dangerList != null) {
				for (Danger pt : dangerList) {
					g.setColor(Color.getHSBColor(.66f - 0.66f * (float) pt.danger, 1f, 1f));
					g.fill(M.point(pt.reachedPosition, 2));
				}
			}
		}
	}
}
