package ag.neuralgir;

/**
 * A backward-propagation neural net
 * @author agrosse
 *
 */
public class NeuralNet {

	private int mNumLayers;
	private int[] mLayerSize;
	private double mLearningRate;
	private double mLearningRateChange;
	private int mBatchSize;
	private int mBatchCount;
	private double mMomentum;
	private int mNumTrainings;
	
	// mNeuron[h][i] = Neuron number i in layer h
	private double[][] mNeuron;
	// mNeuronError[h][i] = Error produced in mNeuron[h+1][i]
	private double[][] mNeuronError;
	// output of Neuron[][]
	private double[][] mNeuronOutput;
	
	/* 
	 * mWeights[h][i][k] = Weigth from Neuron number i in Layer h to
	 * Neuron number k in layer h + 1.
	 * For Layer 0 to layer 'mNumLayers - 2', the virtual neuron 
	 * "Neuron[mLayeSize[h]]" is a bias-neuron with constant output of 1.
	 */ 
	private double[][][] mWeights;
	private double[][][] mDeltaW;
	
	/**
	 * Constructor.
	 * @param data the neural net data
	 */
	public NeuralNet(NeuralNetData data) {
		mLayerSize = data.getLayerSize();
		mNumLayers = mLayerSize.length;

		mLearningRate = data.getLearningRate();
		mLearningRateChange = data.getLearningRateChange();
		mBatchSize = data.getBatchSize();
		mMomentum = data.getMomentum();
		mNumTrainings = data.getNumTrainings();	
		
		int h;
		mNeuron = new double[mNumLayers][];
		mNeuronOutput = new double[mNumLayers][];
		mNeuronError = new double[mNumLayers - 1][];
		
		for(h = 0; h < mNumLayers; h++){
			mNeuron[h] = new double[mLayerSize[h]]; 
			mNeuronOutput[h] = new double[mLayerSize[h]]; 			
		}
		
		
		// create neuron errors (bias neurons don't need to save
		// their error)
		for(h = 0; h < mNumLayers - 1; h++)
			mNeuronError[h] = new double[mLayerSize[h+1]]; 
		
		
		// ---- create deltaW ----
		mDeltaW = new double[mNumLayers - 1][][];
		
		// create weights
		for(h = 0; h < mNumLayers - 1; h++){				
			mDeltaW[h] = new double[mLayerSize[h]+1][mLayerSize[h+1]]; 
		}
		
		// ---- create weights ----
		if(data.getWeights() == null){
			mWeights = new double[mNumLayers - 1][][];
			
			for(h = 0; h < mNumLayers - 1; h++){				
				mWeights[h] = new double[mLayerSize[h]+1][mLayerSize[h+1]]; 
			}

			initRandomWeights();
		}
		else{
			mWeights = data.getWeights();
			init();
		}
		
	}
	
	/**
	 * Initialises the neural net.
	 *
	 */
	public void init(){
		mBatchCount = 0;
		
		int h, i, k;
		// Init Neurons
		for(h = 0; h < mNumLayers; h++)
			for(i = 0; i < mLayerSize[h]; i++){
				mNeuron[h][i] = .0; 
				mNeuronOutput[h][i] = .0;
			}
		
		// Init errors
		for(h = 0; h < mNumLayers - 1; h++)
			for(i = 0; i < mLayerSize[h+1]; i++)
				mNeuronError[h][i] = .0;
		
		// Init DeltaWs
		for(h = 0; h < mNumLayers - 1; h++){			
			for(i = 0; i < mLayerSize[h]+1; i++)
				for(k = 0; k < mLayerSize[h + 1]; k++){
					mDeltaW[h][i][k] = .0;
				}
		}	
	}
	
	/**
	 * Initialises the weights to random values.
	 */
	public void initRandomWeights(){
		init();
		int h, i, k;
		// Init weights with random numbers between 0 and 0.5
		for(h = 0; h < mNumLayers - 1; h++){			
			for(i = 0; i < mLayerSize[h]+1; i++)
				for(k = 0; k < mLayerSize[h + 1]; k++){
					mWeights[h][i][k] = Math.random() / 2;
				}
		}		
	}

	// ================= getters & setters =================
	

	/**
	 * Returns the batch size.
	 */
	public int getBatchSize() {
		return mBatchSize;
	}

	/**
	 * Returns the learning rate.
	 */
	public double getLearningRate() {
		return mLearningRate;
	}

	/**
	 * Returns the learning rate modification
	 */
	public double getLearningRateChange() {
		return mLearningRateChange;
	}

	/**
	 * Returns the momentum
	 */
	public double getMomentum() {
		return mMomentum;
	}
	
	/**
	 * Returns the number of trained data sets
	 */
	public int getNumTrainings(){
		return mNumTrainings;
	}
	
	/**
	 * Returns the weights.
	 */
	public double[][][] getWeights(){
		return mWeights;
	}
	
	/**
	 * Sets the batch size
	 */
	public void setBatchSize(int batchSize) {
		mBatchSize = batchSize;
		mBatchCount = 0;
	}

	/**
	 * Sets the learning rate
	 */
	public void setLearningRate(double learningRate) {
		mLearningRate = learningRate;
	}

	/**
	 * Sets the learning rate modification
	 */
	public void setLearningRateChange(double learningRateChange) {
		mLearningRateChange = learningRateChange;
	}

	/**
	 * Sets the momentum
	 */
	public void setMomentum(double momentum) {
		mMomentum = momentum;
	}
	
	/**
	 * Sets the weights
	 */
	public void setWeights(double[][][] weight){
		mWeights = weight;		
	}
	
	/**
	 * Returns the lazer sizes
	 * @return
	 */
	public int[] getLayerSize(){
		return mLayerSize;
	}
	
	//	 =====================================================
	
	/**
	 * Print the neural values
	 */
	public void printNeurons(){
		int h, i;
		System.out.println("--- Neurons ---");
		for(h = 0; h < mNumLayers; h++){
			System.out.println("Layer " + h + ":"); 
			for(i = 0; i < mLayerSize[h]; i++)
				System.out.print(mNeuron[h][i] + "  "); 

			System.out.print("\n"); 
		}		
	}
	
	/**
	 * Print the weight values
	 *
	 */
	public void printWeights(){
		int h, i, k;
		System.out.println("--- Weights ---");
		for(h = 0; h < mNumLayers - 2; h++){
			System.out.println("Layer " + h + ":"); 
			for(i = 0; i < mLayerSize[h]+1; i++){
				for(k = 0; k < mLayerSize[h + 1]; k++)
					System.out.print(mWeights[h][i][k] + "  ");
				System.out.print("\n"); 
			}
			System.out.print("\n"); 
		}
		h = mNumLayers - 2;
		System.out.println("Layer " + h + " (output):"); 
		for(i = 0; i < mLayerSize[h]; i++){
			for(k = 0; k < mLayerSize[h + 1]; k++)
				System.out.print(mWeights[h][i][k] + "  ");
			System.out.print("\n"); 
		}
		System.out.print("\n"); 
	}

	/**
	 * Print the layer sizes
	 *
	 */
	public void printLayerSize(){
		for(int i = 0; i < mNumLayers; i++)
			System.out.print(mLayerSize[i] + " - ");
		System.out.print("\n");
	}
		
	/**
	 * Calculates the output of the neural net
	 * @param input input data
	 * @return output
	 */
	public double[] calcOutput(double input[]) {
		mNeuron[0] = input;
		mNeuronOutput[0] = input;
		int h, i, k;
		double temp;
		for(h = 0; h < mNumLayers - 2; h++){
			// calculate weighted sum for all neurons in the next layer
			for(k = 0; k < mLayerSize[h+1]; k++){
				temp = .0;
				for(i = 0; i < mLayerSize[h]; i++)
					temp += mWeights[h][i][k] * mNeuronOutput[h][i];
				// add bias neuron output
				temp += mWeights[h][mLayerSize[h]][k];
				
				mNeuron[h+1][k] = temp;
				// use sigmoid activation function
				mNeuronOutput[h+1][k] = 1.0/(1.0 + Math.exp(-temp));
			}
		}
		
		// output layer uses the identity as activation function
		h = mNumLayers - 2;
		for(k = 0; k < mLayerSize[h + 1]; k++){
			temp = .0;
			for(i = 0; i < mLayerSize[h]; i++)
				temp += mWeights[h][i][k] * mNeuronOutput[h][i];
			// no bias neuron
			mNeuron[h+1][k] = temp;
			// use identity activation function
			mNeuronOutput[h+1][k] = temp;
		}
		
		return mNeuron[mNumLayers - 1];		
	}
	
	/**
	 * Calculates the mean square error. out1 and out2 must have the same length.
	 */	
	public double calcMSE(double out1[], double[] out2) {
		double error = .0;
		double diff;
		for(int i = 0; i < out1.length; i++) {
			diff= out1[i] - out2[i];
			error += diff * diff;
		}
		return error;
	}
		
	/**
	 * Trains the neural net
	 * @param input input data
	 * @param output correct output data
	 * @return the output of the neural net (before training)
	 */
	public double[] trainBackpropagation(double input[], double[] output) {
		double[] outputNet = calcOutput(input);
		int h, i, k;
		// calculate the error in the output layer
		h = mNumLayers - 1;
		for(i = 0; i < mLayerSize[h]; i++)
			mNeuronError[h-1][i] = output[i] - outputNet[i];

		// calculate the error of each Neuron in each hidden layer
		for(h = mNumLayers - 2; h > 0; h--){
			for(i = 0; i < mLayerSize[h]; i++){
				for(k = 0; k < mLayerSize[h+1]; k++)
					mNeuronError[h-1][i] += mWeights[h][i][k] * mNeuronError[h][k];
				mNeuronError[h-1][i] *= mNeuronOutput[h][i]*(1 - mNeuronOutput[h][i]);
			}
		}
		
		// calculate deltaW (weight adjust) for hidden layers
		for(h = mNumLayers - 2; h >= 0; h--){			
			// calculate deltaW for bias neurons
			i = mLayerSize[h];
			for(k = 0; k < mLayerSize[h+1]; k++)
				mDeltaW[h][i][k] = mMomentum * mDeltaW[h][i][k] + mNeuronError[h][k];					
						
			// calculate deltaW for normal neurons
			for(i = 0; i < mLayerSize[h]; i++)
				for(k = 0; k < mLayerSize[h+1]; k++)
					mDeltaW[h][i][k] = mMomentum * mDeltaW[h][i][k] + mNeuronError[h][k]*mNeuronOutput[h][i];		

		}
		
		mBatchCount++;
		
		if(mBatchCount == mBatchSize) {
			double temp = (double)1/mBatchSize;
			
			// adjust weights
			for(h = 0; h < mNumLayers - 1; h++){				
				for(i = 0; i < mLayerSize[h]+1; i++){
					for(k = 0; k < mLayerSize[h+1]; k++){
						mWeights[h][i][k] += mLearningRate * temp * mDeltaW[h][i][k];	
						mDeltaW[h][i][k] = .0;
					}					
				}
			}

			mBatchCount = 0;
			changeLearningRate();
		}
		mNumTrainings++;
		// return the output before training
		return outputNet;
	}
		
	/**
	 * Changes the learning rate according to the learnin rate 
	 * modification variable.
	 */
	private void changeLearningRate(){
		if(mLearningRate >= 0.1 - mLearningRateChange)
			mLearningRate = mLearningRate + mLearningRateChange; 
	}

	
}
