package pl.Patton.Strategies;

import robocode.*;

import pl.Abstract.*;
import pl.Enemy.AdvancedEnemy;
import pl.Patton.Guns.*;
import pl.Patton.Tanks.WaveSurfingTank;
import pl.Utilities.AngleUtils;

/*******************************************************************************
 * A strategy for duels.
 * 
 * Since: dev
 * 
 * Last Changed: 1.54
 * 
 * It uses wave surfing, two guns - Segmented Guess-Factor, and Predictive - and
 * an oscillating radar.
 ******************************************************************************/
public class Duel extends Strategy {
	// Track my enemy
	public static AdvancedEnemy enemy = new AdvancedEnemy();

	// I want to stay away from the walls
	public static final int WALL_MARGIN = 54;
	public int wall;

	// Virtual Gun Array - SGFT, PT
	public Gun[] gun;
	public int myGun;

	// My Tank movement using crappy WaveSurfing
	public WaveSurfingTank tank;

	public Duel(AdvancedRobot robot) {
		super(robot);
		// alreadyTriedToObtainInfo = false;
		gun = new Gun[] { new SGFTGun("Segmented Guess-Factor Gun", this.robot),
				new PTGun("Predictive Gun", this.robot),
				new PMGun("Pattern-Matching Gun", this.robot) };
		myGun = 0;
		tank = new WaveSurfingTank(this.robot);
	}

	public void run() {
		// Make myself flexible - very flexible
		robot.setAdjustRadarForRobotTurn(true);
		robot.setAdjustRadarForGunTurn(true);
		robot.setAdjustGunForRobotTurn(true);
		// Sometimes I want to switch guns in round...
		robot.addCustomEvent(new Condition("next_gun") {
			public boolean test() {
				// Basically, if my ratio drops too low, call it, but only if
				// I'm done testing each gun or I have previous info
				return gun[myGun].getGrade() < 0.25 && (robot.getRoundNum() >= gun.length * 2);
			}
		});
		// For debugging purposes: spit out my current gun
		robot.out.println("Using: " + gun[myGun].name + "\n");
		// Set my radar off on an infinite loop
		while (true) {
			robot.turnRadarRightRadians(Double.POSITIVE_INFINITY);
		}
	}

	public void onScannedRobot(ScannedRobotEvent e) {
		// Update enemy data
		enemy.update(e, robot);
		// Oscillate the radar
		robot.setTurnRadarRightRadians(AngleUtils.normalizeBearing(e.getBearingRadians()
				+ robot.getHeadingRadians() - robot.getRadarHeadingRadians()) * 2);
		// And do my gun
		doGun();
		// Do my movement
		doTank();
	}

	public void onDeath(DeathEvent e) {
		roundOver();
	}

	public void onBulletHit(BulletHitEvent e) {
		// Yay, my gun hit him
		gun[myGun].hits++;
		// Of course, his energy has changed...
		enemy.energy = e.getEnergy();
	}

	public void onBulletMissed(BulletMissedEvent e) {
		// Darn! I missed!
		gun[myGun].misses++;
	}

	public void onBulletHitBullet(BulletHitBulletEvent e) {
		// Well, I probably hit the bullet out of luck, but since I didn't
		// hit him...
		gun[myGun].misses++;
		tank.logBulletHitBullet(e);
	}

	public void onRobotDeath(RobotDeathEvent e) {
		// If I won this round
		if (robot.getOthers() == 0) {
			roundOver();
		}
	}

	public void onHitByBullet(HitByBulletEvent e) {
		// My enemy has recharged
		enemy.energy += e.getBullet().getPower() * 3.0;
		// I was hit
		tank.hits++;
		tank.logHitByBullet(e);
	}

	public void onHitRobot(HitRobotEvent e) {
		// Anyway, I want to get away from him...
		robot.setAhead(100 * -tank.moveDir);
	}

	public void onHitWall(HitWallEvent e) {
		robot.setAhead(100 * -tank.moveDir);
	}

	public void onCustomEvent(CustomEvent e) {
		// If I want to change guns in round, then do so
		if (e.getCondition().name.equals("next_gun")) {
			int oldGun = myGun;
			myGun = chooseBestGun();
			// For debugging purposes: if I switched a gun, say it
			if (oldGun != myGun)
				robot.out.println(robot.getTime() + ": Switching to: " + gun[myGun].name);
		}
	}

	/**
	 * Does the action I want when the round is over (for me, anyway). This
	 * includes when I win and when I lose.
	 */
	public void roundOver() {
		// During the first few rounds, test all my guns
		if (robot.getRoundNum() < gun.length * 2 - 1) {
			myGun = (robot.getRoundNum() + 1) / 2;
		}
		// After that, choose the best gun to use
		else
			myGun = chooseBestGun();
		// For debugging purposes: print out the score of each gun
		robot.out.println();
		for (int i = 0; i < gun.length; i++)
			robot.out.println(gun[i].name + ": " + gun[i].getGrade());
	}

	/**
	 * Decides which gun is best and gives it back.
	 * 
	 * @return the index of the new gun
	 */
	public int chooseBestGun() {
		int newGun = -1;
		// Choose the smallest grade first, that way anything is better
		double grade = Double.NEGATIVE_INFINITY;
		// A simple loop-through
		for (int i = 0; i < gun.length; i++)
			if (gun[i].getGrade() > grade) {
				grade = gun[i].getGrade();
				newGun = i;
			}
		return newGun;
	}

	/**
	 * Controls the gun for this strategy.
	 * 
	 * It calls the designated gun's doGun() method.
	 */
	public void doGun() {
		// Yeah, just call my current gun's run
		gun[myGun].run(enemy);
	}

	/**
	 * Controls the tank for this strategy.
	 * 
	 * It does crappy wave surfing.
	 */
	public void doTank() {
		// I'm probably moving away from the wall
		if (wall > 0)
			wall--;
		// And call my tank
		tank.run(enemy);
	}

}