package lucasslf.wavesurfing;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.List;

import lucasslf.LucasslfBot;
import lucasslf.utility.FastTrig;
import lucasslf.utility.LimitedList;
import lucasslf.utility.Utils;
import lucasslf.wavesurfing.buffer.SurfBuffer;
import lucasslf.wavesurfing.buffer.SurfBufferB;
import lucasslf.wavesurfing.buffer.SurfBufferG;
import lucasslf.wavesurfing.buffer.SurfBufferP;
import lucasslf.wavesurfing.buffer.SurfBufferQ;
import lucasslf.wavesurfing.buffer.SurfBufferS;
import lucasslf.wavesurfing.buffer.VirtualSurfBufferA;
import lucasslf.wavesurfing.buffer.VirtualSurfBufferB;
import lucasslf.wavesurfing.buffer.VirtualSurfBufferC;
import robocode.Bullet;
import robocode.BulletHitBulletEvent;
import robocode.BulletHitEvent;
import robocode.CustomEvent;
import robocode.HitByBulletEvent;
import robocode.HitRobotEvent;
import robocode.Rules;
import robocode.ScannedRobotEvent;

//Based on Basic Wave Surfer Tutorial and lots of other robots
//Thanks to all the Bot Authors in Robowiki!
public class BasicWaveSurfer extends WaveSurfingMovement {

	private int globalAwayMovementDirection = 1;
	private int globalSurfingMovementDirection = 1;

	private double lastBulletPower = 1.9;

	private static double bulletShotCount = 0;
	private static double bulletHitCount = 0;
	private static double hitRate = 0;

	// Wall Smoothing
	private static Rectangle2D.Double fieldRect = new java.awt.geom.Rectangle2D.Double(
			18, 18, 764, 564);
	private static double WALL_STICK = 160;
	public static final double WALL_MARGIN = 19;
	public static final double S = WALL_MARGIN;
	public static final double W = WALL_MARGIN;
	public static final double N = 600 - WALL_MARGIN;
	public static final double E = 800 - WALL_MARGIN;

	private static final int SAFE_DISTANCE = 100;

	// STATS
	public static final int BINS = 71;

	public static final int SEG_DISTANCE = 7;
	public static final int SEG_HEADING_DIFFERENCE = 5;
	public static final int SEG_VELOCITY = 5;
	public static final int SEG_BULLET_POWER = 4;
	public static final int SEG_WALL_PROXIMITY = 3;
	public static final int SEG_ACCEL = 3;
	public static final int SEG_MOVE_TIMES = 10;
	private static List<SurfBuffer> buffers = new ArrayList<SurfBuffer>();
	private static List<SurfBuffer> virtualWaveBuffers = new ArrayList<SurfBuffer>();

	// Surfing
	private List<Integer> surfDirections = new LimitedList<Integer>(3);
	private List<Double> surfAbsBearings = new LimitedList<Double>(3);
	private List<Double> lateralVelocities = new LimitedList<Double>(3);
	private List<Double> velocities = new LimitedList<Double>(3);
	private List<Double> distances = new LimitedList<Double>(3);
	private List<Double> headingDifferences = new LimitedList<Double>(3);
	private List<Double> advancingVelocities = new LimitedList<Double>(3);
	private List<Double> deltaBearings = new LimitedList<Double>(3);
	private List<Point2D> positions = new LimitedList<Point2D>(3);

	private int moveTimes = 0;

	private static final int MAX_CHECK_DANGER_RECURSION = 10;

	private boolean useVirtualBuffers = false;

	// Waves
	private List<EnemyWave> enemyWaves = new ArrayList<EnemyWave>();

	static {

		// BP: bullet power
		// HD: heading difference
		// LV: lateral velocity
		// D: distance at shot time
		// WP: Wall or Corner proximity
		// ACC: is accelerating, stopped, constant speed
		// ADV: advancing velocity
		// MT: Ticks since stopped last time
		// VSCT: velocity at scan time
		// VST: velocity at shot time

		buffers.add(new SurfBufferG(32)); // BP,WP, ACC, ADV, D 78,4 
		buffers.add(new SurfBufferS(32));// MT,HD,WP,ACC,D 59,42
		buffers.add(new SurfBufferB(32)); // BP, VSCT, VSHT,D 67,8
		buffers.add(new SurfBufferQ(32));// MT,HD,WP,LV,D
		buffers.add(new SurfBufferP(32)); // MT,HD,WP,ADV,D
		virtualWaveBuffers.add(new VirtualSurfBufferA(1));// MT,HD,WP,ACC,D
		virtualWaveBuffers.add(new VirtualSurfBufferC(1)); // BP,WP, ACC, ADV, D 
		virtualWaveBuffers.add(new VirtualSurfBufferB(1));// BP,HD,VSCT,VSHT,D
		/*
		buffers.add(new SurfBufferW(4));// BP,HD,VSCT,VSHT,D
		buffers.add(new SurfBufferA(16)); // WP, LV, D
		buffers.add(new SurfBufferE(32));// BP,HD, LV,D
		buffers.add(new SurfBufferT(32));// MT,WP,VSCT,VSHT,D
		buffers.add(new SurfBufferL(64));// BP,LV,D
		buffers.add(new SurfBufferF(32));// ACC, ADV, D
		buffers.add(new SurfBufferO(4));// WP,HD,ADV,D
		buffers.add(new SurfBufferM(16));// WP,HD
		buffers.add(new SurfBufferU(16));// BP,WP,VSCT,VSHT,D
		buffers.add(new SurfBufferZ(1));// MT,ACC, LV,WP
		buffers.add(new SurfBufferN(32));// WP, DP, ACC, LV
		buffers.add(new SurfBufferV(16));// ACC, WP, BP
		buffers.add(new SurfBufferK(16));// ACC,LV,D
		buffers.add(new SurfBufferY(4)); // VSCT, VSHT,D
		






		buffers.add(new SurfBufferC(2)); // WP,D
		buffers.add(new SurfBufferD(2)); // BP,D
		buffers.add(new SurfBufferH(2)); // ADV, D
		buffers.add(new SurfBufferI(2)); // LV,D
		buffers.add(new SurfBufferR(2));// MT,D
		buffers.add(new SurfBufferJ(2));// HD,D
		buffers.add(new SurfBufferX(2));// ACC,D
		 */
	}

	public BasicWaveSurfer(LucasslfBot r) {
		super(r);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * lucasslf.wavesurfing.WaveSurferMovement#onHitRobot(robocode.HitRobotEvent
	 * )
	 */
	public void onHitRobot(HitRobotEvent e2) {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see lucasslf.wavesurfing.WaveSurferMovement#onBulletHitBullet(robocode.
	 * BulletHitBulletEvent)
	 */
	public void onBulletHitBullet(BulletHitBulletEvent e) {
		if (!enemyWaves.isEmpty()) {
			Point2D hitBulletLocation = new Point2D.Double(
					e.getBullet().getX(), e.getBullet().getY());
			EnemyWave hitWave = getHitWave(hitBulletLocation, e.getHitBullet());
			if (hitWave != null) {
				hitWave.logHit(hitBulletLocation);
				getRobot().removeCustomEvent(hitWave);
				enemyWaves.remove(enemyWaves.lastIndexOf(hitWave));

			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * lucasslf.wavesurfing.WaveSurferMovement#onBulletHit(robocode.BulletHitEvent
	 * )
	 */
	public void onBulletHit(BulletHitEvent e) {

	}

	private void updateHitRate() {
		hitRate = bulletHitCount / bulletShotCount;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see lucasslf.wavesurfing.WaveSurferMovement#onHitByBullet(robocode.
	 * HitByBulletEvent)
	 */
	public void onHitByBullet(HitByBulletEvent e) {

		Point2D position = new Point2D.Double(getRobot().getX(), getRobot()
				.getY());
		bulletHitCount++;
		updateHitRate();
		if (!enemyWaves.isEmpty()) {

			Point2D hitBulletLocation = new Point2D.Double(
					e.getBullet().getX(), e.getBullet().getY());
			EnemyWave hitWave = getHitWave(position, e.getBullet());

			// look through the Enemy Waves, and find one that could've hit us.
			if (hitWave != null) {
				hitWave.logHit(hitBulletLocation);
				getRobot().removeCustomEvent(hitWave);
				enemyWaves.remove(enemyWaves.lastIndexOf(hitWave));
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see lucasslf.wavesurfing.WaveSurferMovement#onScannedRobot(robocode.
	 * ScannedRobotEvent)
	 */
	public void onScannedRobot(ScannedRobotEvent e) {
		EnemyWave wave = null;
		double enemyAbsoluteBearing = getRobot().getHeadingRadians()
				+ e.getBearingRadians();
		double headingDifference = e.getHeadingRadians() - enemyAbsoluteBearing;
		double deltaBearing = enemyAbsoluteBearing
				- getRobot().lastScannedRobot.bearing;
		double lateralVelocity = getRobot().getVelocity()
				* FastTrig.sin(e.getBearingRadians());
		double advancingVelocity = e.getVelocity() * -1
				* FastTrig.cos(headingDifference);
		surfDirections.add(new Integer((lateralVelocity >= 0) ? 1 : -1));
		velocities.add(getRobot().getVelocity());
		surfAbsBearings.add(new Double(enemyAbsoluteBearing + FastTrig.PI));
		lateralVelocities.add(lateralVelocity);
		distances.add(e.getDistance());
		headingDifferences.add(Utils.toDegrees(robocode.util.Utils
				.normalAbsoluteAngle(headingDifference)));
		deltaBearings.add(deltaBearing);
		positions.add(new Point2D.Double(getRobot().getX(), getRobot().getY()));
		advancingVelocities.add(advancingVelocity);
		if (getRobot().getVelocity() == 0)
			moveTimes = 0;
		else
			moveTimes++;
		int moveTimesIndex = Math.min(9, moveTimes / 5);

		if (getRobot().lastScannedRobot.energy > e.getEnergy()
				&& (getRobot().lastScannedRobot.energy - e.getEnergy()) <= 3) {
			lastBulletPower = getRobot().lastScannedRobot.energy
					- e.getEnergy();
			bulletShotCount++;
			updateHitRate();

			useVirtualBuffers = (hitRate > 0.1);

			wave = new EnemyWave(getRobot().lastScannedRobot.position,
					surfAbsBearings.get(2), e.getTime() - 1, lastBulletPower,
					surfDirections.get(2), distances.get(2), e.getVelocity(),
					lateralVelocities.get(2), positions.get(2),
					velocities.get(2), velocities.get(1),
					headingDifferences.get(2), headingDifferences.get(1),
					advancingVelocities.get(2), deltaBearings.get(2),
					deltaBearings.get(1), moveTimesIndex, false, getRobot());

		} else {

			if (surfAbsBearings.size() > 2) {
				wave = new EnemyWave(getRobot().lastScannedRobot.position,
						surfAbsBearings.get(2), e.getTime() - 1,
						lastBulletPower, surfDirections.get(2),
						distances.get(2), e.getVelocity(),
						lateralVelocities.get(2), positions.get(2),
						velocities.get(2), velocities.get(1),
						headingDifferences.get(2), headingDifferences.get(1),
						advancingVelocities.get(2), deltaBearings.get(2),
						deltaBearings.get(1), moveTimesIndex, true, getRobot());
			}

		}

		if (wave != null) {
			for (SurfBuffer sb : buffers) {
				sb.setSegmentationOn(wave);
			}
			for (SurfBuffer sb : virtualWaveBuffers) {
				sb.setSegmentationOn(wave);
			}
			enemyWaves.add(wave);
			getRobot().addCustomEvent(wave);
		}
		doSurfing();
	}

	private EnemyWave getHitWave(Point2D localizacao, Bullet b) {
		for (int x = 0; x < enemyWaves.size(); x++) {
			EnemyWave o = (EnemyWave) enemyWaves.get(x);
			if (!o.isVirtual()) {
				if (Math.abs(o.getTraveledDistance()
						- localizacao.distance(o.getInitialPosition())) < 50
						&& Math.abs(Utils.bulletVelocity(b.getPower())
								- o.getBulletSpeed()) < 0.001) {
					return o;
				}
			}
		}
		return null;
	}

	private EnemyWave getNextSurfableWaveAfter(EnemyWave wave) {
		EnemyWave surfWave = null;
		Point2D localizacao = new Point2D.Double(getRobot().getX(), getRobot()
				.getY());
		long longestTimeToHit = Long.MAX_VALUE;
		for (EnemyWave ew : enemyWaves) {
			if (!ew.isVirtual()) {
				double distance = localizacao.distance(ew.getInitialPosition())
						- ew.getTraveledDistance();
				long timeToHit = (long) (distance / ew.getBulletSpeed());
				if (wave == null) {
					if (ew.getTraveledDistance() < localizacao.distance(ew
							.getInitialPosition())
							&& timeToHit < longestTimeToHit) {
						surfWave = ew;
						longestTimeToHit = timeToHit;
					}
				} else {
					if (ew.getTraveledDistance() < localizacao.distance(ew
							.getInitialPosition())
							&& timeToHit < longestTimeToHit
							&& timeToHit > distance / wave.getBulletSpeed()) {
						surfWave = ew;
						longestTimeToHit = timeToHit;
					}
				}
			}
		}

		return surfWave;
	}

	// angle = the angle you'd like to go if there weren't any walls
	// oDir = 1 if you are currently orbiting the enemy clockwise
	// -1 if you are currently orbiting the enemy counter-clockwise
	// returns the angle you should travel to avoid walls
	double wallSmoothing(Point2D botLocation, double angle, int oDir) {
		// if(!_fieldRect.contains(project(botLocation,angle + FastTrig.PI*(oDir
		// +
		// 1),WALKING_STICK))){
		angle = smoothWest(N - botLocation.getY(), angle - FastTrig.HALF_PI,
				oDir) + FastTrig.HALF_PI;
		angle = smoothWest(E - botLocation.getX(), angle + FastTrig.PI, oDir)
				- FastTrig.PI;
		angle = smoothWest(botLocation.getY() - S, angle + FastTrig.HALF_PI,
				oDir) - FastTrig.HALF_PI;
		angle = smoothWest(botLocation.getX() - W, angle, oDir);

		// for bots that could calculate an angle that is pointing pretty far
		// into a corner, these three lines may be necessary when travelling
		// counter-clockwise (since the smoothing above may have moved the
		// walking stick into another wall)
		angle = smoothWest(botLocation.getY() - S, angle + FastTrig.HALF_PI,
				oDir) - FastTrig.HALF_PI;
		angle = smoothWest(E - botLocation.getX(), angle + FastTrig.PI, oDir)
				- FastTrig.PI;
		angle = smoothWest(N - botLocation.getY(), angle - FastTrig.HALF_PI,
				oDir) + FastTrig.HALF_PI;
		// }
		return angle;
	}

	// smooths agains the west wall
	static double smoothWest(double dist, double angle, int oDir) {
		if (dist < -WALL_STICK * FastTrig.sin(angle)) {
			return FastTrig.acos(oDir * dist / WALL_STICK) - oDir
					* FastTrig.HALF_PI;
		}
		return angle;
	}

	/*
	 * // Iterative wall smoothing by PEZ private double wallSmoothing(Point2D
	 * botLocation, double angle, int orientation) { while
	 * (!fieldRect.contains(Utils .project(botLocation, angle, WALL_STICK))) {
	 * angle += orientation * 0.05; } return angle; }
	 */

	private PredictedPosition predictAwayPosition(EnemyWave surfWave,
			double enemyDistance) {
		Point2D predictedPosition = (Point2D) new Point2D.Double(getRobot()
				.getX(), getRobot().getY()).clone();
		double predictedVelocity = getRobot().getVelocity();
		double predictedHeading = getRobot().getHeadingRadians();
		double absBearing = Utils.absoluteBearing(predictedPosition,
				getRobot().lastScannedRobot.position);
		double maxTurning, moveAngle;
		int counter = 0; // number of ticks in the future
		boolean intercepted = false;
		int orientation = -globalAwayMovementDirection;
		do {

			double v2;
			double offset = FastTrig.PI / 2 + (1 - enemyDistance / 400);

			while (!fieldRect.contains(Utils.project(predictedPosition,
					v2 = absBearing + globalAwayMovementDirection
							* (offset -= 0.02), WALL_STICK)))
				;

			if (offset < FastTrig.PI / 3)
				orientation = -orientation;
			moveAngle = FastTrig.tan(v2 - predictedHeading);

			// maxTurning is built in like this, you can't turn more then
			// this in one tick
			maxTurning = Utils.getMaxTurningInOneTick(predictedVelocity);
			predictedHeading = robocode.util.Utils
					.normalRelativeAngle(predictedHeading
							+ Utils.limit(-maxTurning, moveAngle, maxTurning));

			// this one is nice ;). if predictedVelocity and moveDir have
			// different signs you want to break down
			// otherwise you want to accelerate (look at the factor "2")
			predictedVelocity += (predictedVelocity * orientation < 0 ? 2 * orientation
					: orientation);
			predictedVelocity = Utils.limit(-8, predictedVelocity, 8);

			// calculate the new predicted position
			predictedPosition = Utils.project(predictedPosition,
					predictedHeading, predictedVelocity);

			counter++;
			if (predictedPosition.distance(surfWave.getInitialPosition()) < surfWave
					.getTraveledDistance()
					+ ((counter + 1) * surfWave.getBulletSpeed())) {

				intercepted = true;
			}

		} while (!intercepted && counter < 500);

		return new PredictedPosition(predictedPosition, counter, orientation);
	}

	private PredictedPosition predictPosition(Point2D predictedPosition,
			EnemyWave surfWave, int orientation) {
		double predictedVelocity = getRobot().getVelocity();
		double predictedHeading = getRobot().getHeadingRadians();
		double maxTurning, moveAngle, moveDir;

		int counter = 0; // number of ticks in the future
		boolean intercepted = false;

		do {

			if (orientation != 0) {
				moveAngle = wallSmoothing(
						predictedPosition,
						Utils.absoluteBearing(surfWave.getInitialPosition(),
								predictedPosition)
								+ (orientation * (FastTrig.A_LITTLE_LESS_THAN_HALF_PI)),
						orientation)
						- predictedHeading;
				moveDir = 1;

				if (FastTrig.cos(moveAngle) < 0) {
					moveAngle += FastTrig.PI;
					moveDir = -1;
				}

				moveAngle = robocode.util.Utils.normalRelativeAngle(moveAngle);

				// maxTurning is built in like this, you can't turn more then
				// this in one tick
				maxTurning = Utils.getMaxTurningInOneTick(predictedVelocity);
				predictedHeading = robocode.util.Utils
						.normalRelativeAngle(predictedHeading
								+ Utils.limit(-maxTurning, moveAngle,
										maxTurning));

				// this one is nice ;). if predictedVelocity and moveDir have
				// different signs you want to breack down
				// otherwise you want to accelerate (look at the factor "2")
				predictedVelocity += (predictedVelocity * moveDir < 0 ? 2 * moveDir
						: moveDir);
				predictedVelocity = Utils.limit(-8, predictedVelocity, 8);

				// calculate the new predicted position
				predictedPosition = Utils.project(predictedPosition,
						predictedHeading, predictedVelocity);

			}
			counter++;
			if (predictedPosition.distance(surfWave.getInitialPosition()) < surfWave
					.getTraveledDistance()
					+ ((counter + 1) * surfWave.getBulletSpeed())) {

				intercepted = true;
			}
		} while (!intercepted && counter < 500);

		return new PredictedPosition(predictedPosition, counter, orientation);
	}

	private double checkDanger(EnemyWave surfWave,
			PredictedPosition predictedPosition, int recursionCounter) {
		List<Integer> indexes = surfWave.getFactorIndexes(
				predictedPosition.position, predictedPosition.ticks);
		double danger = 0;
		for (int i : indexes) {
			danger += surfWave.getBinDanger(i);
			if (useVirtualBuffers) {
				danger += 0.5 * surfWave.getVirtualBuffersBinDanger(i);
			}
		}
		// danger = danger / indexes.size();

		danger /= getRobot().lastScannedRobot.position
				.distance(predictedPosition.position);

		danger *= (1 + (0.2 / (4 - surfWave.getBulletPower())));

		if (recursionCounter++ < Math.min(MAX_CHECK_DANGER_RECURSION,
				getNonVirtualEnemyWaveCount())) {
			EnemyWave nextSurfable = getNextSurfableWaveAfter(surfWave);
			if (nextSurfable != null) {
				PredictedPosition pl = predictPosition(
						predictedPosition.position, nextSurfable, -1);
				PredictedPosition pr = predictPosition(
						predictedPosition.position, nextSurfable, 1);
				PredictedPosition ps = predictPosition(
						predictedPosition.position, nextSurfable, 0);
				danger += Math.min(Math.min(
						checkDanger(nextSurfable, pl, recursionCounter),
						checkDanger(nextSurfable, pr, recursionCounter)),
						checkDanger(nextSurfable, ps, recursionCounter));
			}
		}
		return danger;
	}

	private int getNonVirtualEnemyWaveCount() {
		int count = 0;
		for (EnemyWave ew : enemyWaves) {
			if (!ew.isVirtual())
				count++;
		}
		return count;
	}

	private void setBackAsFront(double goAngle, double distance) {
		double angle = robocode.util.Utils.normalRelativeAngle(goAngle
				- getRobot().getHeadingRadians());
		if (Math.abs(angle) > (FastTrig.PI / 2)) {
			if (angle < 0) {
				getRobot().setTurnRightRadians(FastTrig.PI + angle);
			} else {
				getRobot().setTurnLeftRadians(FastTrig.PI - angle);
			}
			getRobot().setBack(distance);
		} else {
			if (angle < 0) {
				getRobot().setTurnLeftRadians(-1 * angle);
			} else {
				getRobot().setTurnRightRadians(angle);
			}
			getRobot().setAhead(distance);
		}
	}

	private void doAwayMovement(Point2D position) {
		double[] awayMovementData = getAwayMovementData(position);
		getRobot().setMaxVelocity(Rules.MAX_VELOCITY);
		getRobot().setAhead(awayMovementData[0]);
		getRobot().setTurnRightRadians(awayMovementData[1]);
	}

	private double[] getAwayMovementData(Point2D position) {
		double distance = getRobot().lastScannedRobot.position
				.distance(position);

		// From BasicGTSurfer http://robowiki.net/wiki/BasicGTSurfer/Code
		// do 'away' movement best distance of 400 - modified from RaikoNano

		double absBearing = Utils.absoluteBearing(position,
				getRobot().lastScannedRobot.position);
		double headingRadians = getRobot().getHeadingRadians();
		double v2;
		double offset = FastTrig.PI / 2 + (1 - distance / 400);

		while (!fieldRect.contains(Utils.project(position, v2 = absBearing
				+ globalAwayMovementDirection * (offset -= 0.02), WALL_STICK)))
			;
		if (offset < FastTrig.PI / 3)
			globalAwayMovementDirection = -globalAwayMovementDirection;
		return new double[] { 50 * FastTrig.cos(v2 - headingRadians),
				FastTrig.tan(v2 - headingRadians) };
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see lucasslf.wavesurfing.WaveSurferMovement#doSurfing()
	 */
	@Override
	public void doSurfing() {

		EnemyWave surfWave = getNextSurfableWaveAfter(null);
		Point2D position = new Point2D.Double(getRobot().getX(), getRobot()
				.getY());

		if (surfWave == null) {

			doAwayMovement(position);
		} else {

			double distanceNow = position
					.distance(getRobot().lastScannedRobot.position);

			PredictedPosition predictedPositionRight = predictPosition(
					position, surfWave, 1);
			PredictedPosition predictedPositionLeft = predictPosition(position,
					surfWave, -1);
			PredictedPosition predictedPositionStopped = predictPosition(
					position, surfWave, 0);

			double leftDanger = checkDanger(surfWave, predictedPositionLeft, 0);
			double rightDanger = checkDanger(surfWave, predictedPositionRight,
					0);
			double stopDanger = checkDanger(surfWave, predictedPositionStopped,
					0);
			if (distanceNow < SAFE_DISTANCE) {
				PredictedPosition predictedAwayPosition = predictAwayPosition(
						surfWave, distanceNow);
				double awayDanger = checkDanger(surfWave,
						predictedAwayPosition, MAX_CHECK_DANGER_RECURSION);
				if (awayDanger <= leftDanger && awayDanger <= rightDanger
						&& awayDanger <= stopDanger) {
					doAwayMovement(position);
					return;
				}
			}
			int orientation = globalSurfingMovementDirection;

			double goAngleAux = Utils.absoluteBearing(
					surfWave.getInitialPosition(), position);
			if (stopDanger < leftDanger && stopDanger < rightDanger) {
				getRobot().setMaxVelocity(0);
			} else {
				getRobot().setMaxVelocity(Rules.MAX_VELOCITY);
				if (leftDanger < rightDanger) {
					orientation = -1;
				} else if (leftDanger > rightDanger) {
					orientation = 1;
				}
				double goAngle = wallSmoothing(position, goAngleAux
						+ FastTrig.A_LITTLE_LESS_THAN_HALF_PI * orientation,
						orientation);
				if (Math.abs(goAngle - goAngleAux) > 2 * FastTrig.PI / 3) {
					globalSurfingMovementDirection = -orientation;
				} else {
					globalSurfingMovementDirection = orientation;
				}
				setBackAsFront(goAngle, 100);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see lucasslf.wavesurfing.WaveSurferMovement#onPaint(java.awt.Graphics2D)
	 */
	public void onPaint(Graphics2D g) {
		g.setColor(Color.yellow);
		g.draw(fieldRect);
		for (EnemyWave wave : enemyWaves) {
			wave.draw(g);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * lucasslf.wavesurfing.WaveSurferMovement#onCustomEvent(robocode.CustomEvent
	 * )
	 */
	public void onCustomEvent(CustomEvent e) {
		if (e.getCondition() instanceof EnemyWave) {

			/*
			 * EnemyWave w = (EnemyWave) e.getCondition();
			 * 
			 * System.out.println("BINS: "); for(int i=0;i<BINS;i++){
			 * System.out.println(i+":"+w.getBinDanger(i)+" "); }
			 */
			getRobot().removeCustomEvent(e.getCondition());
			enemyWaves.remove(e.getCondition());
		}

	}

	class PredictedPosition {

		public Point2D position;
		public int ticks;
		public int direction;

		public PredictedPosition(Point2D position, int ticks, int direction) {
			this.position = position;
			this.direction = direction;
			this.ticks = ticks;
		}

	}

}
