package djc;
import robocode.*;
//import java.util.*;

/**
 * FuturePositionArray - a class by Simon Parker
 */
public class FuturePositionArray
{
    private PatternAnalyser analyser;
    static protected int MAX_LOOKAHEAD_TIME = 150;
    final private int MAX_VELOCITY = 8;
    final private int NUM_VELOCITIES = MAX_VELOCITY * 2 + 1;

    private int[] updateCountArray = new int[NUM_VELOCITIES];
    private Coordinate[][] positionArray = new Coordinate[NUM_VELOCITIES][MAX_LOOKAHEAD_TIME];
    private double[][] errorArray = new double[NUM_VELOCITIES][MAX_LOOKAHEAD_TIME];

    public FuturePositionArray(PatternAnalyser patternAnalyser)
    {
	analyser = patternAnalyser;
	for (int i = 0; i < NUM_VELOCITIES; i++) {
	    for (int j = 0; j < MAX_LOOKAHEAD_TIME; j++) {
		positionArray[i][j] = new Coordinate();
		errorArray[i][j] = 0.0;
	    }
	}
    }
    
    public Coordinate getPosition(double velocity, double time)
    {
	int velocityIndex = getVelocityIndex(velocity);
	int floorTime = (int)Math.floor(time);
	int ceilingTime = (int)Math.ceil(time);
	
	if ((floorTime >= MAX_LOOKAHEAD_TIME) || (ceilingTime >= MAX_LOOKAHEAD_TIME)) {
	    floorTime = MAX_LOOKAHEAD_TIME - 2;
	    ceilingTime = floorTime + 1;
	}
	
	if ((floorTime < 0) || (ceilingTime < 0)) {
	    floorTime = 0;
	    ceilingTime = 1;
	}
	
	Coordinate pos0 = positionArray[velocityIndex][floorTime];//???.getMedian();
	Coordinate pos1 = positionArray[velocityIndex][ceilingTime];//???.getMedian();
	
	if (pos0 == null) 
	    return new Coordinate(Coordinate.FIELD_WIDTH / 2, 
				  Coordinate.FIELD_HEIGHT / 2);
	if (pos1 == null) 
	    return pos0;
	
	double dX = pos1.x - pos0.x;
	double dY = pos1.y - pos0.y;
	double x = pos0.x + dX * (time - (double)floorTime);
	double y = pos0.y + dY * (time - (double)floorTime);
	
	return new Coordinate(x, y);
    }

    public double getError(double velocity, double time)
    {
	int velocityIndex = getVelocityIndex(velocity);
	int floorTime = (int)Math.floor(time);
	int ceilingTime = (int)Math.ceil(time);
	
	if ((floorTime >= MAX_LOOKAHEAD_TIME) || (ceilingTime >= MAX_LOOKAHEAD_TIME)) {
	    floorTime = MAX_LOOKAHEAD_TIME - 2;
	    ceilingTime = floorTime + 1;
	    
	}
	
	if ((floorTime < 0) || (ceilingTime < 0)) {
	    floorTime = 0;
	    ceilingTime = 1;
	}
	
	double error = Math.sqrt(errorArray[velocityIndex][floorTime]);
	return error;
    }
    
    
    public void update(Coordinate position, double velocity)
    {
	int velocityIndex = getVelocityIndex(velocity);
	
	updateCountArray[velocityIndex]++;
	
	for (int lookaheadTime = 0; lookaheadTime < MAX_LOOKAHEAD_TIME; lookaheadTime++) {
	    //calculate the normalised delta position for each future time
	    Position p = analyser.getPosition(lookaheadTime);
	    if (p == null) continue;
	    
	    Coordinate latestPosition = position.minus(p).rotate(-p.heading);
	    
	    //cumulative average position
	    //number of samples that the average has been calculated over
	    double N = (double)updateCountArray[velocityIndex];
	    N = Math.min(N, 10);
	    
	    //positionArray[velocityIndex][lookaheadTime].add(latestPosition);
	    
		
	    positionArray[velocityIndex][lookaheadTime].set(((N - 1.0) * 
							     positionArray[velocityIndex][lookaheadTime].x + 
							     latestPosition.x) / N,
							    ((N - 1.0) * 
							     positionArray[velocityIndex][lookaheadTime].y + 
							     latestPosition.y) / N);
		
		
	    Coordinate errorVector = latestPosition.minus(positionArray[velocityIndex][lookaheadTime]);
	    double errorSquared = errorVector.x * errorVector.x + errorVector.y * errorVector.y;
		
	    errorArray[velocityIndex][lookaheadTime] = ((N - 1.0) * errorArray[velocityIndex][lookaheadTime] + 
							errorSquared) / N;
	}
    }

    private int getVelocityIndex(double velocity)
    {
	return (int)Math.round(velocity) + MAX_VELOCITY;
    }
}
