package sch;

import robocode.*;
import java.util.*;
import java.io.*;

/**
 * Tracks all information about scanned robot
 */
public class Bot implements Constants, Comparable {

	// State data

	private String botName;
	private double heading;
	private double speed;
	private double ang_speed;
	private Point pos;
	private double distance;
	private boolean alive;
	private double energy;
	private double energyDrop;
	private double lastTimeUpdated;
	private boolean stateDataValid;

	
	// Some statistic
	private List interceptStatsList;
	private int diedBeforeMe;
	private int numRounds;
	private int numScanned;
	private double totalSpeed;
	private double totalAng_speed;

	// Helpers variable

	private double lastDamage;
	private Map bulletMap;
	private Map interceptKeyToStats;
	private EnemyManager enemyManager;
	private AdvancedRobot myBot;
	private Intercept currentIntercept;
	

	
	/**
	 * Create an empty instance of a Bot and set 
	 */

	public Bot (String botName, EnemyManager enemyManager) {
		this.botName=botName;
		this.heading=0.0;
		this.speed=0.0;
		this.distance=MAX_DISTANCE;
		this.pos=new Point();
		this.alive=true;
		this.energy=100.0;
		this.lastTimeUpdated=0.0;
		this.ang_speed=0.0;
		this.stateDataValid=true;
		this.energyDrop=0.0;
		
		this.lastDamage=0;
		this.interceptStatsList=new ArrayList();
		this.interceptKeyToStats=new HashMap(INTERCEPT_NUM);
		this.bulletMap=new HashMap();
		this.enemyManager=enemyManager;
		myBot=enemyManager.getMyBot();
		initialize();
		readFromFile();
	}

	public void initialize() {
		for (Iterator i=enemyManager.getInterceptMethods().listIterator();i.hasNext();) {
			Intercept currIntercept=(Intercept)i.next();
			InterceptStats tempInterceptStats=new InterceptStats(currIntercept);
			interceptStatsList.add(tempInterceptStats);
			interceptKeyToStats.put(currIntercept,tempInterceptStats);
		}
		Collections.sort(interceptStatsList);
		this.lastDamage=0;
		numRounds++;
	}
	
	public void reinitialize() {
		this.numRounds++;
		this.lastDamage=0;
		this.energy=0;
		update(0.0, 0.0, 0.0, new Point(), 100.0, 0.0);
		for (Iterator i=interceptStatsList.listIterator();i.hasNext();) {
			InterceptStats currInterceptStats = (InterceptStats)i.next();
			currInterceptStats.reset();
		}
		this.alive=true;
		Collections.sort(interceptStatsList);
		bulletMap.clear();
	}
	
	/**
	 * Return the name of the Bot
	 * @return	the name of the Bot
	 */
	public String getName() {return this.botName;}

	/**
	 * Return the direction the Bot is facing last time I saw him
	 * @return	the heading of the Bot
	 */	
	public double getHeading() {return this.heading;}

	/**
	 * Return the speed the Bot was moving last time I saw him
	 * @return	the speed of the Bot
	 */	
	public double getSpeed() {return this.speed;}

	/**
	 * Return the angular speed the Bot was turning since last time I saw him
	 * @return	the angular speed of the Bot
	 */	
	public double getAngSpeed() {return this.ang_speed;}

	/**
	 * Return the average speed the Bot was moving last time I saw him
	 * @return	the average speed of the Bot
	 */	
	public double getAvgSpeed() {return this.totalSpeed/this.numScanned;}

	/**
	 * Return the average angular speed the Bot was turning since last time I saw him
	 * @return	the average angular speed of the Bot
	 */	
	public double getAvgAngSpeed() {return this.totalAng_speed/this.numScanned;}

	/**
	 * Return the distance I had from the bot last time I saw him
	 * @return	the distance from the Bot
	 */
	public double getDistance() {return this.distance;}

	/**
	 * Return the position of the bot last time I saw him
	 * @return	the position of the Bot
	 */	
	public Point getPos() {return this.pos;}

	/**
	 * Return the energy of the bot last time I saw him
	 * @return	the energy of the Bot
	 */			
	public double getEnergy() {return this.energy;}

	/**
	 * Return wheater the Bot is alive or not
	 * @return	true is Bot is alive
	 */		
	public boolean isAlive() {return alive;}

	/**
	 * Return the time of the position informations
	 * @return	time of the last update
	 */			
	public double getLastUpdateTime() {return lastTimeUpdated;}
	
	/**
	 * Return the time of the position informations
	 * @return	time of the last update
	 */			
	public boolean isUpdated() {return (myBot.getTime()-lastTimeUpdated)<10;}

	/**
	 * Return the number of times the Bot died before me:<br>used to decide if this robot is a theat or not
	 * @return	num of time died before me
	 */			
	public double getNumDiedBeforeMe() {
		if (diedBeforeMe==0.0)
			return 0.000001;
		else
			return (double) diedBeforeMe;}

	/**
	 * Return the number of rounds we faced in a battle
	 * @return	num rounds
	 */		
	public int getNumRounds() {return numRounds;}

	/**
	 * Tells if the bot has shoot since the last scanning
	 * @return		True if the bot has shoot
	 */
	public boolean hasShoot() {
		double realDrop = this.energyDrop-this.lastDamage;
		this.lastDamage=0.0;
		if (realDrop<=3.0 && realDrop>=0.09)
			return true;
		else
			return false;
	}
		
	/**
	 * Return the number of shot I fired upon the Bot
	 * @return	num of shot fired
	 */			
	public double getNumShotAt() {return getAllShots();}

	/**
	 * Return the number of times I hit the Bot
	 * @return	num of successul shots
	 */		
	public double getNumHit() {return getAllHits();}

	/**
	 * Return the hit ratio upon this bot.<br>
	 * Usefull to decide wheater if my targeting analysis is good or not
	 * @return	hit ratio
	 */		
	public double getHitRatio() {return (getAllHits()/getAllShots())*100;}

	/**
	 * Return the number of shot I fired upon the Bot
	 * @return	num of shot fired
	 */			
	public double getRoundNumShotAt() {return getAllShotsRound();}

	/**
	 * Return the number of times I hit the Bot
	 * @return	num of successul shots
	 */		
	public double getRoundNumHit() {return getAllHitsRound();}

	/**
	 * Return the hit ratio upon this bot.<br>
	 * Usefull to decide wheater if my targeting analysis is good or not
	 * @return	hit ratio
	 */		
	public double getRoundHitRatio() {
		double shots=getAllShotsRound();
		if (shots==0.0)
			shots=0.000001;
		return (getAllHitsRound()/shots)*100;}
	
	/**
	 * Updates the postion information for the Bot
	 * @param	heading		direction the robot is facing
	 * @param	speed		speed of the robot
	 * @param	distance	distance of the enemy robot from my robot
	 * @param	pos			position of the robot
	 * @param	energy		current energy of the robot
	 * @param	time		time of all position informations
	 */
	public void update(double heading, double speed, double distance, Point pos, double energy, double time) {
		
		double dT = time - this.lastTimeUpdated;
		double dH = BotMath.normaliseBearing(heading-this.heading);
		
		if (dT!=0 && dT < ANG_SPEED_SENSIBLE_TIME) {
			this.ang_speed = dH / dT;
		}
		else {
			this.ang_speed = 0.0;
		}
		
		this.heading=heading;
		this.speed=speed;
		this.distance=distance;
		this.pos=pos;
		this.energyDrop=this.energy-energy;
		this.energy=energy;
		this.lastTimeUpdated=time;
		numScanned++;
		totalSpeed+=speed;
		totalAng_speed+=ang_speed;
	}
	
	/**
	 * Called when the Bot dies
	 */
	public void onDeath() {
		this.alive=false;
		this.diedBeforeMe++;
		if (BotUtil.varContains(DEBUG,DEBUG_STATS)) {
			System.out.println(" __________BOT NAME__________|_DIED_BEFORE_ME_|_____HIT_RATIO______|");
			System.out.println("|"+BotUtil.makeFixedLength(this.getName(),28,ALIGN_LEFT)+"|"+BotUtil.makeFixedLength(this.getNumDiedBeforeMe()+"",16,ALIGN_CENTER)+"|"+BotUtil.makeFixedLength(this.getRoundNumHit()+"/"+this.getRoundNumShotAt()+" "+BotMath.round2(this.getRoundHitRatio())+"%",20,ALIGN_CENTER)+"|");			
		}
	}
	
	/**
	 * Calculate the estimated intercept position<br>
	 * The calculation is based on past behaveur of the current enemy bot
	 * @param	myBotPos	the current position
	 * @param	shotPower	power the bullet is goin to be shot
	 * @param	now			current time
	 * @return	guessed position
	 */
	public Point guessInterceptPosition (Point myBotPos, double shotPower, double now) {
		chooseAimingMethod();
		Point estimatedPosition=this.currentIntercept.calculate(myBotPos,getPos(),now,getLastUpdateTime(),getHeading(),getSpeed(),BotMath.getBulletVelocity(shotPower),getAngSpeed());
		return new Point(estimatedPosition);
	}
	
	public void addShot(Bullet bulletShot) {
		((InterceptStats)interceptKeyToStats.get(currentIntercept)).incrementShots();
		bulletMap.put(bulletShot,currentIntercept);
		Collections.sort(interceptStatsList);		
	}

	public void addHit(Bullet bulletHit) {
		double bulletPower=bulletHit.getPower();
		this.lastDamage=BotMath.getBulletDamage(bulletPower);
		if (bulletMap.containsKey(bulletHit)) {
			((InterceptStats)interceptKeyToStats.get(currentIntercept)).incrementHits();
			Collections.sort(interceptStatsList);
		}
		
	}
		
	public void calculateStats() {

		String victimStr=null;
		for (Iterator i=bulletMap.entrySet().iterator();i.hasNext();) {
			Map.Entry e=(Map.Entry) i.next();
			Bullet currentBullet=(Bullet) e.getKey();
			Intercept bulletInterceptMethod=(Intercept) e.getValue();
			if(currentBullet!=null) {
				((InterceptStats)interceptKeyToStats.get(bulletInterceptMethod)).incrementTotShots();
				victimStr=currentBullet.getVictim();
				if (victimStr!=null) {
					if (victimStr.equals(getName())) {
						((InterceptStats)interceptKeyToStats.get(bulletInterceptMethod)).incrementTotHits();
					}
				}
			}
		}
	}
	
	private void resetPersistentData() {
		diedBeforeMe=0;
		numRounds=1;
		numScanned=0;
		totalSpeed=0;
		totalAng_speed=0;
	}
	
    /**
     * Read previous data on this robot from file.
     */
    public void readFromFile() {

        try {
            DataInputStream in =  new DataInputStream(new FileInputStream(myBot.getDataFile(determineFileName())));
			readObject(in);
        } catch (IOException e) {resetPersistentData();}
    }

    /**
     * Store data on this robot to file.
     */
    public void writeToFile() {

        try {
            DataOutputStream out = new DataOutputStream(new RobocodeFileOutputStream(myBot.getDataFile(determineFileName())));
            writeObject(out);
            out.close();
        } catch (IOException e) {e.printStackTrace();}
    }

    /**
     * Determine the file name for this bot.  The file name will be the
     * same as the bot name, minus any copy number (ie: (1), (2) used
     * when there are multiple instances of the same bot) plus a ".dat"
     * extension.
     */
    private String determineFileName() {

        String fileName = getName();
        int i = fileName.lastIndexOf(" (");
        if (i > -1) {
            int j = fileName.indexOf(')', i + 1);
            if ((j > i) && (j == fileName.length() - 1)) {
                fileName = fileName.substring(0, i);
            }
        }
        return fileName + ".dat";
    }

    /**
     * Write this bot object to file.
     *
     * NOTE: ANY CHANGES TO THIS METHOD MUST ALSO BE REFLECTED IN
     * THE readObject() METHOD BELOW.  THE FILE_VERSION CONSTANT
     * MUST ALSO BE UPDATED TO AVOID VERSIONING PROBLEMS AND THE
     * ROBOT_FILE_SIZE CONSTANT MUST ALSO BE UPDATED.
     */
    private void writeObject(DataOutputStream stream) throws IOException {
		if (BotUtil.varContains(DEBUG,DEBUG_IO)) System.out.println(getName());
        stream.writeInt(BOT_STATS_FILE_VERSION);
        
        stream.writeInt(diedBeforeMe);
		stream.writeInt(numRounds);
		stream.writeInt(numScanned);
		stream.writeDouble(totalSpeed);
		stream.writeDouble(totalAng_speed);
		if (BotUtil.varContains(DEBUG,DEBUG_STATS)) System.out.println(getName());
		for (Iterator i=enemyManager.getInterceptMethods().listIterator();i.hasNext();) {
			Intercept tempIntercept = (Intercept)i.next();
			InterceptStats currInterceptStats=(InterceptStats)interceptKeyToStats.get(tempIntercept);
			if (BotUtil.varContains(DEBUG,DEBUG_STATS)) System.out.println(currInterceptStats.getIntercept().getKey() + " - " + currInterceptStats.getTotHits()+"/"+currInterceptStats.getTotShots() + " " + BotMath.round2(currInterceptStats.getTotHitRatio()) + "%");
			stream.writeDouble(currInterceptStats.getTotHits());
			stream.writeDouble(currInterceptStats.getTotShots());
		}
        
		if (BotUtil.varContains(DEBUG,DEBUG_IO)) System.out.println("Saved to file: " + determineFileName());
    }

    /**
     * Restore this bot's state from a previously written file.
     *
     * NOTE: ANY CHANGES TO THIS METHOD MUST ALSO BE REFLECTED IN
     * THE writeObject() METHOD ABOVE.  THE FILE_VERSION CONSTANT
     * MUST ALSO BE UPDATED TO AVOID VERSIONING PROBLEMS AND THE
     * ROBOT_FILE_SIZE CONSTANT MUST ALSO BE UPDATED.
     */
    private void readObject(DataInputStream stream) throws IOException {
    	int fileVersion=stream.readInt();
    	if (fileVersion==BOT_STATS_FILE_VERSION) {
			if (BotUtil.varContains(DEBUG,DEBUG_IO)) System.out.println("File version: "+fileVersion);
			
			diedBeforeMe=stream.readInt();
			numRounds=stream.readInt();
			numScanned=stream.readInt();
			totalSpeed=stream.readDouble();
			totalAng_speed=stream.readDouble();
			for (Iterator i=enemyManager.getInterceptMethods().listIterator();i.hasNext();) {
				Intercept tempIntercept = (Intercept)i.next();
				InterceptStats currInterceptStats=(InterceptStats)interceptKeyToStats.get(tempIntercept);
				currInterceptStats.setTotHits(stream.readDouble());
				currInterceptStats.setTotShots(stream.readDouble());
			}
			Collections.sort(interceptStatsList);
    	}
    	else {
    		if (BotUtil.varContains(DEBUG,DEBUG_IO)) System.out.println("Stats File obsolete, resetting stats");
    		resetPersistentData();
    	}	
    }

    private void chooseAimingMethod(){
    	if (getAllShots()+getAllShotsRound()<MIN_SAMPLING_SHOTS){
	    	int randInterceptMethod=(int)Math.floor(Math.random()*INTERCEPT_NUM);
	    	int j=0;
	    	for (Iterator i=enemyManager.getInterceptMethods().listIterator();i.hasNext() && j<=randInterceptMethod;j++) {
	    		currentIntercept=(Intercept)i.next();
	    	}
	    }
	    else
	    	currentIntercept=(Intercept) ((InterceptStats) interceptStatsList.get(0)).getIntercept();
    	if (BotUtil.varContains(DEBUG,DEBUG_INTECEPT_SELECTION)) System.out.println("Using " + currentIntercept.getName());
    }

    private double getAllHits() {
    	double totals=0;
		for (Iterator i=interceptStatsList.listIterator();i.hasNext();) {
			InterceptStats currInterceptStats = (InterceptStats)i.next();
			totals+=currInterceptStats.getTotHits();
		}
		return totals;
    }

    private double getAllShots() {
    	double totals=0;
		for (Iterator i=interceptStatsList.listIterator();i.hasNext();) {
			InterceptStats currInterceptStats = (InterceptStats)i.next();
			totals+=currInterceptStats.getTotShots();
		}
		return totals;    	
    }

    private double getAllHitsRound() {
    	double totals=0;
		for (Iterator i=interceptStatsList.listIterator();i.hasNext();) {
			InterceptStats currInterceptStats = (InterceptStats)i.next();
			totals+=currInterceptStats.getHits();
		}
		return totals;
    }

    private double getAllShotsRound() {
    	double totals=0;
		for (Iterator i=interceptStatsList.listIterator();i.hasNext();) {
			InterceptStats currInterceptStats = (InterceptStats)i.next();
			totals+=currInterceptStats.getShots();
		}
		return totals;    	
    }

    public int compareTo(Object o) {
    	Bot enemy = (Bot) o;
    	if (isAlive() && !enemy.isAlive())
    		return -1;
    	else if (!isAlive() && enemy.isAlive())
    		return 1;
    	else
	    	if(getTargetingGoodness()<enemy.getTargetingGoodness())
	    		return -1;
	    	else if (getTargetingGoodness()>enemy.getTargetingGoodness())
	    		return 1;
	    	else
	    		return 0;
    }
    
	public double getBearingFromGun() {
		double targetBearing=BotMath.absBearing(new Point(myBot.getX(), myBot.getY()), pos);
		double gunTurnAmount=BotMath.normaliseBearing(targetBearing-myBot.getGunHeadingRadians());
		return Math.abs(gunTurnAmount);
	}
	
	public double getTargetingWeightedGoodness() {
    	return 2.0*getDistance()+20*getBearingFromGun()+10000/(0.8*getHitRatio()+0.2*getRoundHitRatio())+0.6*getEnergy()+100/(getNumDiedBeforeMe()/getNumRounds());
    }
    
    public double getTargetingGoodness() {
    	//return getDistance();
    	return getTargetingWeightedGoodness();
    }
    
    /**
     * Calculate the power to be used shooting this robot.<br>
     * Takes into account the distance between my robot and the target, mybot energy (to avoid getting disabled) and the target energy (to avoid spending more power than needed to destroy the enemy)
     * @param	distance	the distance between the 2 robots
     * @return				the needed power
     */
	public double calcFirePower(double distance) {
		double tempPower=FIRING_DISTANCE_CONST/distance;
		double maxPower=myBot.getEnergy();
		double estimadedDamage;
		if (maxPower>0.1) {
			if (tempPower>3)
				tempPower=3;
			if ((maxPower-tempPower)<0.1)
				tempPower=maxPower-0.1;
			estimadedDamage=BotMath.getBulletDamage(tempPower);
			if ((this.energy-estimadedDamage)<0.0)
				tempPower=BotMath.getNeededFirePower(this.energy);
			}
		else
			tempPower=0.0;
		return tempPower;
	}
}

