package ags.rougedc.waves;

import ags.rougedc.base.*;

import ags.rougedc.robots.*;
import ags.utils.newtree.KdTree;
import ags.utils.points.*;
import java.util.List;
import java.util.ArrayList;

public class EnemyWaveManager {
    private final Rules rules;
    private final StatusRobot status;
    private final EnemyList enemies;
    private final List<EnemyWave> waves;
    private final List<EnemyWave> heatwaves;
    private static long gunheat_wavesfired = 0;
    private static long gunheat_wavesimmediate = 0;
    private static final KdTree<Double> gunheat_firepowerstats = new KdTree<Double>(3);
    
    public EnemyWaveManager(Rules rules, StatusRobot status, EnemyList enemies) {
        this.rules = rules;
        this.status = status;
        this.enemies = enemies;
        waves = new ArrayList<EnemyWave>();
        heatwaves = new ArrayList<EnemyWave>();
    }
    
    public void run() {
        // Loop through our waves!
        for (java.util.Iterator<EnemyWave> iter = waves.iterator(); iter.hasNext(); ) {
            EnemyWave wave = iter.next();
            wave.move();
            wave.checkHit(status);
            if (wave.expired(status))
                iter.remove();
        }
        
        // Loop through our waves!
        for (java.util.Iterator<EnemyWave> iter = heatwaves.iterator(); iter.hasNext(); ) {
            EnemyWave wave = iter.next();
            wave.move();
            wave.checkHit(status);
            if (wave.getRadius()/wave.getSpeed() >= 1)
                iter.remove();
        }
        
        // Loop through the enemies!
        for (EnemyRobot e : enemies.getAll()) {
            if (!e.isAlive() || !e.isScanned())
                continue;
            double firepower = e.getFirePower();
            // If they've fired, make a wave!
            if (firepower > 0) {
                fireWave(e, firepower);
            }
            considerGunheat(e);
        }
    }
    
    private void considerGunheat(EnemyRobot e) {
        // If the enemy is abou to have 0 gunheat, and it fires right away most of the time
        if (immediateChance() > 0.5 && (e.getGunheat() > 0 && e.getGunheat() < rules.GUN_COOLING_RATE)) {
            // They're about to fire!
            
            double myenergy = status.getEnergy();
            double enemyenergy = e.getEnergy();
            double distance = e.getRelativeLocation().magnitude;
            
            // Determine how powerfully they fired in the most similar prior scenario to this
            List<KdTree.Entry<Double>> entries = KdTree.nearestNeighbor(gunheat_firepowerstats, new double[] {myenergy, enemyenergy, distance}, 1);
            if (entries.size() == 0)
                return;
            double firepower = entries.get(0).value;
            AbsolutePoint origin = e.getLocation().addRelativePoint(e.getVelocity());
            
            EnemyWave wave = new EnemyWave(e, origin, status.clone(), firepower, -1);
            
            heatwaves.add(wave);
        }
    }
    
    private void fireWave(EnemyRobot e, double firepower) {
        if (!Double.isNaN(e.getCoolticks()))
            gunheat_wavesfired++;
        if (e.getCoolticks() == 1) {
            // Store some stats on their firing
            gunheat_wavesimmediate++;
            
            double myenergy = status.ticksAgo(2).getEnergy();
            double enemyenergy = e.getEnergy();
            double distance = e.getRelativeLocation().magnitude;
            
            //List<FirepowerEntry> entries = gunheat_firepowerstats.NearestNeighbors(new HyperPoint(myenergy, enemyenergy, distance), 1);
            //if (entries.size() != 0) {
            //    System.out.println("Predict : " +entries.get(0).getFirepower()+ "  Real: "+ firepower);
            //}
            
            KdTree.addPoint(gunheat_firepowerstats, new double[] {myenergy, enemyenergy, distance}, firepower);
        }
        
        e.firedBullet(firepower, 1);        
        EnemyWave wave = new EnemyWave(e, e.getLastLocation(), status.ticksAgo(2), firepower, 1);
        
        wave.checkHit(status);
        waves.add(wave);
    }
    
    // Returns the chance that the enemy will fire immediately when gunheat is 0
    private double immediateChance() {
        return ((double)gunheat_wavesimmediate / (double)gunheat_wavesfired);
    }
    
    public List<EnemyWave> getAll() {
        return waves;
    }
    
    public List<EnemyWave> getAllHeat() {
        return heatwaves;
    }
    
    // Called when we for some reason know the wave no longer exists
    public void remove(EnemyWave w) {
        waves.remove(w);
    }
}
