/*


 Copyright (C) 2005 Stefan Westen

 e-mail: stefwhome-rc@yahoo.co.uk

 Based on lots of ideas from the Robocode Wiki (http://robowiki.net).

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

 */

package stefw;

import robocode.*;
import robocode.util.*;

import java.awt.geom.*;
import java.util.*;
import java.awt.*;

public class Tigger extends AdvancedRobot {

	int timeSinceVelocityChange = 0;
    double lastEnemyVelocity = 0.0;
    double lastEnemyLateralVelocity = 0.0;
    double lastLateralVelocity = 0.0;
    double lastEnemyEnergy;
    boolean enemyLeft = true;
    Point2D lastEnemyLocation = null;
    ArrayList waves = new ArrayList();
    ArrayList enemyWaves = new ArrayList();
    Rectangle field,bigField;
    private static double[] tileWeights = new double[1000000];
    double fieldWidth;
    double fieldHeight;
    double width;
    double height;
    ScannedRobotEvent lastScannedRobotEvent = null;

    public void run() {

        fieldWidth = getBattleFieldWidth();
        fieldHeight = getBattleFieldHeight();
        width = getWidth();
        height = getHeight();
        field = new Rectangle((int)(0.5*width),(int)(0.5*height),
                (int)(fieldWidth-width),
                (int)(fieldHeight-height));
        bigField = new Rectangle(0,0,(int)fieldWidth,(int)fieldHeight);
        lastEnemyEnergy = 100.0;
        setAdjustRadarForGunTurn(true);
        setAdjustGunForRobotTurn(true);
        setTurnRadarRightRadians(Double.POSITIVE_INFINITY);
        do {
            execute();
            if (getOthers()==0)
            	onScannedRobot(lastScannedRobotEvent);
        } while (true);
    }

    public void onPaint(java.awt.Graphics2D g) {
        long time = getTime();
        Iterator iter = enemyWaves.iterator();
        while (iter.hasNext()) {
            Wave wave = (Wave) iter.next();
            wave.onPaint(g, time, Color.red);
        }
        iter = waves.iterator();
        while (iter.hasNext()) {
            Wave wave = (Wave) iter.next();
            wave.onPaint(g, time, Color.green);
        }
    }

    public void onBulletHit(BulletHitEvent e) {
        double power = e.getBullet().getPower();
        lastEnemyEnergy -= 4 * power + Math.max(2 * power - 1, 0);
    }

    public void onHitByBullet(HitByBulletEvent e) {
        lastEnemyEnergy += 3 * e.getBullet().getPower();

        Wave closest = this.closestEnemyWave();

        if (closest==null)
        	return;

        double originalBearing = Math.atan2(closest.enemyLocation.getX()-closest.startLocation.getX(),
        		closest.enemyLocation.getY()-closest.startLocation.getY());

        double newBearing = Math.atan2(e.getBullet().getX()
                - closest.startLocation.getX(),e.getBullet().getY()
                - closest.startLocation.getY());

        double offset = Utils.normalRelativeAngle(newBearing-originalBearing)*180/Math.PI;

        double width = Math.atan2(26,closest.distance)*180/Math.PI;
    	for (int i=-50;i<=50;i+=1) {
    		double v = (i-offset)/width;
    		if (v*v<1.0) {
    			learnDanger(closest,i,1.0-0.5*v*v);
    		}
    		else {
    			learnDanger(closest,i,0.0);
    		}
    	}
    }

    public void onHitRobot(HitRobotEvent e) {
        lastEnemyEnergy -= 0.6;
    }

    double danger(double enemyX,double enemyY,double x,double y,double velocity,
                                double heading,int direction,int timeSteps)
    {
        double time = getTime();

        double result = 0.0;

        for (int t=0;t<timeSteps;t++) {
            time = time + 1;
            double absoluteBearing = Math.atan2(enemyX-x,
                    enemyY-y);

            double bearing = absoluteBearing-heading;
            double newAngle = bearing;
            if (enemyLeft)
                newAngle += Math.PI/2;
            else
                newAngle -= Math.PI/2;

            boolean clockWise = enemyLeft ^ (velocity>0);

            double wantedHeading = adjustHeadingForWalls(x,y,velocity,newAngle+heading,clockWise);

// double wantedHeading = newAngle+heading;

            double maxHeadingChange = (10-0.75*Math.abs(velocity))*Math.PI/180.0;

            double headingDelta = Utils.normalRelativeAngle(wantedHeading-heading);

            if (headingDelta>maxHeadingChange)
                headingDelta = maxHeadingChange;

            if (headingDelta<-maxHeadingChange)
                headingDelta = -maxHeadingChange;

            heading = heading + headingDelta;

            if (direction==1) {
                if (velocity<0)
                    velocity = velocity + 2;
                else
                    velocity = velocity + 1;
                if (velocity>8)
                    velocity = 8;
            }

            if (direction==0) {
                if (velocity>0) {
                    velocity -= 2;
                    if (velocity<0)
                        velocity = 0;
                }
                if (velocity<0) {
                    velocity += 2;
                    if (velocity>0)
                        velocity = 0;
                }
            }

            if (direction==-1) {
                if (velocity>0)
                    velocity = velocity - 2;
                else
                    velocity = velocity - 1;
                if (velocity<-8)
                    velocity = -8;
            }

            x = x + Math.sin(heading)*velocity;
            y = y + Math.cos(heading)*velocity;

            if (x<0.5*width) {
                x = 0.5*width; velocity = 0;
            }
            if (y<0.5*height) {
                y = 0.5*height; velocity = 0;
            }
            if (x>fieldWidth-0.5*width) {
                x = fieldWidth-0.5*width;
                velocity = 0;
            }
            if (y>fieldHeight-0.5*height) {
                y = fieldHeight-0.5*height;
                velocity = 0;
            }
            Iterator iter = enemyWaves.iterator();
            while (iter.hasNext()) {
                Wave wave = (Wave) iter.next();
                if (wave.bulletVelocity * (time - wave.startTime) >= wave.startLocation.distance(x,y) - 18 &&
                		wave.bulletVelocity * (time - wave.startTime) <= wave.startLocation.distance(x,y) + 18) {
                	double originalBearing = Math.atan2(wave.enemyLocation.getX()-wave.startLocation.getX(),
                    		wave.enemyLocation.getY()-wave.startLocation.getY());

                    double newBearing = Math.atan2(x
                            - wave.startLocation.getX(),y
                            - wave.startLocation.getY());

                    double offset = Utils.normalRelativeAngle(newBearing-originalBearing)*180/Math.PI;

               		result += danger(wave,offset);
                }
            }
        }

        return result;
    }

    double adjustHeadingForWalls(double x,double y,double velocity,double heading,boolean
            clockWise) {
        double stick;

        if (velocity==0)
            return heading;

        if (velocity>0)
            stick= 120.0;
        else
            stick=-120.0;

        while (!field.contains(x+stick*Math.sin(heading),y+stick*Math.cos(heading))) {
            if (clockWise)
                heading += 0.01;
            else
                heading -= 0.01;
        }

        return heading;
    }
 
    public double wallFactor (double x,double y, double distance, double bearing,
    		double maxAngle,int direction) {
		for (int i=0;i<100;i++) {
			double factor = i*0.01;
			if (!bigField.contains(x+distance*Math.sin(bearing+factor*direction*maxAngle),
					y+distance*Math.cos(bearing+factor*direction*maxAngle)))
			   return factor;
		}
		return 1.0;
	}

    public void onScannedRobot(ScannedRobotEvent e) {

        double absoluteBearing = getHeadingRadians() + e.getBearingRadians();

        double enemyLateralVelocity = e.getVelocity()*Math.sin(e.getHeadingRadians()-absoluteBearing);
        double enemyApproachVelocity = -e.getVelocity()*Math.cos(e.getHeadingRadians()-absoluteBearing);
        double approach = -getDistanceRemaining()*Math.cos(getHeadingRadians()-absoluteBearing+Math.PI);

        double lateralVelocity = getVelocity()*Math.sin(getHeadingRadians()-absoluteBearing+Math.PI);

        setTurnRadarRightRadians(Utils.normalRelativeAngle(absoluteBearing
                - getRadarHeadingRadians()) * 2);

        enemyLeft = Math.sin(e.getBearingRadians())<0.0;

        double newAngle = e.getBearingRadians();

        if (enemyLeft)
            newAngle += Math.PI/2;
        else
            newAngle -= Math.PI/2;

        double heading = getHeadingRadians();

        boolean clockWise = enemyLeft ^ (getVelocity()>0);
        
        double evasion = 500/e.getDistance();
        if (getTime()<16&&evasion>2.0) 
        	evasion=2.0;
        if (getTime()>=16&&evasion>1.15)
        	evasion=1.15;
        if (evasion>1.0) {
        	if (clockWise)
        		newAngle -= 0.5*(evasion-1)*Math.PI;
        	else
        		newAngle += 0.5*(evasion-1)*Math.PI;
        }

		double adjustedAngle = adjustHeadingForWalls(getX(), getY(), getVelocity(),
				newAngle + heading, clockWise)
				- heading;

		adjustedAngle = Utils.normalRelativeAngle(adjustedAngle);

		setTurnRightRadians(adjustedAngle);

        double enemyDistance = e.getDistance();

        double enemyX = getX() + enemyDistance * Math.sin(absoluteBearing);
        double enemyY = getY() + enemyDistance * Math.cos(absoluteBearing);

        Point2D enemyLocation = new Point2D.Double(enemyX, enemyY);
        Point2D location = new Point2D.Double(getX(), getY());

        double enemyVelocity = e.getVelocity();

        int forward = getVelocity()>=0?1:-1;
        double forwardDanger = danger(enemyX,enemyY,getX(),getY(),getVelocity(),getHeadingRadians(),forward,20);
        double stopDanger = danger(enemyX,enemyY,getX(),getY(),getVelocity(),getHeadingRadians(),0,20);
        double reverseDanger = danger(enemyX,enemyY,getX(),getY(),getVelocity(),getHeadingRadians(),-forward,20);

        if (reverseDanger<forwardDanger||approach>75)
        	forward=-forward;
        if (stopDanger<forwardDanger&&stopDanger<reverseDanger)
        	forward=0;

        setAhead(100*forward);

        double enemyEnergy = e.getEnergy();
        double energyDifference = enemyEnergy - lastEnemyEnergy;

        if (energyDifference >= -3 && energyDifference <= -0.1
                && lastEnemyLocation != null
                && !(Math.abs(lastEnemyVelocity) >= 2.0 && enemyVelocity == 0)) {
            double firePower = -energyDifference;
            enemyWaves.add(new Wave(lastEnemyLocation, location,
                    20 - 3 * firePower, getTime() - 1,lateralVelocity,lastLateralVelocity,e.getDistance()));
        }

        updateWaves(waves, enemyLocation);
        updateEnemyWaves(enemyWaves, location);

        double bestOffset = 0;
        double bestValue = 0.0;
        int direction = enemyLateralVelocity>=0?1:-1;

        double firePower = 2.0;

        if (enemyDistance<100)
        	firePower=3.0;
        if (e.getEnergy()<10.0)
        	firePower=(e.getEnergy()+2.0)/6.0;
        if (e.getEnergy()<=4.0)
        	firePower=e.getEnergy()/4.0;
        if (firePower<0.1)
        	firePower=0.1;

        timeSinceVelocityChange++;

        if (Math.abs(Math.abs(enemyVelocity)-Math.abs(lastEnemyVelocity))>0.5) {
        	timeSinceVelocityChange=0;
        }

        double bulletTime = e.getDistance()/(20-3*firePower);

        double normalizedTimeSinceVelocityChange = timeSinceVelocityChange/bulletTime;

        if (normalizedTimeSinceVelocityChange>1.2)
        	normalizedTimeSinceVelocityChange=1.2;

        double maxAngleRadians = Math.asin(8.0/(20-3*firePower));

        double wallAngle = wallFactor(getX(),getY(),enemyDistance,absoluteBearing,maxAngleRadians,direction);
        double reverseWallAngle = wallFactor(getX(),getY(),enemyDistance,absoluteBearing,maxAngleRadians,-direction);

        Wave wave = new Wave(location, enemyLocation, 20 - 3 * firePower,
				getTime(), enemyLateralVelocity, enemyApproachVelocity, e.getDistance(),
				wallAngle,reverseWallAngle,lastEnemyLateralVelocity, firePower,
				enemyVelocity,lastEnemyVelocity,normalizedTimeSinceVelocityChange);

        if (firePower < getEnergy() + 0.1 && getOthers() > 0) {
			// only aim gun just before firing
			if (this.getGunHeat() / this.getGunCoolingRate() <= 4.0) {
				for (int i = -50; i <= 50; i += 1) {
					double p = predict(wave, i);
					if (p > bestValue) {
						bestValue = p;
						bestOffset = i;
					}
				}
			}

			setTurnGunRightRadians(Utils.normalRelativeAngle(absoluteBearing
					- getGunHeadingRadians() + bestOffset * Math.PI / 180.0));

			setFireBullet(firePower);
		}

		waves.add(wave);

        lastEnemyLocation = enemyLocation;
        lastEnemyEnergy = enemyEnergy;
        lastEnemyVelocity = enemyVelocity;
        lastEnemyLateralVelocity = enemyLateralVelocity;
        lastLateralVelocity = lateralVelocity;

        lastScannedRobotEvent = e;
    }

    Wave closestEnemyWave() {
        long time = getTime();
        Wave closest = null;
        double best = 1e6;
        Iterator iter = enemyWaves.iterator();
        while (iter.hasNext()) {
            Wave wave = (Wave) iter.next();

            double distance = wave.startLocation.distance(getX(),getY())-
                wave.bulletVelocity*(time-wave.startTime);

            if (distance>0)
            if (closest==null||distance<best) {
                closest = wave;
                best = distance;
            }
        }
        return closest;
    }

    void updateWaves(ArrayList waves, Point2D enemyLocation) {
        long time = getTime();
        Iterator iter = waves.iterator();
        while (iter.hasNext()) {
            Wave wave = (Wave) iter.next();
            if (wave.bulletVelocity * (time - wave.startTime) >= wave.startLocation
                    .distance(enemyLocation) - 18) {
            	if (getOthers() != 0) {
					double originalBearing = Math.atan2(wave.enemyLocation
							.getX()
							- wave.startLocation.getX(), wave.enemyLocation
							.getY()
							- wave.startLocation.getY());
					double newBearing = Math.atan2(enemyLocation.getX()
							- wave.startLocation.getX(), enemyLocation.getY()
							- wave.startLocation.getY());
					double offset = Utils.normalRelativeAngle(newBearing
							- originalBearing);
					double offsetDegrees = offset * 180 / Math.PI;
					double width = Math.atan2(18, wave.distance) * 180
							/ Math.PI;
					for (int i = -50; i <= 50; i += 1) {
						double v = (i-offsetDegrees)/width;
						if (v*v<1.0) {
							learn(wave, i,1-v*v);
						} else {
							learn(wave, i, 0.0);
						}
					}
				}
                iter.remove();
            }
        }
    }

    void updateEnemyWaves(ArrayList waves, Point2D enemyLocation) {
        long time = getTime();
        Iterator iter = waves.iterator();
        while (iter.hasNext()) {
            Wave wave = (Wave) iter.next();
            if (wave.bulletVelocity * (time - wave.startTime) >= wave.startLocation
                    .distance(enemyLocation) + 18)
                iter.remove();
        }
    }

    int[] getEnemyTiles(Wave wave,double offset) {
		int[] tiles = new int[2];
		double[] vars = new double[8];
		double velocityDifference =Math.abs(wave.velocity)-Math.abs(wave.lastVelocity); 
		int acceleration;
		if (velocityDifference<-0.5)
			acceleration=-1;
		else if (velocityDifference>0.5)
			acceleration=1;
		else acceleration=0;
		if (wave.lateralVelocity<0)
			offset=-offset;
		vars[0]=Math.round(offset);
		vars[1]=Math.abs(wave.lateralVelocity/2.0);
		vars[2]=acceleration;
		vars[3]=wave.distance/150;
		vars[4]=wave.wallAngle/0.2;
		vars[5]=wave.reverseWallAngle/0.35;
		vars[6]=wave.timer/0.2;
		vars[7]=wave.bulletVelocity;
		TileCoding.GetTiles(tiles,1,0,vars,8,tileWeights.length,0);
		vars[0]=Math.round(offset);
		vars[1]=Math.abs(wave.velocity/2.0);
		vars[2]=acceleration;
		vars[3]=wave.distance/300;
		vars[4]=wave.wallAngle/0.4;
		vars[5]=wave.timer/0.5;
		vars[6]=wave.bulletVelocity;
		TileCoding.GetTiles(tiles,1,1,vars,7,tileWeights.length,0);
		return tiles;
	}

    public void learn(Wave wave,double offset,double reward) {
		int[] tiles = getEnemyTiles(wave,offset);
		double result=0.0;
		for (int j=0;j<tiles.length;j++)
			result += tileWeights[tiles[j]];
		double alpha=0.1/tiles.length;  // step-size parameter
		for (int j=0;j<tiles.length;j++)
			tileWeights[tiles[j]] += alpha * (reward - result);
    }

    public double predict(Wave wave,double offset) {
		int[] tiles = getEnemyTiles(wave,offset);
		double result=0.0;
		for (int j=0;j<tiles.length;j++)
			result += tileWeights[tiles[j]];
		return result;
    }

    int[] getTiles(Wave wave,double offset) {
		int[] tiles = new int[5];
		double[] vars = new double[5];

		double velocityDifference =Math.abs(wave.velocity)-Math.abs(wave.lastVelocity); 
		int acceleration;
		if (velocityDifference<-0.5)
			acceleration=-1;
		else if (velocityDifference>0.5)
			acceleration=1;
		else acceleration=0;
		if (wave.lateralVelocity<0)
			offset=-offset;
		vars[0]=Math.round(offset);
		vars[1]=Math.abs(wave.lateralVelocity/2.0);
		vars[2]=acceleration;
		vars[3]=wave.firePower/0.5;
		vars[4]=wave.distance/150;

		TileCoding.GetTiles(tiles,1,0,vars,5,tileWeights.length,8);
		TileCoding.GetTiles(tiles,1,1,vars,4,tileWeights.length,9);
		TileCoding.GetTiles(tiles,1,2,vars,3,tileWeights.length,10);
		TileCoding.GetTiles(tiles,1,3,vars,2,tileWeights.length,11);
		TileCoding.GetTiles(tiles,1,4,vars,1,tileWeights.length,12);
		return tiles;
	}

    public void learnDanger (Wave wave,double offset,double danger) {
		int[] tiles = getTiles(wave,offset);

		double result=0.0;
		for (int j=0;j<tiles.length;j++)
			result += tileWeights[tiles[j]];
		double alpha=0.5/tiles.length;  // step-size parameter
		for (int j=0;j<tiles.length;j++)
			tileWeights[tiles[j]] += alpha * (danger - result);
    }

    public double danger(Wave wave,double offset) {
		int[] tiles = getTiles(wave,offset);

		double result=0.0;
		for (int j=0;j<tiles.length;j++)
			result += tileWeights[tiles[j]];
		return result;
    }
}

class Wave {
    Point2D startLocation;
    Point2D enemyLocation;
    double bulletVelocity;
    long startTime;
    double firePower;
    double lateralVelocity;
    double lastLateralVelocity;
    double approachVelocity;
    double distance;
    double wallAngle;
    double reverseWallAngle;
    double velocity;
    double lastVelocity;
    double timer;

    public Wave(Point2D startLocation, Point2D enemyLocation, double bulletVelocity,
            long startTime,double lateralVelocity,double lastLateralVelocity,double distance) {
        this.startLocation = startLocation;
        this.enemyLocation = enemyLocation;
        this.bulletVelocity = bulletVelocity;
        this.startTime = startTime;
        this.lateralVelocity = lateralVelocity;
        this.lastLateralVelocity = lastLateralVelocity;
        this.distance = distance;

    }

    public Wave(Point2D startLocation, Point2D enemyLocation, double bulletVelocity,
            long startTime,double lateralVelocity,double approachVelocity,double distance,
			double wallAngle,double reverseWallAngle,double lastLateralVelocity,double firePower,
			double velocity,double lastVelocity,double timer) {
        this.startLocation = startLocation;
        this.enemyLocation = enemyLocation;
        this.bulletVelocity = bulletVelocity;
        this.startTime = startTime;
        this.lateralVelocity = lateralVelocity;
        this.approachVelocity = approachVelocity;
        this.distance = distance;
        this.wallAngle = wallAngle;
        this.reverseWallAngle = reverseWallAngle;
        this.lastLateralVelocity = lastLateralVelocity;
        this.firePower = firePower;
        this.velocity = velocity;
        this.lastVelocity = lastVelocity;
        this.timer = timer;
    }

    public void onPaint(java.awt.Graphics2D g, long time, Color color) {
        g.setColor(color);
        double r = (time - startTime) * bulletVelocity;
        double x = startLocation.getX();
        double y = 600 - startLocation.getY();
        g.drawOval((int) (x - r + 0.5), (int) (y - r + 0.5), (int) (2 * r),
                (int) (2 * r));
    }
}

/*
 * Based on C code by Richard S. Sutton
 *
 * Translated by Stefan Westen
 *
 * See http://www.cs.ualberta.ca/~sutton/tiles.html
 *
 */

class TileCoding {

	final static int MAX_NUM_VARS = 20;

	private static int[] qstate = new int[MAX_NUM_VARS];

	private static int[] base = new int[MAX_NUM_VARS];

	private static int[] coordinates = new int[MAX_NUM_VARS + 4];
		/* one interval number per rel dimension */

	public static void GetTiles(int tiles[],int num_tilings,int tiling_offset, // provided array contains returned tiles (tile indices)
			double variables[], // array of variables
			int num_variables,
			int memory_size, // total number of possible tiles (memory size)
			int hash1) {
		int i, j;

		int num_coordinates;

		num_coordinates = num_variables + 2; // one additional hashing coordinates
		coordinates[num_variables + 1] = hash1;

		/* quantize state to integers (henceforth, tile widths == num_tilings) */
		for (i = 0; i < num_variables; i++) {
			qstate[i] = (int) Math.floor(variables[i] * num_tilings);
			base[i] = 0;
		}

		/*compute the tile numbers */
		for (j = 0; j < num_tilings; j++) {

			/* loop over each relevant dimension */
			for (i = 0; i < num_variables; i++) {

				/* find coordinates of activated tile in tiling space */
				if (qstate[i] >= base[i])
					coordinates[i] = qstate[i]
							- ((qstate[i] - base[i]) % num_tilings);
				else
					coordinates[i] = qstate[i] + 1
							+ ((base[i] - qstate[i] - 1) % num_tilings)
							- num_tilings;

				/* compute displacement of next tiling in quantized space */
				base[i] += 1 + (2 * i);
			}
			/* add additional indices for tiling and hashing_set so they hash differently */
			coordinates[i++] = tiling_offset+j;

			tiles[tiling_offset+j] = hash_coordinates(coordinates, num_coordinates,
					memory_size);
		}
		return;
	}

	static boolean first_call = true;
	static int[] rndseq = new int[2048];
	static Random random = new Random();

	private static int hash_coordinates(int[] coordinates, int num_indices, int memory_size)
	{
		int i,k;
		long index;
		long sum = 0;

		/* if first call to hashing, initialize table of random numbers */
	    if (first_call) {
			for (k = 0; k < 2048; k++) {
				rndseq[k] = 0;
				for (i=0; i < 4; ++i)
		    		rndseq[k] = (rndseq[k] << 8) | (random.nextInt()& 0xff);
			}
	        first_call = false;
	    }

		for (i = 0; i < num_indices; i++) {
			/* add random table offset for this dimension and wrap around */
			index = coordinates[i];
			index += (449 * i);
			index %= 2048;
			while (index < 0) index += 2048;

			/* add selected random number to sum */
			sum += (long)rndseq[(int)index];
		}
		index = (int)(sum % memory_size);
		while (index < 0) index += memory_size;

		return((int)index);
	}
}


