package nat;

import java.awt.*;
import java.awt.geom.*;
import java.util.*;

import robocode.*;
import robocode.util.*;
import chase.s2.stat.*;

public class BHGun {
	private final AdvancedRobot robot;

	private static int BINS = GunBuffer.GUN_BIN;
	private static int MIDDLE_BIN = BINS / 2;
	private Point2D.Double me;
	private Point2D.Double enemy;

	private int lastDirection = 1;
	private double lastVelocity = 0d;

	// Stats
	private static StatBufferSet gunStats = GunBuffer.getGunBuffer();
	private WeakHashMap<GunWave, StatData> dataMap = new WeakHashMap<GunWave, StatData>();

	// Dynamic Factor
	private static int bulletHit = 0, bulletFired = 0;
	private static double energyHit = 0, energyFired = 0;

	// Stats
	private ArrayList<GunWave> waves;
	private ArrayList<double[]> dataHistory;
	private ArrayList<Point2D.Double> locationHistory;

	public BHGun(AdvancedRobot r) {
		robot = r;

		waves = new ArrayList<GunWave>();
		dataHistory = new ArrayList<double[]>();
		locationHistory = new ArrayList<Point2D.Double>();
		
		dataHistory.add(new double[] {
			0,0,0,0,0
		});
	}

	public final void onScannedRobot(ScannedRobotEvent e) {
		double absBearing = e.getBearingRadians() + robot.getHeadingRadians();
		double lateralVelocity = e.getVelocity()
				* Math.sin(e.getHeadingRadians() - absBearing);
		
		me = new Point2D.Double(robot.getX(), robot.getY());
		enemy = project(me, absBearing, e.getDistance());
		locationHistory.add(0, enemy);

		int direction = (int) Math.signum(lateralVelocity);
		if (direction == 0)
			direction = lastDirection;
		lastDirection = direction;
		
		double bulletPower = Math.min(2d, Math.min(robot.getEnergy()/6d, e.getEnergy()/4d));
		if (robot.getEnergy() < 0.1) bulletPower = 0d;
		if (BlackHole.ISTC) bulletPower = 3d;
		
		GunWave w = new GunWave();
		w.fireLocation = me;
		w.fireTime = robot.getTime() + 1;
		w.directAngle = absBearing;
		w.direction = direction;
		w.bulletPower = bulletPower;
		w.bulletVelocity = bulletVelocity(w.bulletPower);
		
		waves.add(w);
		dataMap.put(w, new StatData(getDataSet(e, direction, absBearing, bulletPower),0,0));

		updateWaves();
		doGun(w);
		
		if (w.isFiring) {
			bulletFired ++;
			energyFired += w.bulletPower;
		}
	}
	
	public final void onBulletHit(BulletHitEvent e) {
		bulletHit++;
		energyHit += e.getBullet().getPower();
	}
	
	private final double[] getDataSet(ScannedRobotEvent e, int direction, double absBearing, double bulletPower) {
		double[] oldData = dataHistory.get(0);
		
		int BFT = (int)Math.ceil(e.getDistance() / (20 - 3*bulletPower));
		double MEA = Math.asin(8 / (20 - 3 * bulletPower));
		
		// Now we record all my robot information for surfing clustering
		// distance, vel, latvel, advvel, accl, tsvc, tsdc, tss, tsdecel
		// walldistf, walldistr, dl3, dl5, dl7, dl10, dl15, dl20, dl30
		double[] data = new double[18];
		data[0] = BFT;
		data[1] = Math.abs(e.getVelocity());
		data[2] = Math.abs(e.getVelocity() * Math.sin(e.getHeadingRadians() - absBearing));
		data[3] = e.getVelocity() * Math.cos(e.getHeadingRadians() - absBearing);
		data[4] = e.getVelocity() - lastVelocity;
		data[5] = (data[1] == oldData[0] ? oldData[1] + 1 : 0) / BFT;
		data[6] = (direction == lastDirection ? oldData[2] + 1 : 0) / BFT;
		data[7] = (data[1] != 0d ? oldData[3] + 1 : 0) / BFT;
		data[8] = (data[4] >= 0 ? oldData[4] + 1 : 0) / BFT;
		data[9] = wallDistance(enemy, e.getDistance(), absBearing, 1) / MEA;
		data[10] = wallDistance(enemy, e.getDistance(), absBearing, -1) / MEA;
		data[11] = getDistanceInLastNTick(3) * 4.167d;
		data[12] = getDistanceInLastNTick(5) * 2.5d;
		data[13] = getDistanceInLastNTick(7) * 1.7857d;
		data[14] = getDistanceInLastNTick(10) * 1.25d;
		data[15] = getDistanceInLastNTick(15) * 0.833d;
		data[16] = getDistanceInLastNTick(20) * 0.625d;
		data[17] = getDistanceInLastNTick(30) * 0.416d;
		
		// Stored Data Format:
		// Velocity, TSVC, TSDC, TSS, TSDECEL
		
		dataHistory.add(new double[] {
				data[1], data[5], data[6], data[7], data[8]
		});

		lastVelocity = e.getVelocity();
		
		return data;
	}

	private final void updateWaves() {
		for (int i = 0; i < waves.size(); i++) {
			GunWave ew = waves.get(i);

			ew.distanceTraveled = (robot.getTime() - ew.fireTime)
					* ew.bulletVelocity;
			if (ew.distanceTraveled > enemy.distance(ew.fireLocation)) {
				logHit(ew, enemy);
				waves.remove(i);
				i--;
			}
		}
	}
	
	private static final double getAngle(GunWave w, int bin) {
		return w.directAngle + (double)w.direction * (double)(bin - MIDDLE_BIN) * maxEscapeAngle(w.bulletVelocity) / (double)MIDDLE_BIN;
	}

	private final static double getGFFromAbsoluteAngle(GunWave ew, double angle) {
		double offsetAngle = (angle - ew.directAngle);
		return Utils.normalRelativeAngle(offsetAngle)
				/ maxEscapeAngle(ew.bulletVelocity) * ew.direction;
	}

	private final void logHit(GunWave ew, Point2D.Double targetLocation) {

		// Rough BotWidth calculation
		// TODO Precise intersection
		double botWidth = Math.atan(22 / ew.fireLocation
				.distance(targetLocation));
		double angle = absoluteBearing(ew.fireLocation, targetLocation);

		double lowerGF = getGFFromAbsoluteAngle(ew, Utils
				.normalAbsoluteAngle(angle - botWidth));
		double upperGF = getGFFromAbsoluteAngle(ew, Utils
				.normalAbsoluteAngle(angle + botWidth));

		double maxGF = Math.max(lowerGF, upperGF);
		double minGF = Math.min(lowerGF, upperGF);

		StatData data = dataMap.get(ew);
		if (data == null) {
			robot.out.println("Error! No wave data!");
			return;
		}

		data._lgf = minGF;
		data._ugf = maxGF;

		// Record new data
		gunStats.update(data, ew.isFiring ? 1f : 0.2f);
	}
	
	private final void doGun(GunWave w) {
		
		w.isFiring = robot.setFireBullet(w.bulletPower) != null;
		if (robot.getGunHeat() < 0.25) { 
			float[] stats = gunStats.getCombinedScoreData(dataMap.get(w));
			int max = MIDDLE_BIN;
			for (int i = 0; i < stats.length; i++)
				if (stats[i] > stats[max])
					max = i;
			double angle = getAngle(w, max);
			robot.setTurnGunRightRadians(Utils.normalRelativeAngle(angle - robot.getGunHeadingRadians()) + (Math.random() - 0.5) * 0.02);
		} else 
			robot.setTurnGunRightRadians(Utils.normalRelativeAngle(w.directAngle - robot.getGunHeadingRadians()));
		
	}

	private static final class GunWave {
		Point2D.Double fireLocation;
		long fireTime;
		double bulletVelocity, directAngle, distanceTraveled, bulletPower;
		int direction;
		boolean isFiring;
	}

	private static final Point2D.Double project(Point2D.Double sourceLocation,
			double angle, double length) {
		return new Point2D.Double(sourceLocation.x + Math.sin(angle) * length,
				sourceLocation.y + Math.cos(angle) * length);
	}

	private static final double absoluteBearing(Point2D.Double source,
			Point2D.Double target) {
		return Math.atan2(target.x - source.x, target.y - source.y);
	}

	private static final double limit(double min, double value, double max) {
		return Math.max(min, Math.min(value, max));
	}

	private static final double bulletVelocity(double power) {
		return (20D - (3D * power));
	}

	private static final double maxEscapeAngle(double velocity) {
		return Math.asin(8.0 / velocity);
	}
	
	private final double getDistanceInLastNTick(int tick) {
		tick = Math.min(tick, locationHistory.size()-1);
		return me.distance(locationHistory.get(tick));
	}

	// CREDIT: MORE STUFF BY SIMONTON =)
	double wallDistance(Point2D.Double sourceLocation, double eDist,
			double eAngle, int oDir) {
		return Math.min(Math.min(Math.min(distanceWest(robot.getBattleFieldWidth() - 25
				- sourceLocation.getY(), eDist, eAngle - Math.PI / 2, oDir),
				distanceWest(robot.getBattleFieldHeight() - 25 - sourceLocation.getX(), eDist,
						eAngle + Math.PI, oDir)), distanceWest(sourceLocation
				.getY() - 25, eDist, eAngle + Math.PI / 2, oDir)),
				distanceWest(sourceLocation.getX() - 25, eDist, eAngle, oDir));
	}

	double distanceWest(double toWall, double eDist, double eAngle, int oDir) {
		if (eDist <= toWall) {
			return Double.POSITIVE_INFINITY;
		}
		double wallAngle = Math.acos(-oDir * toWall / eDist) + oDir * Math.PI/2;
		return Utils.normalAbsoluteAngle(oDir * (wallAngle - eAngle));
	}

	public final void onPaint(java.awt.Graphics2D g) {
		g.setFont(new Font(Font.DIALOG, Font.PLAIN, 11));
		java.text.NumberFormat nf = java.text.NumberFormat.getNumberInstance();
		nf.setMaximumFractionDigits(2);
		nf.setMinimumFractionDigits(2);

		for (int i = 0; i < waves.size(); i++) {
			GunWave w = (waves.get(i));
			
			if (!w.isFiring) continue;
			
			Point2D.Double center = w.fireLocation;
			int radius = (int) w.distanceTraveled;

			g.setColor(Color.lightGray);
			int x = (int) Math.round(center.x);
			int y = (int) Math.round(center.y);
			g.drawOval(x - radius, y - radius, radius * 2, radius * 2);
			
			Point2D.Double hand = project(center, w.directAngle, radius);
			g.drawLine((int)center.x, (int)center.y, (int)hand.x, (int)hand.y);
			g.drawOval((int)hand.x-5, (int)hand.y-5, 10, 10);
		}
		
		for (int i = 0; i < waves.size(); i++) {
			GunWave w = (waves.get(i));
			
			if (!w.isFiring) continue;
			//if (w.isFiring) continue;
			
			Point2D.Double center = w.fireLocation;
			int radius = (int) w.distanceTraveled;
			double MEA = maxEscapeAngle(w.bulletVelocity);

			float[] _surfStats = gunStats.getCombinedScoreData(dataMap.get(w));
			if (_surfStats != null) {
				float[] normalized = new float[_surfStats.length];
				float max = 0f;
				for (float a : _surfStats)
					max = Math.max(a, max);
				if (!Utils.isNear(0d, max)) {
					for (int j = 0; j < _surfStats.length; j++) {
						normalized[j] = _surfStats[j] / max;
					}
				} else {
					Arrays.fill(normalized, 0);
				}
				for (int j = 0; j < BINS; j++) {
					double thisDanger = limit(0, normalized[j], 1);
					Point2D.Double thisPoint = project(center, w.directAngle
							+ w.direction * (j - MIDDLE_BIN)
							/ (double) MIDDLE_BIN * MEA, radius + 10);
					int x = (int) Math.round(thisPoint.x);
					int y = (int) Math.round(thisPoint.y);

					// Color-based wave
					 g.setColor(Color.getHSBColor(
					 .66f - 0.66f * (float) thisDanger, 1f, 1f));
					 g.fillOval(x - 2, y - 2, 4, 4);
					
					// Size-based wave
					// g.setColor(Color.orange);
					//int size = (int) Math.round(thisDanger * DANGER_SIZE_FACTOR);
					//int s = (int) Math.round(thisDanger * DANGER_SIZE_FACTOR2);
					//g.fillOval(x - s, y - s, size, size);
				}
			}
			g.setColor(Color.darkGray);
			int x = (int) Math.round(center.x);
			int y = (int) Math.round(center.y);
			g.drawOval(x - radius, y - radius, radius * 2, radius * 2);
			
			Point2D.Double hand = project(center, w.directAngle, radius);
			g.drawLine((int)center.x, (int)center.y, (int)hand.x, (int)hand.y);
			g.drawOval((int)hand.x-5, (int)hand.y-5, 10, 10);
		}
	}
}
