package sgp;
import robocode.*;
import java.util.*;
import java.lang.String;

/**
 * GameState - a class by Simon Parker
 */

public class GameState extends Object
{
	private double RECENT_HIT_TIME_THRESHOLD = 25;
	private double CLOSEST_TARGET_BULLET_AVOIDANCE_WEIGHT = 1.0;
	private double RECENT_HIT_BULLET_AVOIDANCE_WEIGHT = 1.0;
	private double RECENT_HIT_BEARING_WEIGHT = 0.1;
	private double NEMESIS_AVOIDANCE_WEIGHT = 0.1;
	private double BEARING_DIFFERENCE_COSINE_WEIGHT = 10;
	private double DISTANCE_BOUNDARY = 70.0;
	private double CROSSFIRE_WEIGHT = 1.0;
	private double IMPACT_POINT_STILL_WEIGHT = 3;
	private double IMPACT_POINT_LINEAR_WEIGHT = 1;
	private double IMPACT_POINT_CIRCULAR_WEIGHT = 1;

	private double TARGET_DISTANCE_WEIGHT = -1.0;
	private double TARGET_ENERGY_WEIGHT = -0.5;
	private double TARGET_DAMAGE_INFLICTED_WEIGHT = 0.0;
	private double TARGET_DAMAGE_SUSTAINED_WEIGHT = 0.0;
	private double TARGET_LAST_HIT_TIME_WEIGHT = 0.0;
	private double TARGET_LAST_BULLET_POWER = 0.0;
	private double TARGET_DEATH_COUNT_WEIGHT = -1.0;
	private double TARGET_HIT_RATE_WEIGHT = 1;
	private double TARGET_DE_WEIGHT = -1.0;
	private double TARGET_CLOSEST_ROBOT_DISTANCE = -1.0;
	private double TARGET_PREDICTION_ERROR_WEIGHT = -1.0;

	private double AVOIDANCE_EXTRA_DISTANCE = 100;
	private double AVOIDANCE_DISTANCE_WEIGHT = 100;

//???	private TargetTable targetTable = new TargetTable();

	//We'll have a hash table that stores indexes references by robot names
	// and a vector that stores the target info at the referenced indexes.
	Hashtable targetIndexTable = new Hashtable();
	Vector targetVector = new Vector();
	int nextTargetIndex = 0;
	JollyNinja robot;

	public GameState()
	{
	}

	public void reset()
	{
		robot = JollyNinja.getInstance();

		//set all Targets to be alive
		for (int i = 0; i < targetVector.size(); i++)
		{
			Target target = (Target)targetVector.elementAt(i);
			target.reset();
		}

		RECENT_HIT_TIME_THRESHOLD = robot.parameters.recentHitTimeThreshold.getValue();
		CLOSEST_TARGET_BULLET_AVOIDANCE_WEIGHT = robot.parameters.closestTargetBulletAvoidanceWeight.getValue();
		RECENT_HIT_BULLET_AVOIDANCE_WEIGHT = robot.parameters.recentHitBulletAvoidanceWeight.getValue();
		RECENT_HIT_BEARING_WEIGHT = robot.parameters.recentHitBearingWeight.getValue();
		NEMESIS_AVOIDANCE_WEIGHT = robot.parameters.nemesisAvoidanceWeight.getValue();
		BEARING_DIFFERENCE_COSINE_WEIGHT = robot.parameters.bearingDifferenceCosineWeight.getValue();
		DISTANCE_BOUNDARY = robot.parameters.crossfireDistanceBoundary.getValue();
		CROSSFIRE_WEIGHT = robot.parameters.crossfireWeight.getValue();
		IMPACT_POINT_STILL_WEIGHT = robot.parameters.impactPointStillWeight.getValue();
		IMPACT_POINT_LINEAR_WEIGHT = robot.parameters.impactPointLinearWeight.getValue();
		IMPACT_POINT_CIRCULAR_WEIGHT = robot.parameters.impactPointCircularWeight.getValue();

	 	TARGET_DISTANCE_WEIGHT = robot.parameters.targetDistanceWeight.getValue();
	 	TARGET_ENERGY_WEIGHT = robot.parameters.targetEnergyWeight.getValue();
	 	TARGET_DAMAGE_INFLICTED_WEIGHT = robot.parameters.targetDamageToMeWeight.getValue();
	 	TARGET_DAMAGE_SUSTAINED_WEIGHT = robot.parameters.targetDamageToHimWeight.getValue();
	 	TARGET_LAST_HIT_TIME_WEIGHT = robot.parameters.targetLastTimeHitWeight.getValue();
	 	TARGET_LAST_BULLET_POWER = robot.parameters.targetLastBulletPowerWeight.getValue();
	 	TARGET_DEATH_COUNT_WEIGHT = robot.parameters.targetDeathCountWeight.getValue();
	 	TARGET_HIT_RATE_WEIGHT = robot.parameters.targetHitRateWeight.getValue();
	 	TARGET_DE_WEIGHT = robot.parameters.targetDeltaEnergyWeight.getValue();
	 	TARGET_CLOSEST_ROBOT_DISTANCE = robot.parameters.targetClosestRobotDistanceWeight.getValue();

		AVOIDANCE_EXTRA_DISTANCE = robot.parameters.avoidanceExtraDistance.getValue();
		AVOIDANCE_DISTANCE_WEIGHT = robot.parameters.avoidanceDistanceWeight.getValue();
		
		TARGET_PREDICTION_ERROR_WEIGHT = robot.parameters.targetPredictionErrorWeight.getValue();


	}

	public void updateTarget(ScannedRobotEvent e)
	{
		Target target = getTarget(e.getName());

		target.update(e, Environment.getTime());

		//update all of the closest target distances
		setClosestTargets(Environment.getRobotPosition());
	}

	public Target getTarget(String targetName)
	{
		Target target;
		Integer targetIndex = (Integer)targetIndexTable.get(targetName);
		if (targetIndex == null)
		{
			target = new Target("Melee", targetName, robot);
			targetIndexTable.put(targetName, new Integer(nextTargetIndex));
			targetVector.add(target);
			nextTargetIndex++;
		}
		else
		{
			target = (Target)targetVector.elementAt(targetIndex.intValue());
		}

		return target;
	}

	public int getNum()
	{
		return targetVector.size();
	}





	public void setRobotDead(String targetName)
	{
		Target target = getTarget(targetName);
		target.setDead();
		robot.out.println("DEAD " + targetName + " " + target.deathCount + ", " + (int)target.damageInflicted);
	}

	public double getGoodness(Coordinate robotPosition, double currentTime)
	{
		double goodness = 0;
		
		goodness += Strategy.evaluateWallGoodness(robotPosition);

		goodness += getCrossfireGoodness(robotPosition);

	   	goodness += getRecentAttackerGoodness(robotPosition, currentTime);

	   	goodness += getAvoidanceGoodness(robotPosition);

		//keep away from potential bullets from the closest robot.
		Target closestTarget = getClosestTarget(robotPosition);
		if (closestTarget != null)
		{

			goodness += CLOSEST_TARGET_BULLET_AVOIDANCE_WEIGHT * closestTarget.getImpactDistance(robotPosition, currentTime);
			//???goodness += CLOSEST_TARGET_BULLET_AVOIDANCE_WEIGHT * closestTarget.bulletList.getBulletAvoidanceGoodness(robotPosition, currentTime);

			//avoid hitting other robots
			double distance = closestTarget.position.distanceFrom(robotPosition);
			goodness += 100 * Math.min(distance, 60);
		}

/*
		//keep the bearing to the closest robot aligned with it's heading
		Target bestTarget = getBestTarget(robotPosition);
		goodness -= BEARING_DIFFERENCE_COSINE_WEIGHT * Math.abs(Math.sin(Math.toRadians(bestTarget.heading_deg - robot.getHeading())));
*/
		//keep away from the most dangerous opponents
		Target nemesisTarget = getMostDangerousTarget(robotPosition);
		if (nemesisTarget != null)
		{
			goodness += NEMESIS_AVOIDANCE_WEIGHT * robot.getOthers() / 10.0 * nemesisTarget.position.distanceFrom(robotPosition);
		}


		return goodness;
	}

	private double getCrossfireGoodness(Coordinate robotPosition)
	{
		//make it the sum of the distances from the lines connecting
		//the live robots, if there is more than one

		double goodness = 0.0;
		if (getNumAlive() < 2) return 0.0;

		for (int i = 0; i < (getNum() - 1); i++)
		{
			Target target1 = (Target)targetVector.elementAt(i);
			if (!target1.isAlive) continue;

			for (int j = i + 1; j < getNum(); j++)
			{
				Target target2 = (Target)targetVector.elementAt(j);
				if (!target2.isAlive) continue;

				double distance = robotPosition.distanceFrom(target1.position, target2.position);
				distance = Math.min(distance, DISTANCE_BOUNDARY);
				goodness += distance;
			}
		}

		return CROSSFIRE_WEIGHT * goodness;

	}

	private double getRecentAttackerGoodness(Coordinate robotPosition, double currentTime)
	{
		double goodness = 0.0;
		//keep away from potetial bullets from robots that have recently hit me
		for (int i = 0; i < getNum(); i++)
		{
			Target target = (Target)targetVector.elementAt(i);

			if (!target.isAlive) continue;
			if 	((currentTime - target.lastBulletHitTime) < RECENT_HIT_TIME_THRESHOLD)
			{
				goodness += RECENT_HIT_BULLET_AVOIDANCE_WEIGHT * target.getImpactDistance(robotPosition, currentTime);
				//???goodness += RECENT_HIT_BULLET_AVOIDANCE_WEIGHT * target.bulletList.getBulletAvoidanceGoodness(robotPosition, currentTime);
				// angle difference between 90 degrees off the bearing to the other robots that
				//  have recently hit me
				double bearingToTarget = Strategy.normalRelativeAngle(robotPosition.headingTo(target.position) - Environment.getHeading());
				goodness += RECENT_HIT_BEARING_WEIGHT * Math.abs(Math.abs(bearingToTarget) - 90);

			}
		}
		return goodness;

	}

	private double getAvoidanceGoodness(Coordinate robotPosition)
	{
		double goodness = 0.0;
		//keep away from potetial bullets from robots that have recently hit me
		for (int i = 0; i < getNum(); i++)
		{
			Target target = (Target)targetVector.elementAt(i);

			if (!target.isAlive) continue;

			double distance = target.position.distanceFrom(robotPosition);

			Target closestTargetToTarget = getClosestTargetToTarget(target);
			if (closestTargetToTarget != null)
			{
				double d2 = closestTargetToTarget.position.distanceFrom(target.position);
				goodness += AVOIDANCE_DISTANCE_WEIGHT * Math.min(distance, AVOIDANCE_EXTRA_DISTANCE + d2);
			}
		}
		return goodness;

	}

	public int getNumAlive()
	{
		int aliveCount = 0;

		for (int i = 0; i < getNum(); i++)
		{
			Target target1 = (Target)targetVector.elementAt(i);
			if (target1.isAlive) aliveCount++;
		}

		return aliveCount;
	}

	public void updateImpacts(Coordinate robotPosition, double robotVelocity, double robotHeading, double robotAngularVelocity, double currentTime)
	{
		for (int i = 0; i < getNum(); i++)
		{
			Target target1 = (Target)targetVector.elementAt(i);
			if (target1.isAlive)
			{
				target1.updateImpactPoints(robotPosition, robotVelocity, robotHeading, robotAngularVelocity, currentTime, IMPACT_POINT_CIRCULAR_WEIGHT);
				target1.updateImpactPoints(robotPosition, robotVelocity, robotHeading, 0, currentTime, IMPACT_POINT_LINEAR_WEIGHT);
				target1.updateImpactPoints(robotPosition, 0, robotHeading, 0, currentTime, IMPACT_POINT_STILL_WEIGHT);
				target1.updateInterceptImpactPoints(robotPosition, robotVelocity, robotHeading, robotAngularVelocity, currentTime, IMPACT_POINT_CIRCULAR_WEIGHT);

//???				target1.updateImpactPoints(robotPosition, robotVelocity, robotHeading, robotAngularVelocity, currentTime);

			}
		}
	}

	public void onBulletHit(BulletHitEvent event)
	{
		for (int i = 0; i < getNum(); i++)
		{
			Target target = (Target)targetVector.elementAt(i);
			target.onBulletHit(event);
		}
	}


	private Target getWeightedTarget(TargetWeight weight)
	{
		//chooses the maximum wieghted value for a target

		Target bestTarget = null;
		double targetValue = 0;
		double comparedTargetValue = 0;
		
		double currentTime = Environment.getTime();

		for (int i = 0; i < targetVector.size(); i++)
		{
			Target comparedTarget = (Target)targetVector.elementAt(i);

			if (bestTarget == null)
			{
				double dT = comparedTarget.lastTimeScanned - currentTime;
				//have not found the first alive target yet
				if ((comparedTarget.isAlive) && (dT < 15))
				{
					bestTarget = comparedTarget;
					targetValue = comparedTarget.getWeightedValue(weight);
				}
			}
			else
			{
				//compare with current best
				if (comparedTarget.getIsValid(weight.robotPosition, currentTime))
				{
					comparedTargetValue = comparedTarget.getWeightedValue(weight);
					if (comparedTargetValue > targetValue)
					{
						bestTarget = comparedTarget;
						targetValue = comparedTargetValue;
					}
				}
			}
		}

		return bestTarget;
	}

	public Target getMostDangerousTarget(Coordinate robotPosition)
	{
		TargetWeight weight = new TargetWeight(robotPosition, 0);
		weight.kDamageInflictedPerRound = 10.0;
		weight.kDeathPerRound = -50.0;
		weight.kTimeSinceLastBulletHit = -1.0;

		return getWeightedTarget(weight);
	}

	public Target getClosestTargetToTarget(Target focusedTarget)
	{
		TargetWeight weight = new TargetWeight(focusedTarget.position, 0);
		weight.kDistance = -1.0;

		Target closestTarget = getWeightedTarget(weight);

		if (closestTarget == focusedTarget)
		{
			closestTarget = null;
		}

		return closestTarget;
	}

	public Target getMostOutDatedTarget()
	{
		Coordinate currentRobotPosition = Environment.getRobotPosition();
		TargetWeight weight = new TargetWeight(currentRobotPosition, Environment.getHeading());
		weight.kTimeSinceLastScanned = 1.0;

		return getWeightedTarget(weight);
	}

	public Target getClosestTarget(Coordinate robotPosition)
	{
		TargetWeight weight = new TargetWeight(robotPosition, 0);
		weight.kDistance = -1.0;

		return getWeightedTarget(weight);
	}

	public Target getBestTarget(Coordinate robotPosition)
	{
		Target chosenTarget = getRuleChosenTarget(robotPosition);
		if (chosenTarget != null) return chosenTarget;
		
		//used for target selection (to shoot at)
		TargetWeight weight = new TargetWeight(robotPosition, Environment.getHeading());
		weight.kDistance = TARGET_DISTANCE_WEIGHT;
		weight.kEnergy = TARGET_ENERGY_WEIGHT;
		weight.kDamageInflictedPerRound = TARGET_DAMAGE_INFLICTED_WEIGHT;
		weight.kDamageSustainedPerRound = TARGET_DAMAGE_SUSTAINED_WEIGHT;
		weight.kTimeSinceLastBulletHit = TARGET_LAST_HIT_TIME_WEIGHT;
		weight.kLastBulletPower = TARGET_LAST_BULLET_POWER;
		weight.kHitRate = TARGET_HIT_RATE_WEIGHT;
		weight.kDeltaEnergyEstimate = TARGET_DE_WEIGHT;
		weight.kDeathPerRound = TARGET_DEATH_COUNT_WEIGHT;
		weight.kClosestRobotDistance = TARGET_CLOSEST_ROBOT_DISTANCE;
		weight.kPredictionError = TARGET_PREDICTION_ERROR_WEIGHT;

		return getWeightedTarget(weight);
	}
	
	private Target getRuleChosenTarget(Coordinate robotPosition)
	{
		Target chosenTarget = null;
		//if very close
		//if it has hit me recently and is close
		//if it is close to another robot
		//otherwise whoever died the least

		//if very close
		chosenTarget = getClosestTarget(robotPosition);
		if (chosenTarget == null) return null;
		
		if (robotPosition.distanceFrom(chosenTarget.position) < 100)
		{
			return chosenTarget;
		}
		
		return null;
		
//		
//		
//		//if it has hit me recently and is close
//		TargetWeight weight = new TargetWeight(robotPosition, Environment.getHeading());
//		weight.kTimeSinceLastBulletHit = -1.0;
//		chosenTarget = getWeightedTarget(weight);
//		
//		double dT = Environment.getTime() - chosenTarget.lastBulletHitTime;
//		if ((robotPosition.distanceFrom(chosenTarget.position) < 300) && (dT < 100))
//		{
//			return chosenTarget;
//		}
//		
//		//if it is close to another robot
//		weight.reset();
//		weight.kClosestRobotDistance = -1.0;
//		chosenTarget = getWeightedTarget(weight);
//
//		if (chosenTarget.closestRobotDistance < 150)
//		{
//			return chosenTarget;
//		}
//		
//		//otherwise whoever died the least
//		weight.reset();
//		weight.kDeathPerRound = -1.0;
//		chosenTarget = getWeightedTarget(weight);
//
//		return chosenTarget;
	}

	public void setClosestTargets(Coordinate robotPosition)
	{
		long currentTime = Environment.getTime();
		for (int i = 0; i < targetVector.size(); i++)
		{
			Target target = (Target)targetVector.elementAt(i);
			//compare with current best
			if (target.getIsValid(robotPosition, currentTime))
			{
				Target closestTarget = getClosestTargetToTarget(target);

				if (closestTarget != null)
				{
					target.closestRobotDistance = target.position.distanceFrom(closestTarget.position);
					target.closestRobotPosition = closestTarget.position;
					
					if (target.closestRobotDistance < closestTarget.closestRobotDistance)
					{
						closestTarget.closestRobotPosition = target.position;
						closestTarget.closestRobotDistance = target.closestRobotDistance;
					}
				}
			}
		}
	}
	
	public void onHitByBullet(HitByBulletEvent e)
	{
		//robot is hit by a certain target.
		Target bulletFirer = getTarget(e.getName());
		
		bulletFirer.onHitByBullet(e);

	}
	
	public void save()
	{
		for (int i = 0; i < getNum(); i++)
		{
			Target target1 = (Target)targetVector.elementAt(i);
			target1.save();
		}
	}
}


