package ags.rougedc.robots;

import robocode.Event;
import robocode.RobotStatus;
import robocode.StatusEvent;
import static robocode.util.Utils.normalRelativeAngle;
import ags.rougedc.base.*;
import ags.utils.points.AbsolutePoint;
import ags.utils.points.RelativePoint;
import java.util.List;
import java.util.ArrayList;

/**
 * Transforms StatusEvent into SelfRobotEvent.
 * Stores history of itself. Cloning is "shallow" in that it doesn't clone history
 * 
 * @author Alexander Schultz
 */
public class StatusRobot implements VirtualRobot {
    private static final int history_length = 2;
    
    private final IntentRobotPredictor intent;
    private final Rules rules;
    
    private long time;
    private RobotStatus status;
    private AbsolutePoint location;
    private RelativePoint velocity;
    private final List<StatusRobot> history;
    private double angularvelocity;
    
    public StatusRobot(Rules rules) {
        this.rules = rules;
        history = new ArrayList<StatusRobot>();
        intent = new IntentRobotPredictor(this, AbsolutePoint.fromXY(rules.BATTLEFIELD_WIDTH, rules.BATTLEFIELD_HEIGHT));
    }
    
    // Cloning constructor
    private StatusRobot(StatusRobot source) {
        this.rules = source.rules;
        this.time = source.time;
        this.status = source.status;
        if (source.location != null)
            this.location = (AbsolutePoint)source.location.clone();
        if (source.velocity != null)
            this.velocity = (RelativePoint)source.velocity.clone();
        history = new ArrayList<StatusRobot>(source.history);
        intent = new IntentRobotPredictor(this, AbsolutePoint.fromXY(rules.BATTLEFIELD_WIDTH, rules.BATTLEFIELD_HEIGHT));
    }
    public StatusRobot clone() {
        return new StatusRobot(this);
    }
    
    public long getTime() {
        return time;
    }
    
    public double getEnergy() {
        return status.getEnergy();
    }
    public AbsolutePoint getLocation() {
        return location;
    }
    public AbsolutePoint getNextLocation() {
        return intent.getNextLocation();
    }
    public RelativePoint getVelocity() {
        return velocity;
    }
    public double getGunHeading() {
        return status.getGunHeadingRadians();
    }
    public double getRadarHeading() {
        return status.getRadarHeadingRadians();
    }
    public double getHeading() {
        return velocity.getDirection();
    }
    public double getGunHeat() {
        return status.getGunHeat();
    }
    public double getAngularVelocity() {
        return angularvelocity;
    }
    
    public void handleEvents(Iterable<Event> i) {
        for (Event e : i)
            if (e instanceof StatusEvent) {
                StatusEvent event = (StatusEvent)e;
                
                history.add(this.clone());
                if (history.size() > history_length)
                    history.remove(0);
                
                if (velocity != null)
                    angularvelocity = normalRelativeAngle(status.getHeadingRadians() - velocity.direction);
                
                time = event.getTime();
                status = event.getStatus();
                location = AbsolutePoint.fromXY(status.getX(), status.getY());
                velocity = RelativePoint.fromDM(status.getHeadingRadians(), status.getVelocity());
            }
    }
    
    // For setting the intended movement!
    public void setMove(double arg0) {
        intent.setMove(arg0);
    }
    public void setTurnBody(double arg0) {
        intent.setTurnBody(arg0);
    }
    
    // Returns the StatusRobot so many ticks ago
    public StatusRobot ticksAgo(int ago) {
        if (ago == 0)
            return this.clone();
        return history.get(ago-1);
    }
}
