/*
 * Created on Oct 25, 2004
 *
 */
package davidalves.net.movement;

import java.awt.Color;
import java.util.*;

import davidalves.PhoenixOS;
import davidalves.net.*;
import davidalves.net.gun.*;
import davidalves.net.gun.segmentation.*;
import davidalves.net.util.*;

import robocode.*;

/**
 * @author David Alves
 *
 */
public class WaveSurfing {
	
	private boolean clockwise = true;
	
	public static final int MAX_WAVES_TO_SURF = 2;
	
	public double damageInflictedByKnownWaves;
	public double damageInflictedByUnknownWaves;
	
	private static final int GF0 = 15;
	private static final double DONT_SURF_THRESHOLD = -1;
	private static final double DISCARD_WAVE_THRESHOLD = -2;
	
	private boolean isHOT = true;
	
	private boolean enemyLocated = false;
	private RobotState lastKnownState;
	
	private GuessFactorGunManager enemyRealGuns;
	
	private ArrayList enemyRealWaves;
	
	private Point destination;
	
	private WallSmoother wallSmoother;
	
	private double battleFieldWidth, battleFieldHeight;
	
	private Point timeToImpactComparatorPoint;
	private long timeToImpactComparatorTime;

	private Comparator timeToImpactComparator = new Comparator(){
		
		public int compare(Object o1, Object o2){
			Wave w1 = (Wave) o1;
			Wave w2 = (Wave) o2;
			return w1.timeToImpact(timeToImpactComparatorPoint, timeToImpactComparatorTime) > w2.timeToImpact(timeToImpactComparatorPoint, timeToImpactComparatorTime) ? 1 : -1;
		}
		
	};
	
	
	
	public WaveSurfing(){
		
		GuessFactorGun[] guns = new GuessFactorGun[2];
		guns[0] = new GuessFactorGun(
				new Segmentation[]{
					new AccelerationSegmentation(3),
					new SpeedSegmentation(3),
				}, GF0, .7f, true);
		guns[1] = new GuessFactorGun(
				new Segmentation[]{
					new AccelerationSegmentation(3),
					new OrbitalWallsSegmentation(1, .6, 3),
					new OrbitalWallsSegmentation(1, -.4, 2),
					new SpeedSegmentation(4),
					new TimeAtVelocityFactorSegmentation(3),
					new NonlinearDistanceSegmentation(),
				}, GF0, .7f, false);

		
		enemyRealGuns = new GuessFactorGunManager(guns,1);
		
		guns = new GuessFactorGun[1];
		
		
		enemyRealWaves = new ArrayList();
		
		//Someday I'll get wall smoothing down right...
		//wallSmoother = new RaikoMXWallSmoother(); 
		//wallSmoother = new NoFearTheWallsWallSmoother();
		wallSmoother = new LimitedWallSmoother();
		//wallSmoother = new HybridWallSmoother();
		//wallSmoother = new CircularWallSmoother();
	}
	
	public void init(){
		battleFieldWidth = PhoenixOS.bot.getBattleFieldWidth();
		battleFieldHeight = PhoenixOS.bot.getBattleFieldHeight();
		
		lastKnownState = new RobotState();
		lastKnownState.x = PhoenixOS.bot.getBattleFieldWidth() / 2.0;
		lastKnownState.y = PhoenixOS.bot.getBattleFieldHeight() / 2.0;

	}
	
	public void reset() {
		enemyLocated = false;
		enemyRealWaves.clear();
		destination = new Point(PhoenixOS.bot.getX(), PhoenixOS.bot.getY());
	}
	
	public void manageWaves(RobotState enemy, RobotState me, long now) {
		
		Wave wave = null;
		double soonestImpact = Double.POSITIVE_INFINITY;
		Iterator i = enemyRealWaves.iterator();
		while(i.hasNext()){
			wave = (Wave) i.next();
			double timeToImpact = wave.timeToImpact(me, now);
			if(timeToImpact < soonestImpact) {
				//orbitPoint = (Point) wave.shooterAtFireTime;
				soonestImpact = timeToImpact;
			}
			if (timeToImpact < DISCARD_WAVE_THRESHOLD){
				//Phoenix.info("Discarding wave! dist traveled: " + wave.distanceTraveled(now) + " dist to me from origin: " + wave.shooterAtFireTime.distanceTo(me));
				//Phoenix.info("shooter" + wave.shooterAtFireTime);
				
				//Phoenix.info("me: " + me);
				i.remove();
			}
		}
	}
	
	public Point getDestination(){
			return destination;
	}
	
	public void chooseDestination(RobotState me, RobotState enemy, long now){
		
		
		if(enemy == null && enemyLocated){
			enemy = lastKnownState;
		} else if (enemy != null){
			lastKnownState = enemy;
			enemyLocated = true;
		} else if(!enemyLocated && enemy == null){
			return;
		}
		//if(true){
		if(enemyRealWaves.size() > 0){
			double cwDanger = 0;
			double stopDanger = Double.POSITIVE_INFINITY;
			double ccwDanger = 0;
			
			Point orbitPoint = enemy;
			Point cwDirection = wallSmoother.getSmoothedDestination(me, orbitPoint, 1.0);
			Point ccwDirection = wallSmoother.getSmoothedDestination(me, orbitPoint, -1.0);
			
			LinkedList waveList = new LinkedList();
			waveList.addAll(enemyRealWaves);
			timeToImpactComparatorPoint = me;
			timeToImpactComparatorTime = now;
			Collections.sort(waveList, timeToImpactComparator);
			while(waveList.size() > MAX_WAVES_TO_SURF) waveList.removeLast();
			
			//Run away!
			cwDanger = getDanger(me, 1.0, now, 0, waveList, true);
			//stopDanger = getDanger(me, 0.0, now, 0, waveList);
			ccwDanger = getDanger(me, -1.0, now, 0, waveList, true);
			
			
			if(ccwDanger < cwDanger && ccwDanger < stopDanger){
				destination = ccwDirection;
				clockwise = false;
			} else if(cwDanger < ccwDanger && cwDanger < stopDanger){
				destination = cwDirection;
				clockwise = true;
			} else {
				if(clockwise){
					destination = cwDirection;
				} else {
					destination = ccwDirection;
				}
			}
		} else { // no waves to surf, retreat!
			if(PhoenixOS.bot.getDistanceRemaining() < 50){
				Point cwDirection = wallSmoother.getSmoothedDestination(me, enemy, 1.0);
			
				Point ccwDirection = wallSmoother.getSmoothedDestination(me, enemy, -1.0);
				if(enemy.distanceTo(cwDirection) > enemy.distanceTo(ccwDirection)){
					destination = cwDirection;
					clockwise = true;
				} else {
					destination = ccwDirection;
					clockwise = false;
				}
			}
		}
	}
	public static double[] getScores(float[] realData, float[] flattenerData, int guessFactor0, double bulletPower, double waveDistanceTraveled){
		double[] scores = new double[guessFactor0 * 2 + 1];
		for(int i = 0; i < scores.length; i++){
			scores[i] = getScore(realData, flattenerData, i, guessFactor0, bulletPower, waveDistanceTraveled);
		}
		return scores;
	}
	
	public static double getScore(float[] realData, float[] flattenerData, int GF, int guessFactor0, double bulletPower, double waveDistanceTraveled){
		float score = 0;
		double max = 0;
		double pixelsPerGF = waveDistanceTraveled * Utils.maxEscapeAngle(bulletPower) / realData.length;
		for(int i = 0; i < realData.length; i++){
			double distance = 1 + pixelsPerGF * Math.abs(GF - i) / guessFactor0;
			//score += (realData[i] * flattenerData[i]) / Math.sqrt(distance);
			double oneOverSqrtDistance = 1.0 / Math.sqrt(distance);
			score += realData[i] * oneOverSqrtDistance;
			max += oneOverSqrtDistance;
		}
		return score / max;
	}
	
	private double getDanger(MotionState state, double orbitDirection, long time, int depth, LinkedList waveList, boolean sameDirectionForFullTree){
		if(waveList.size() == 0){
			//Phoenix.info("Max depth reached, time: " + time);
			//return getDanger(state,time);
			return 0;
		}
		
		
		MotionState projectedState = state;
		//MotionState predictedState = getNextMotionState(state, lastKnownState, orbitDirection);
		double danger = 0;
		
		//Find wave that will hit us first
		Wave surfedWave = (Wave) waveList.getFirst();
		
		do{
			time++;
			projectedState = getNextMotionState(projectedState, lastKnownState, orbitDirection);
		} while(surfedWave.timeToImpact(projectedState, time + 1) > DONT_SURF_THRESHOLD);
		
		GuessFactorGun gunForThisWave = enemyRealGuns.getBestGun(surfedWave.shooterAtFireTime, surfedWave.targetAtFireTime, surfedWave.power);
		int[] realGunSegments = gunForThisWave.getSegments(surfedWave.shooterAtFireTime, surfedWave.targetAtFireTime, surfedWave.power);
		float[] realGunData = gunForThisWave.getData(realGunSegments);
		
		danger = getScore(realGunData, null, surfedWave.getSolutionGF(projectedState, GF0), GF0, surfedWave.power, surfedWave.distanceTraveled(time));
		LinkedList nextList = new LinkedList();
		nextList.addAll(waveList);
		nextList.removeFirst();
		return danger + Math.min(getDanger(projectedState, 1.0, time, depth+1, nextList, sameDirectionForFullTree && orbitDirection == 1.0), getDanger(projectedState, -1.0, time, depth+1, nextList, sameDirectionForFullTree && orbitDirection == -1.0));
		
	}
	
	private MotionState getNextMotionState(MotionState state, Point orbitPoint, double orbitDirection){
		Point projectedDestination = wallSmoother.getSmoothedDestination(state, orbitPoint, orbitDirection);
		double distanceRemaining = state.distanceTo(projectedDestination);
		double angleToTurn = Utils.normalRelativeAngle(state.absoluteAngleTo(projectedDestination) - state.heading);
		
		if (Math.abs(angleToTurn) > Math.PI / 2.0) {
			distanceRemaining *= -1;
			if (angleToTurn > 0) {
				angleToTurn -= Math.PI;
			}
			else {
				angleToTurn += Math.PI;
			}
		}
		
		double velocity = state.velocity;
		double heading = state.heading;
		double x = state.x;
		double y = state.y;
		
		double maxVelocity = Math.min(Constants.SPEED_MAX,PhoenixOS.SHARP_TURN_FACTOR / Math.toDegrees(Math.abs(angleToTurn)));
		
		double acceleration = 0;
		boolean slowingDown = false;
		double moveDirection;
	
		if (distanceRemaining == 0) moveDirection = 0; else if (distanceRemaining < 0.0) moveDirection = -1; else moveDirection = 1;
		
		double turnRate = Math.toRadians(10 - .75 * velocity);
		if (angleToTurn > 0.0) {
    		if (angleToTurn < turnRate) { heading += angleToTurn; angleToTurn = 0.0; } 
			else { heading += turnRate; angleToTurn -= turnRate; }
		} else if (angleToTurn < 0.0) {
    		if (angleToTurn > -turnRate) { heading += angleToTurn; angleToTurn = 0.0; } 
			else { heading -= turnRate;	angleToTurn += turnRate; }
		}
		heading = Utils.normalAbsoluteAngle(heading);
		
		
		if (distanceRemaining != 0.0 || velocity != 0.0) { 
			if (!slowingDown && moveDirection == 0) {
				slowingDown = true;
				if (velocity > 0.0) moveDirection = 1;
				else if (velocity < 0.0) moveDirection = -1;
				else moveDirection = 0;
		    }
		    double desiredDistanceRemaining = distanceRemaining;
    		if (slowingDown) {
				if (moveDirection == 1 && distanceRemaining < 0.0) desiredDistanceRemaining = 0.0;
				else if (moveDirection == -1 && distanceRemaining > 1.0) desiredDistanceRemaining = 0.0;
		    }
    		double slowDownVelocity	= (int) (Constants.MAX_BRAKING / 2.0 * ((Math.sqrt(4.0 * Math.abs(desiredDistanceRemaining)+ 1.0)) - 1.0));
	    	if (moveDirection == -1) slowDownVelocity = -slowDownVelocity;
	    	if (!slowingDown) {
				if (moveDirection == 1) {
	    			if (velocity < 0.0) acceleration = Constants.MAX_BRAKING;
		    		else acceleration = Constants.MAX_ACCELERATION;
			    	if (velocity + acceleration > slowDownVelocity) slowingDown = true;
				} else if (moveDirection == -1) {
			    	if (velocity > 0.0) acceleration = -Constants.MAX_BRAKING;
	    			else acceleration = -Constants.MAX_ACCELERATION;
	    			if (velocity + acceleration < slowDownVelocity)	slowingDown = true;
				}
		    }
    		if (slowingDown) {
				if (distanceRemaining != 0.0 && Math.abs(velocity) <= Constants.MAX_BRAKING && Math.abs(distanceRemaining) <= Constants.MAX_BRAKING) slowDownVelocity = distanceRemaining;
				double perfectAccel = slowDownVelocity - velocity;
				if (perfectAccel > Constants.MAX_BRAKING) perfectAccel = Constants.MAX_BRAKING;
				else if (perfectAccel < -Constants.MAX_BRAKING) perfectAccel = -Constants.MAX_BRAKING;
				acceleration = perfectAccel;
	    	}
			if (velocity > maxVelocity || velocity < -maxVelocity) acceleration = 0.0;
			velocity += acceleration;
			if (velocity > maxVelocity)	velocity -= Math.min(Constants.MAX_BRAKING, velocity - maxVelocity);
			if (velocity < -maxVelocity) velocity += Math.min(Constants.MAX_BRAKING, -velocity - maxVelocity);
			double dx = velocity * Math.sin(heading); double dy = velocity * Math.cos(heading);
			x += dx; y += dy;
			if (slowingDown && velocity == 0.0) { distanceRemaining = 0.0; moveDirection = 0; slowingDown = false; acceleration = 0.0; }
			distanceRemaining -= velocity;
			if (x<18 || y<18 || x>battleFieldWidth-18 || y>battleFieldHeight-18) {
				distanceRemaining = 0;
				angleToTurn = 0;
				velocity = 0;
				moveDirection = 0;
				x = Math.max(18,Math.min(battleFieldWidth-18,x));
				y = Math.max(18,Math.min(battleFieldHeight-18,y));
			}
		}
		
		MotionState next = new MotionState();
		next.x = x;
		next.y = y;
		next.velocity = velocity;
		next.heading = heading;
		return next;
	}

	public void bulletHit(Bullet b, RobotState me, long time, boolean hitMe) {
		Wave wave = null, closestWave = null;
		double soonestImpact = Double.POSITIVE_INFINITY;
		Iterator i = enemyRealWaves.iterator();
		Point impactPoint = new Point(b.getX(), b.getY());
		while(i.hasNext()){
			wave = (Wave) i.next();
			double timeToImpact = wave.timeToImpact(impactPoint, time);
			if(Math.abs(timeToImpact) < Math.abs(soonestImpact)){
				closestWave = wave;
				soonestImpact = timeToImpact;
			}
		}
		
		
		if(closestWave != null && Math.abs(soonestImpact) < 2.0){
			enemyRealGuns.doWave(closestWave, impactPoint, 1);
			if(hitMe){
				if(isHOT){
					double solution = closestWave.getSolutionAngle(me);
					if(Math.abs(solution) > .175){ //roughly +/- 10 degrees
						isHOT = false;
					}
				}
				damageInflictedByKnownWaves += Utils.bulletDamage(b.getPower());
				PhoenixOS.info("------------------- HIT ----------------------------");
				PhoenixOS.info("At " + me);
				PhoenixOS.info("bullet hit " + impactPoint);
				PhoenixOS.info("Enemy fired at me from " + closestWave.shooterAtFireTime);
				PhoenixOS.info("Power " + closestWave.power);
				PhoenixOS.info("Predicted impact time was " + soonestImpact);
				PhoenixOS.info("Solution GF was " + (closestWave.getSolutionGF(me, GF0)- GF0));
				PhoenixOS.info("Closest wave distance traveled: " + (closestWave.speed * (PhoenixOS.bot.getTime() - closestWave.timeFired)));
				PhoenixOS.info("Time fired was: " + closestWave.timeFired);
				PhoenixOS.info("Wave speed was: " + closestWave.speed);
				PhoenixOS.info("Angle To Bullet (+/- 1 = perp.)" + (Utils.angularDifferenceBetween(closestWave.shooterAtFireTime.absoluteAngleTo(me), PhoenixOS.bot.getHeading()) * 2.0 / Math.PI));
			}
			enemyRealWaves.remove(closestWave);
		} else {
			if(hitMe){
				PhoenixOS.warning("Hit by unknown wave of power " + b.getPower());
				damageInflictedByUnknownWaves += Utils.bulletDamage(b.getPower());
			} else {
				PhoenixOS.warning("Shot down unknown wave of power " + b.getPower());
			}
			Iterator iterator = enemyRealWaves.iterator();
			PhoenixOS.info("Known waves:");
			while(iterator.hasNext()){
				Wave w = (Wave) iterator.next();
				PhoenixOS.info(" time to impact: " + w.timeToImpact(me, time));
				PhoenixOS.info(" power: " + w.power);
			}
		}
	}
	
	public void addRealWave(Wave wave) {
		enemyRealWaves.add(wave);
	}
}
