package cs.s2.stat;

import cs.s2.misc.VDA;

/**
 * This is a visit count stat buffer. However, it is such a buffer where it may
 * have any number of segmentation or dimensionality.
 * 
 * For those uninformed, this is part of a statistical data mining system.
 * 
 * @author Chase
 */
public class StatBuffer {
	protected static final int scanWidth = 1;

	/** Handing the VDA manually is dangerous, and could wipe out all your data */
	private VDA<double[]> _binSet;
	public int _binTotal;
	public int[] _dataIndex;
	/** This is fine, since java can handle jagged arrays */
	public double[][] _splits;

	public double halfLife = -1;
	public boolean smooth = false;

	/**
	 * Constructs a new buffer
	 * 
	 * @param splits
	 *            The data defining the data splits, usually a jagged array
	 * @param dataIndex
	 *            the index of the raw stats each split is on
	 * @param binTotal
	 *            total number of indexes for this buffer
	 */
	public StatBuffer(double[][] splits, int[] dataIndex, int binTotal) {
		/* TODO: splits and dataIndex lengths don't equal throw exception */
		_binTotal = binTotal;
		_dataIndex = dataIndex;
		_splits = splits;

		int[] dimensions = new int[Math.max(1, splits.length)];
		dimensions[0] = 1;
		for (int i = 0; i < splits.length; ++i)
			dimensions[i] = splits[i].length + 1;

		_binSet = new VDA<double[]>(dimensions);
	}

	/**
	 * Initializes a given location in the stat buffer, used for memory
	 * conservation via lazy initialization.
	 */
	public void initialize(int[] pos) {
		double[] d = _binSet.get(pos);
		if (d == null) {
			_binSet.set(new double[_binTotal], pos);
		}
	}

	/**
	 * Constructs a new stat buffer, using the lazy SplitSet
	 * 
	 * @param ss
	 *            SplitSet to use
	 * @param binTotal
	 *            total number of indexes for this buffer
	 */
	public StatBuffer(SplitSet ss, int binTotal) {
		this(ss.getSplitSet(), ss.getIndexSet(), binTotal);
	}

	/**
	 * This creates a flat, single dimension stat buffer.
	 * 
	 * @param binTotal
	 *            total number of indexes for this buffer
	 */
	public StatBuffer(int binTotal) {
		this(new double[0][0], new int[0], binTotal);
	}

	/**
	 * This method splits the raw data as defined by the data index and split
	 * information. It outputs a data array which can be fed to the VDA to get
	 * the desired bin set.
	 * 
	 * @param data
	 *            raw data
	 * @return data indexes
	 */
	public int[] convertData(StatData data) {
		int[] indexes = new int[Math.max(1, _dataIndex.length)];
		if (_splits.length == 0)
			return indexes;
		for (int i = 0; i < indexes.length; ++i) {
			int index = 0;
			for (; index < _splits[i].length; ++index) {
				if (data._data[_dataIndex[i]] < _splits[i][index])
					break;
			}
			indexes[i] = index;
		}
		return indexes;
	}

	/**
	 * Gets the individual stat buffer for the give data.
	 */
	public double[] getStatBin(StatData data) {
		int[] indexes = convertData(data);
		initialize(indexes);
		return _binSet.get(indexes);
	}

	/**
	 * Sets the decay half-life of this buffer anything < 0 is infinite.
	 */
	public void setHalflife(double hlife) {
		if (hlife < 0) {
			halfLife = -1;
		} else {
			halfLife = hlife;
		}
	}

	/**
	 * Updates this stat buffer with the given raw data
	 */
	public void update(StatData data, double weight) {
		double[] bins = getStatBin(data);
		double index = GFToIndex(data._gf, _binTotal);

		if (halfLife > 0) {
			if (data.range) {
				double end = GFToIndex(data._max, _binTotal);
				if(smooth) {
					updateBinSmoothRollingRange(bins, index, end, weight, halfLife);
				} else {
					updateBinRollingRange(bins, index, end, weight, halfLife);
				}
			} else {
				if(smooth) {
					updateBinSmoothRolling(bins, index, weight, halfLife);
				} else {
					updateBinRolling(bins, index, weight, halfLife);
				}
			}
		} else {
			if (data.range) {
				double end = GFToIndex(data._max, _binTotal);
				if(smooth) {
					updateBinSmoothRange(bins, index, end, weight);
				} else {
					updateBinRange(bins, index, end, weight);
				}
			} else {
				if(smooth) {
					updateBinSmooth(bins, index, weight);
				} else {
					updateBin(bins, index, weight);
				}
			}
		}
	}

	/**
	 * Gets the best bin from the raw data
	 */
	public int getBestIndex(StatData data) {
		double[] bins = getStatBin(data);
		return getBestIndex(bins, scanWidth);
	}

	/**
	 * Gets the best guessfactor from the raw data
	 */
	public double getBestGF(StatData data) {
		return indexToGF(getBestIndex(data), _binTotal);
	}

	/**
	 * return the array that backs the VDA
	 */
	public Object[] getBackBuffer() {
		return _binSet.getBackingArray();
	}

	/**
	 * TODO: Replace this with some kind of externalized update method. That way
	 * it is easier to replace with rolling and different distributions
	 */
	public static final void updateBin(double[] bin, double index, double weight) {
		int iIndex = (int) Math.round(index);
		for (int i = iIndex - 1; i < bin.length && i <= iIndex + 1; ++i) {
			double amount = Math.abs(i - index);
			if (amount < 1e-6)
				bin[i] += weight;
			else if (amount > 0 && amount < 1)
				bin[i] += (1.0 - amount) * weight;
		}
	}
	
	/**
	 * Updates a bin but smooths it via a simple distribution function
	 */
	public static final void updateBinSmooth(double[] bin, double index, double weight) {
		for (int i = 0; i < bin.length; ++i) {
			double amount = Math.abs(i - index);
			bin[i] = weight*Math.exp(-0.5*amount*amount);
		}
	}

	/**
	 * Updates a bin set based on a gf range.
	 */
	public static final void updateBinRange(double[] bin, double start,
			double end, double weight) {
		for (int i = (int) (start); i < bin.length && i <= (int) (end + 1); ++i) {
			if (i >= start && i <= end) {
				bin[i] += weight;
			} else {
				bin[i] += Math.min(Math.abs(i - start), Math.abs(i - end))
						* weight;
			}
		}
	}
	
	/**
	 * Updates a bin range but smooths it via a simple distribution function
	 */
	public static final void updateBinSmoothRange(double[] bin, double start,
			double end, double weight) {
		for (int i = 0; i < bin.length; ++i) {
			if(i >= start && i <= end) {
				bin[i] += weight;
			} else {
				double amount = Math.min(Math.abs(i - start), Math.abs(i - end));
				bin[i] = Math.exp(-0.5*amount*amount)* weight;
			}
		}
	}

	/**
	 * Updates a bin set, with rolling averages.
	 */
	public static final void updateBinRolling(double[] bin, double index,
			double weight, double halflife) {
		double decay = computeDecayRate(halflife);
		for (int i = 0; i < bin.length; ++i)
			bin[i] *= decay;
		updateBin(bin, index, weight * (1 - decay));
	}
	
	/**
	 * Updates a bin set smoothly, with rolling averages.
	 */
	public static final void updateBinSmoothRolling(double[] bin, double index,
			double weight, double halflife) {
		double decay = computeDecayRate(halflife);
		for (int i = 0; i < bin.length; ++i)
			bin[i] *= decay;
		updateBinSmooth(bin, index, weight * (1 - decay));
	}

	/**
	 * Updates a bin set range, with rolling averages.
	 */
	public static final void updateBinRollingRange(double[] bin, double start,
			double end, double weight, double halflife) {
		double decay = computeDecayRate(halflife);
		for (int i = 0; i < bin.length; ++i)
			bin[i] *= decay;
		updateBinRange(bin, start, end, weight * (1 - decay));
	}
	
	/**
	 * Updates a bin set range smoothly, with rolling averages.
	 */
	public static final void updateBinSmoothRollingRange(double[] bin, double start,
			double end, double weight, double halflife) {
		double decay = computeDecayRate(halflife);
		for (int i = 0; i < bin.length; ++i)
			bin[i] *= decay;
		updateBinSmoothRange(bin, start, end, weight * (1 - decay));
	}

	/**
	 * Static method used to find the best index. Width is number of extra bins
	 * to each side to test.
	 */
	public static final int getBestIndex(double[] bin, int width) {
		double[] score = getBinScores(bin, width);
		int bestIndex = (bin.length - 1) / 2; /* TODO: This could be problematic */
		for (int i = 0; i < bin.length; ++i)
			if (score[i] > score[bestIndex])
				bestIndex = i;
		return bestIndex;
	}

	/**
	 * Gets the scores for each bin in the given stat bin.
	 */
	protected static final double[] getBinScores(double[] bin, int width) {
		double[] scores = new double[bin.length];
		for (int i = 0; i < bin.length; ++i) {
			double score = 0;
			// It is a little complicated in there, but things will
			// never go out of bounds
			for (int j = -width; j < width + 1; ++j) {
				int abs = Math.abs(j) + 1;
				if (i + j < 0 || i + j >= bin.length) {
					if (i == 0 || i == bin.length - 1) {
						score += bin[i - j] / abs;
					} else {
						int offset = (j < 0) ? -3 : 3;
						score += bin[i - j + offset] / abs;
					}
				} else {
					score += bin[i + j] / abs;
				}
			}
			scores[i] = score;
		}
		return scores;
	}

	/**
	 * Converts a bin index to a guessfactor based on total number of bins.
	 */
	public static final double indexToGF(int index, int bins) {
		index = Math.min(Math.max(0, index), bins - 1);
		double half = (bins - 1.0) / 2.0;
		return (index / half) - 1.0;
	}

	/**
	 * Converts a guessfactor to a bin index, based on number of bins. This
	 * returns a double, incase there is any need to know how much it over hangs
	 * each bin. Cast to an int for the raw information.
	 */
	public static final double GFToIndex(double gf, int bins) {
		gf = Math.min(Math.max(-1.0, gf), 1.0) + 1.0;
		double half = (bins - 1.0) / 2.0;
		return gf * half;
	}

	/**
	 * Computation of decay rate
	 */
	public static double computeDecayRate(double halfLife) {
		return Math.exp(Math.log(0.5) / halfLife);
	}
}