package ar.horizon.util;

import static ar.horizon.util.Util.*;

import java.util.*;

import ar.horizon.stats.Pair;

/**
 * @author Aaron Rotenberg
 */
public class GuessFactorArray {
	/**
	 * Every GF array used by the bot has the same number of bins. It's just
	 * easier that way.
	 */
	public static final int BINS = 47;
	public static final int MIDDLE_BIN = (BINS - 1) / 2;

	/**
	 * @return The GF array index (0 to 46) that corresponds to a given
	 *         floating-point guess factor (-1.0 to 1.0).
	 */
	public static int getIndexFromGuessFactor(double guessFactor) {
		return (int) limit(0, (guessFactor * MIDDLE_BIN) + MIDDLE_BIN, BINS - 1);
	}

	/**
	 * @return The floating-point guess factor (-1.0 to 1.0) that corresponds to
	 *         a given GF array index (0 to 46).
	 */
	public static double getGuessFactorFromIndex(int guessFactorIndex) {
		return ((double) (guessFactorIndex) / MIDDLE_BIN) - 1.0;
	}

	private double[] array = new double[BINS];

	public GuessFactorArray(
			Collection<Pair<FireRecording, Double>> guessFactors,
			double imaginaryBulletsWeight, boolean weightOnTimeSinceRecording,
			long currentTime) {
		// If the guess factors list is empty, log a hit at GF 0.0. This makes
		// the Musashi Trick automatic, and is generally more useful than an
		// all-zeros array.
		if (guessFactors.size() == 0) {
			logHitAtIndex(getIndexFromGuessFactor(0.0), 1.0);
		} else {
			List<Double> weights = new LinkedList<Double>();
			for (Pair<FireRecording, Double> guessFactor : guessFactors) {
				FireRecording fireRecording = guessFactor.getKey();

				double realBulletWeight =
						(fireRecording.isRealBullet() ? 1.0
								: imaginaryBulletsWeight);

				long timeSinceRecording =
						currentTime
								- fireRecording.getTargetRecording().getTotalTime();
				double timeSinceRecordingWeight =
						(weightOnTimeSinceRecording ? 1.0 / (1 + timeSinceRecording)
								: 1.0);

				double weight = realBulletWeight * timeSinceRecordingWeight;
				logHitAtIndex(getIndexFromGuessFactor(guessFactor.getValue()),
						weight);
				weights.add(weight);
			}

			// Normalize the weights so that individual weighting of recordings
			// doesn't affect the total weighting in the danger calculations.
			double averageWeight = 0.0;
			for (double weight : weights) {
				averageWeight += weight / weights.size();
			}

			for (int i = 0; i < BINS; i++) {
				array[i] /= averageWeight;
			}
		}
	}

	private void logHitAtIndex(int guessFactorIndex, double weight) {
		for (int i = 0; i < BINS; i++) {
			int distance = guessFactorIndex - i;
			array[i] += weight / (1 + distance * distance);
		}
	}

	public double getAtIndex(int guessFactorIndex) {
		return array[guessFactorIndex];
	}

	// This version of the function has a "fudge factor"--it sums three adjacent
	// bins for each angle. It seems to be better overall than the simple
	// version.
	public int getBestAngleIndex() {
		int bestAngleIndex = -1;
		double bestAngle = Double.NEGATIVE_INFINITY;

		for (int i = 0; i < BINS; i++) {
			double angleValue;
			if (i == 0) {
				angleValue = (getAtIndex(i) + getAtIndex(i + 1)) / 2.0;
			} else if (i == BINS - 1) {
				angleValue = (getAtIndex(i - 1) + getAtIndex(i)) / 2.0;
			} else {
				angleValue =
						(getAtIndex(i - 1) + getAtIndex(i) + getAtIndex(i + 1)) / 3.0;
			}

			if (angleValue > bestAngle) {
				bestAngleIndex = i;
				bestAngle = angleValue;
			}
		}

		return bestAngleIndex;
	}
}
