/*
 * Copyright (c) 2004 Marcin Fuszara
 */  
package fushi.PvP1;
import robocode.*;

/* this logic can be used instead of common logic in PvP1
 * as a death dance;
 */
public class MiddleFingerLogic
{
   // robot for which this manager is used
   AdvancedRobot robot;
   // robot's body
   Body body;
   // robot's gun
   Gun gun;
   
   // possible logic states enumeration
   interface State {
      int
      Start = 1,         //< nothing done, orderd to be done
      Preparing = 2,     //< going towards arena center; rotating gun and radar
      Rotating = 3,      //< arena center reached; positioning elements
      GunRaising = 4,    //< raising the gun
      Idle = 5;          //< sequence complete
   }
   
   // current state of the logic 
   int state;
   
   // radians left for the gun to rotate
   double gunRotationLeft;

   /* the logic needs references to robot and its body
    */
   public MiddleFingerLogic( AdvancedRobot robot, Body body )
   {
      this.robot = robot;
      this.body = body;
      // set the robot to initial state;
      initialize();
   }
   
   /* sets robot to initial state;
    * the following sequence of run method calls will play
    * the whole logic sequence;
    */
   public void initialize()
   {
      state = State.Start;
   }
      
   /* runs the logic;
    * call every run of main loop;
    */
   public void run()
   {
      switch( state ) {
         case State.Start:
            // move to the center of the battlefield
            body.goTo( new Vector2d( robot.getBattleFieldWidth() / 2,
                                     robot.getBattleFieldHeight() / 2 ) );
            // rotate scanner so that it is directed down the arena
            robot.setTurnRadarRightRadians( 
               Util.normalRelativeAngle( Math.PI - robot.getRadarHeadingRadians() ) );
            // rotate gun so tahat it is directed down the arena
            robot.setTurnGunRightRadians(
               Util.normalRelativeAngle( Math.PI - robot.getGunHeadingRadians() ) );
            state = State.Preparing;
            //!no break
         case State.Preparing:
            // move body to destination
            body.run();
            // first orders done, find if the robot reached the center of the field
            if ( body.destinationReached() ) {
               state = State.Rotating;
               // arena center reached; rotate the body; the body will stand horizontally
               double robotHeading = Util.normalRelativeAngle( robot.getHeadingRadians() );
               if ( robotHeading > 0 )
                  robot.setTurnRightRadians( Math.PI/2 - robotHeading );
               else
                  robot.setTurnRightRadians( -Math.PI/2 - robotHeading ); 
            }
            break;
         case State.Rotating:
            // check if the body, radar and gun are rotated enough
            if ( robot.getTurnRemaining() == 0 
              && robot.getRadarTurnRemaining() == 0
              && robot.getGunTurnRemaining() == 0 ) {
               state = State.GunRaising;
               gunRotationLeft = Math.PI;
            }
            break;
         case State.GunRaising:
            // raise the gun
            double gunRotation = Math.min( gunRotationLeft, Math.toRadians( 5 ) );
            robot.setTurnGunLeftRadians( gunRotation );
            gunRotationLeft -= gunRotation;
            if ( gunRotationLeft <= 0 )
               state = State.Idle;
            break;
         case State.Idle:
         default:
            // do nothing
            break;
      }
   }
   
}