package lancel;
import java.awt.*;
import java.io.*;
import lancel.components.*;
import robocode.*;
import robocode.util.Utils;

/**************************************************************************************************
 Lynx - A duelist, based on Dynamic Clustering (DC) and WaveSurfing
 Author: lancel
 Current Version: 1.09
 Date: 6/12/2010
 License: Open Source under the RoboWiki Public Code License RWPCL (http://robowiki.net/wiki/RWPCL)
 Robocode Version: 1.6.1.4 (because of RoboRumble)
 Techniques this bot is based on:
    Gun: a k-nearest neighbors search with a kd-tree using a simple kernel density estimation 
         (dynamic clustering)
    Movement: basic WaveSurfing using the closest wave as a surfing wave
 CREDITS: Much of the ideas and code is based on various sources. A few to mention:
    wiki.BasicGFSurfer by Voidius, Pez and Bayen: initial structure for Lynx and the implementation
        of WaveSurfing
    Komarious by Voidius: the segmentation for WaveSurfing
    ABC: inventor of WaveSurfing
    Rednaxela: the implementation of the kd-tree
    rozu: precise prediction used in the Lynx's movement system
    DrussGT by Skilgannon: normalizing and weighting "features" that describe states for the gun
        ("segmentation" in DC) 
    Raiko by Jamougha: choosing a bullet power
    PEZ: Simple Iterative Wall Smoothing
    Voidious and others: good tutorials for getting into robocoding
 --------------------------------------------------------------------------------------------------
 TODO:
    Some improvements may be done for the next versions:
        Update WaveSurfing to use the two nearest waves.
        Start moving immediately when a round starts and keep moving also when the opponent doesn't
         fire.
 *************************************************************************************************/

public class Lynx extends AdvancedRobot {
//-------------------------------------------------------------------------------------------------
 // State, Movement and Gun are the main components of this bot. While keeping objects static we 
 // can save data for the gun and for the movement system between rounds (but not between battles).
 // The 'state' object contains information about current state of this bot and the opponent's bot. 
       
    static State state = new State(); 
    static Movement mov = new Movement(state);
    static Gun gun = new Gun(state, true);

    public void run() {
    //---------------------------------------------------------------------------------------------
        initRound();
        while(true) {
            turnRadarRightRadians(Double.POSITIVE_INFINITY);
        }
    }

    public void onScannedRobot(ScannedRobotEvent oppState) {
    //---------------------------------------------------------------------------------------------
        // Update state in every tick. The radar lock 'updateRadarLock()' should ensure that no
        // ticks are missed once the opponent is in the radar and no other functions cause misses
        // of turns e.g. by doing too much calculations.
        state.updateState(this, oppState);
        updateRadarLock();
        gun.onScanned();

        double angleFromGunToFiringAngle = Utils.normalRelativeAngle(gun.firingAngle
            - getGunHeadingRadians());
        setTurnGunRightRadians(angleFromGunToFiringAngle);

        double angle = mov.onScannedRobot();

        if(mov.surfWave != null)
            mov.setTurnTo(this, angle);
        if(setFireBullet(gun.firingPower) != null)
            gun.bulletsFired++;
    }
    
    public void updateRadarLock() {
    //---------------------------------------------------------------------------------------------
    // The radar lock
    // See http://robowiki.net/wiki/One_on_One_Radar#Perfect_radar_locks
        double angleFromRadarToOpp = Utils.normalRelativeAngle(state.bearingToOpp
            - getRadarHeadingRadians());
        setTurnRadarRightRadians(angleFromRadarToOpp * 2.0);
    }

    public void onPaint(Graphics2D g) {
    //---------------------------------------------------------------------------------------------
        DebugPaint.movementPaint(g, state, mov);
        DebugPaint.gunPaint(g, state, gun);
    }

    public void onHitByBullet(HitByBulletEvent e) {
    //---------------------------------------------------------------------------------------------
        mov.onHitByBullet(e);
    }

    public void onBulletHitBullet(BulletHitBulletEvent e) {
    //---------------------------------------------------------------------------------------------
        mov.onBulletHitBullet(e);
    }

    public void onBulletHit(BulletHitEvent e) {
    //---------------------------------------------------------------------------------------------
        gun.bulletHits++;
    }

    void initRound() {
    //---------------------------------------------------------------------------------------------
        // Do some initializations before the first round begins.
        if (getRoundNum() == 0) {
            state.myName = getName();
            out.println("Starting " + state.myName);
            setColors     (Color.white, Color.orange, Color.orange);
            setBulletColor(Color.white);
            setScanColor  (Color.white);
        }

        // Do some initializations before a new round begins.
        setAdjustGunForRobotTurn  (true);
        setAdjustRadarForGunTurn  (true);
        setAdjustRadarForRobotTurn(true);
        mov.oppWaves.clear();
        mov.surfDirections.clear();
        mov.surfAbsBearings.clear();
    }
}

