package com.timothyveletta;

import java.awt.*;
import java.util.ArrayList;
import robocode.AdvancedRobot;
import robocode.ScannedRobotEvent;
import robocode.util.Utils;
import java.awt.geom.*;

public class FuzzyBot extends AdvancedRobot {
	String trackingName;
	ArrayList<ForceVector> forces = new ArrayList<ForceVector>();
	AIControl ai;
	
	public double myX = -1;
	public double myY = -1;
	public double enemyX = -1;
	public double enemyY = -1;
	public double enemyDist = -1;
	public double enemyEnergy = 100.0;
	public double battleFieldWidth;
	public double battleFieldHeight;
	
	private double attackStrength = 1.0;
	
	/**
	 * The robots main running function.
	 */
	public void run() {
		setColors();
		
		battleFieldWidth = getBattleFieldWidth();
		battleFieldHeight = getBattleFieldHeight();
		
		myX = getX();
		myY = getY();
		
		ai = new AIControl(this);
		
		ai.Init();
		
		setAdjustGunForRobotTurn(true);
		setAdjustRadarForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
		
		turnRadarRightRadians(Double.POSITIVE_INFINITY);
		
		while(true) {
			myX = getX();
			myY = getY();
			
			ai.Update();
			forceMove();
			
			scan();
		}
	}
	
	private void setColors() {
		setBodyColor(Color.blue);
		setGunColor(Color.red);
		setRadarColor(Color.gray);
		setScanColor(Color.blue);
		setBulletColor(Color.blue);
	}

	public void onScannedRobot(ScannedRobotEvent e) {		
		// follow enemy
		double radarTurn = getHeadingRadians() + e.getBearingRadians() - getRadarHeadingRadians();
		setTurnRadarRightRadians(Utils.normalRelativeAngle(radarTurn));
		
		enemyDist = e.getDistance();
		enemyEnergy = e.getEnergy();
		
		// linear targeting
		double bulletPower = Math.min(3.0, getEnergy());
		myX = getX();
		myY = getY();
		double absoluteBearing = getHeadingRadians() + e.getBearingRadians();
		enemyX = getX() + e.getDistance() * Math.sin(absoluteBearing);
		enemyY = getY() + e.getDistance() * Math.cos(absoluteBearing);
		double enemyHeading = e.getHeadingRadians();
		double enemyVelocity = e.getVelocity();
		
		double deltaTime = 0;
		double battleFieldHeight = getBattleFieldHeight();
		double battleFieldWidth = getBattleFieldWidth();
		double predictedX = enemyX, predictedY = enemyY;
		while((++deltaTime) * (20.0 - 3.0 * bulletPower) <
				Point2D.Double.distance(myX, myY, predictedX, predictedY)) {
			predictedX += Math.sin(enemyHeading) * enemyVelocity;
			predictedY += Math.cos(enemyHeading) * enemyVelocity;
			
			if(predictedX < 18.0 
					|| predictedY < 18.0
					|| predictedX > battleFieldWidth - 18.0
					|| predictedY > battleFieldHeight - 18.0) {
				predictedX = Math.min(Math.max(18.0, predictedX), battleFieldWidth - 18.0);
				predictedY = Math.min(Math.max(18.0, predictedY), battleFieldHeight - 18.0);
				break;
			}
		}
		
		double theta = Utils.normalAbsoluteAngle(Math.atan2(predictedX - getX(), predictedY - getY()));
		setTurnGunRightRadians(Utils.normalRelativeAngle(theta - getGunHeadingRadians()));
		fire(attackStrength);
		
		ai.Update();
		
		forceMove();
	}
	
	public void SetAttackStrength(double attack) {
		if(attack < 1.0) {
			attackStrength = 1.0;
		} else if(attack > 3.0) {
			attackStrength = 3.0;
		}
		
		attackStrength = attack;
	}
	
	private void forceMove() {	
		double force;
		double angle;
		ForceVector result = new ForceVector(0, 0);
		
		for(int i = 0; i < forces.size(); i++) {
			result.x += forces.get(i).x;
			result.y += forces.get(i).y;
		}
		
		forces.clear();
		
		force = Math.sqrt(result.x * result.x + result.y * result.y);
		angle = (int)(Math.atan2(result.y, result.x) * 180 / Math.PI) - 90.0;

		double difference = normaliseAngle(-angle) - normaliseAngle(getHeading());
		
		if(difference > 180) {
			difference -= 360;
		} else if(difference < -180) {
			difference += 360;
		}
		
		if((int)difference != 0) {
			setTurnRight(difference);
		}
		
		
		setAhead(10 * force);
	}
	
	private double normaliseAngle(double angle) {
		if(angle > 180) {
			angle = -(360 - angle);
		} else if(angle < -180) {
			angle = 360 + angle;
		}
		
		return angle;
	}
	
	public void applyForce(ForceVector force) {
		forces.add(force);
	}
}