package ar.horizon;

import static ar.horizon.util.Util.*;
import static java.lang.Math.*;

import java.awt.*;
import java.awt.geom.*;
import java.util.*;
import java.util.List;

import ar.horizon.components.*;
import ar.horizon.components.gun.*;
import ar.horizon.components.movement.*;
import ar.horizon.util.*;
import ar.horizon.util.graphics.ColoredShape;

import robocode.*;

/**
 * @author Aaron Rotenberg
 */
public class Horizon extends AdvancedRobot {
	public static enum MovementMode {
		NO_MOVEMENT, NORMAL,
	}

	public static enum TargetingMode {
		NO_TARGETING, NORMAL, FIRE_FULL_POWER, RAIKO_GUN,
	}

	public static final MovementMode MOVEMENT_MODE = MovementMode.NORMAL;
	public static final TargetingMode TARGETING_MODE = TargetingMode.NORMAL;

	private static final String ON_TICK_BEGIN_EVENT_NAME = "arbitrary name 1";
	private static final String ON_TICK_END_EVENT_NAME = "arbitrary name 2";

	static Rectangle2D.Double fieldRectangle;
	private static long totalTime = 0;
	private static long roundsWon = 0;

	private List<Component> components = new ArrayList<Component>();
	/**
	 * This field is used for graphical debugging. Components add the shapes
	 * they want to draw to the list every frame (using
	 * {@link #debugFillShape(ColoredShape)}), and the robot draws them and
	 * clears the list.
	 */
	private List<ColoredShape> fillShapes = new LinkedList<ColoredShape>();
	/**
	 * Counts the number of turns the robot skips each round so that it can be
	 * outputted at the end of the round.
	 */
	private int skippedTurns = 0;
	/**
	 * This field is necessary because {@link #endRound()} is called both when
	 * the robot wins and when it dies&mdash;but it is possible for the robot to
	 * win, and a bullet that is still traveling to kill it!
	 */
	private boolean roundEnded = false;
	private RobotRecording myRecording;
	RobotLog myLog;
	private RobotRecording enemyRecording;
	private RobotLog enemyLog;

	@Override
	public void run() {
		// At the start of every battle, print the MANDATORY quote. =)
		if (getRoundNum() == 0) {
			out.println("\"Do not be in a hurry to succeed. ");
			out.println("What would you have to live for ");
			out.println("afterwards? Better make the horizon ");
			out.println("your goal; it will always be ahead of you.\"");
			out.println("-- William Makepeace Thackeray");
			out.println();
		}

		setColors(new Color(235, 195, 0), Color.ORANGE, Color.YELLOW,
				Color.YELLOW, Color.ORANGE);

		// It can be assumed that every high-level radar and gun wants these.
		setAdjustGunForRobotTurn(true);
		setAdjustRadarForGunTurn(true);

		// Add the every-tick custom events (see onCustomEvent() below).
		addCustomEvent(new Condition(ON_TICK_BEGIN_EVENT_NAME, 99) {
			@Override
			public boolean test() {
				return true;
			}
		});
		addCustomEvent(new Condition(ON_TICK_END_EVENT_NAME, 1) {
			@Override
			public boolean test() {
				return true;
			}
		});
		// We want onPaint() to be called as late as possible during the tick,
		// so that the components queue all their shapes first.
		setEventPriority("PaintEvent", 0);

		// First-round-only initialization.
		if (getRoundNum() == 0) {
			// NOTE: We have to obtain the fieldRectangle here instead of in the
			// static initialization code because getBattleFieldWidth() and
			// getBattleFieldHeight() are instance methods, for whatever reason.
			fieldRectangle =
					new Rectangle2D.Double(ROBOT_CENTER, ROBOT_CENTER,
							getBattleFieldWidth() - ROBOT_CENTER,
							getBattleFieldHeight() - ROBOT_CENTER);
			Component.fieldRectangle = fieldRectangle;
		}

		// Add all the robot components.
		components.add(new RadarLock());

		switch (MOVEMENT_MODE) {
			case NO_MOVEMENT:
				break;
			case NORMAL:
				components.add(new SolarWindMovement());
				break;
			default:
				throw new AssertionError("Unexpected movement mode.");
		}

		switch (TARGETING_MODE) {
			case NO_TARGETING:
				break;
			case NORMAL:
				components.add(new SolarFlareGun(false));
				break;
			case FIRE_FULL_POWER:
				components.add(new SolarFlareGun(true));
				break;
			case RAIKO_GUN:
				components.add(new RaikoGun());
				break;
			default:
				throw new AssertionError("Unexpected targeting mode.");
		}

		// Initialize each component.
		myLog = new RobotLog(fieldRectangle);
		enemyLog = new RobotLog(fieldRectangle);
		for (Component component : components) {
			component.initialize(this, myLog);
			component.run();
		}
	}

	/**
	 * Custom events are used to receive events at the beginning and end of
	 * every tick.
	 */
	@Override
	public void onCustomEvent(CustomEvent e) {
		String name = e.getCondition().getName();

		if (name.equals(ON_TICK_BEGIN_EVENT_NAME)) {
			fillShapes.clear();

			myRecording =
					new RobotRecording(e.getTime(), totalTime,
							new Point2D.Double(getX(), getY()),
							getHeadingRadians(), getVelocity(), getEnergy());
			totalTime++;

			for (Component component : components) {
				component.myRecording = myRecording;
				component.onTickBegin();
			}
		} else if (name.equals(ON_TICK_END_EVENT_NAME)) {
			for (Component component : components) {
				component.onTickEnd();
			}
		}
	}

	@Override
	public void onScannedRobot(ScannedRobotEvent e) {
		double absoluteBearing = getHeadingRadians() + e.getBearingRadians();

		enemyRecording =
				new RobotRecording(e.getTime(), totalTime, project(
						myRecording.getLocation(), absoluteBearing,
						e.getDistance()), e.getHeadingRadians(),
						e.getVelocity(), e.getEnergy());

		myRecording.setEnemyRecording(enemyRecording);
		enemyRecording.setEnemyRecording(myRecording);
		myLog.addEntry(myRecording);
		enemyLog.addEntry(enemyRecording);

		for (Component component : components) {
			component.enemyRecording = enemyRecording;
			component.onScannedRobot(e);
		}
	}

	@Override
	public void onBulletHit(BulletHitEvent e) {
		for (Component component : components) {
			component.onBulletHit(e);
		}
	}

	@Override
	public void onHitByBullet(HitByBulletEvent e) {
		for (Component component : components) {
			component.onHitByBullet(e);
		}
	}

	@Override
	public void onBulletHitBullet(BulletHitBulletEvent e) {
		for (Component component : components) {
			component.onBulletHitBullet(e);
		}
	}

	@Override
	public void onSkippedTurn(SkippedTurnEvent e) {
		debugPrintLine("Skipped turn.");
		skippedTurns++;
	}

	@Override
	public void onPaint(Graphics2D g) {
		for (ColoredShape shape : fillShapes) {
			g.setColor(shape.getColor());
			g.fill(shape.getShape());
		}
	}

	@Override
	public void onWin(WinEvent e) {
		roundsWon++;
		endRound();
	}

	@Override
	public void onDeath(DeathEvent e) {
		endRound();
	}

	private void endRound() {
		if (!roundEnded) {
			roundEnded = true;
			for (Component component : components) {
				component.roundEnded = true;
				component.onRoundEnd();
			}

			debugPrintLine("Total skipped turns: " + skippedTurns);
			debugPrintLine("Rounds won: " + roundsWon + " / "
					+ (getRoundNum() + 1));
		}
	}

	/**
	 * Orders the robot to move at the specified angle in the most efficient
	 * possible manner.
	 * 
	 * @see #aim(double)
	 * @see #aimHeadOn()
	 */
	public void setBackAsFront(double goAngle) {
		double angle = angleDifference(goAngle, getHeadingRadians());
		if (abs(angle) > (PI / 2)) {
			if (angle < 0) {
				setTurnRightRadians(PI + angle);
			} else {
				setTurnLeftRadians(PI - angle);
			}
			setBack(100);
		} else {
			if (angle < 0) {
				setTurnLeftRadians(-angle);
			} else {
				setTurnRightRadians(angle);
			}
			setAhead(100);
		}
	}

	/**
	 * Orders the robot to aim <code>bearingOffset</code> radians from directly
	 * at the opponent.
	 * 
	 * @see #aimHeadOn()
	 * @see #setBackAsFront(double)
	 */
	public void aim(double bearingOffset) {
		setTurnGunRightRadians(angleDifference(
				myRecording.getEnemyAbsoluteBearing() + bearingOffset,
				getGunHeadingRadians()));
	}

	/**
	 * Orders the robot to aim directly at the opponent.
	 * 
	 * @see #aim(double)
	 * @see #setBackAsFront(double)
	 */
	public void aimHeadOn() {
		aim(0.0);
	}

	/**
	 * This method is used to provide a systematic method of outputting textual
	 * debugging information. Currently, it just sticks the time in front of the
	 * message to be outputted, to make tracking events easier.
	 * 
	 * @see #debugFillShape(ColoredShape)
	 */
	public void debugPrintLine(String line) {
		out.format("%1$4d:  ", getTime());
		out.println(line);
	}

	/**
	 * Queues a shape to be drawn (filled) on the battlefield this tick.
	 * 
	 * @see #debugPrintLine(String)
	 */
	public void debugFillShape(ColoredShape shape) {
		fillShapes.add(shape);
	}
}
