package dam.util;

import java.util.*;
import java.io.*;

import dam.util.*;

public class PatternBuffer implements Serializable
{
	private Vector buf = new Vector();
	private static int MAX_LENGTH = 500;
	private double minCount = 100;
	public int privateTime = 0;
	private transient long lastTime = 0;
	private static int PERIOD_LENGTH = 7;
	private static int INC = 1;
	
	public PatternBuffer(){}
	
	/**
	* This ensures that we get one reading per time slice, however
	* often or rarely the target is scanned. This helps with the
	* calculations of future movement, hopefully.
	*/
	public void addReading(double headingChange, double velocity, long scanTime){
		if(size() < 1){
			buf.addElement(new TargetInfo(headingChange, velocity, privateTime++));
			lastTime = scanTime;
			return;
		}
		TargetInfo last = getAt(size() - 1);
		
		long dt = scanTime - lastTime;
		double dv = (velocity - last.velocity) / dt;
		double v = last.velocity;
		//double hc = headingChange / dt;
		double hc = headingChange; // note the heading change has already been averaged!
		for(long i = 0; i < dt; i++)
		{
			v += dv;
			TargetInfo temp = new TargetInfo( hc, v, privateTime++);
			buf.addElement(temp);
			 
		}
		lastTime = scanTime;
		while(buf.size() > MAX_LENGTH) buf.removeElementAt(0);
	}
	
	
	
	public TargetInfo getAt(int which){
		return (TargetInfo)buf.elementAt(which);
	}
	
	
	/**
	* This tries to figure out a matching segment from the robot's past.
	* it has to be more than diff moves back, in order that it can extrapolate forward
	* Originally, it was 6, now changed it to 7.
	*/
	public int getPeriodRelative(int diff)
	{
		long period = -1;
		if(size() < 3) return -1;
		int endPoint = (size() - 1) - (diff * 2);
		if(endPoint == -1) return -1;
		if(endPoint >= size()) return -1;
		int start = getMatchingPeriod(PERIOD_LENGTH, endPoint);
		if(start >= buf.size()) return -1;
		return start;
	}
	
	public GridPoint getPredictedPosition( int startPoint, int diff, double currentHeading, double x, double y, double velocity)
	{
		GridPoint g;
		int end = startPoint + diff;
		if(end == -1) return null;
		if(end < startPoint + 1) return null;
		if(end > size()) return null;
		g = new GridPoint(x,y,0);
		long dd;
		for(int i = startPoint; i < end; i++){
			TargetInfo ti2 = getAt(i + 1);
			TargetInfo ti = getAt(i);
			dd = ti2.time - ti.time;
			g.x += Math.sin(currentHeading) * velocity * dd;
			g.y += Math.cos(currentHeading) * velocity * dd;
			currentHeading = currentHeading + ti2.headingChange;
			velocity = ti.velocity;
		}
		return g;
	}
	
	/**
	* Gives the best guess at a previous period, or length periodLength
	* that is most similar to the most recent period of activity, in terms
	* of heading change and velocity. Returns the last point (that is, the point
	* that matches the most recent point.
	*/
	public int getMatchingPeriod(int periodLength, int end)
	{
		minCount = 100000;
		int retval = -1;
		int lstart = size() - periodLength;
		for(int i = 0; i < end - periodLength; i+=INC)
		{
			double myCount = 0;
			for(int j = 0; j < periodLength; j+=INC)
			{
				TargetInfo matcher = getAt(lstart + j);
				TargetInfo t = getAt(j + i);
				myCount += t.getRelativeSimilarity(matcher);
			}
			if(myCount < minCount){
				minCount = myCount;
				retval = i + periodLength;
			}
		}
		return retval;
	}
	
	/**
	* Anything less that 1 is good, anything more is bad
	*/
	public double getMinCount(){ return minCount; }
	
	public int size(){ return buf.size(); }
}



