package sgp;
import robocode.*;
import java.util.*;
import java.awt.Color;

/**
 * MeleeStrategy - a class by Simon Parker
 This class does the MadHatter 4.13 robot strategy
 */
public class MeleeStrategy extends Strategy
{
	private JollyNinja robot = null;
	private static GameState gameState = null;
	private final double MAX_VELOCITY = 8;
	private final double MAX_ANGULAR_VELOCITY_DEG_PER_FRAME = 10;

	private Target currentTarget = null;
	private Coordinate currentPosition = new Coordinate();

	private int hitCount = 0;
	private int missCount = 0;

	private double previousHeading = 0;
	
	private double HEADING_DIFFERENCE_WIEGHT = -10;
	private double CENTER_DISTANCE_WIEGHT = -0.1;
	
	private int HEADING_HISTORY_SIZE = 200;
	private int HEADING_OFFSET = 10;
	private DoubleCircularBuffer headingHistory = new DoubleCircularBuffer(HEADING_HISTORY_SIZE, 0);
	private Coordinate desiredPoint;
	
	private double CORNER_OFFSET_X = 100;
	private double CORNER_OFFSET_Y = 100;
	private Coordinate[] potentialDesiredPointArray = new Coordinate[4];
	private double DESIRED_POINT_WEIGHT = 300;
	private double DESIRED_POINT_BOUNDARY = 50;

	public MeleeStrategy(StrategyManager theStrategyManager)
	{
		super(theStrategyManager);
		robot = JollyNinja.getInstance();

		gameState = new GameState();
	}


	public void reset()
	{
		super.reset();
		gameState.reset();
		
		HEADING_DIFFERENCE_WIEGHT = robot.parameters.headingDifferenceWieght.getValue();
		HEADING_OFFSET = (int)robot.parameters.headingHistoryOffset.getValue();
		CENTER_DISTANCE_WIEGHT = robot.parameters.centreDistanceWieght.getValue();
		DESIRED_POINT_WEIGHT = robot.parameters.desiredPointWeight.getValue();
		CORNER_OFFSET_X = robot.parameters.cornerOffset.getValue();
		CORNER_OFFSET_Y = CORNER_OFFSET_X;
		DESIRED_POINT_BOUNDARY = robot.parameters.desiredPointBoundary.getValue();
		
		potentialDesiredPointArray[0] = new Coordinate(CORNER_OFFSET_X, CORNER_OFFSET_Y);
		potentialDesiredPointArray[1] = new Coordinate(Environment.battleFieldWidth - CORNER_OFFSET_X, CORNER_OFFSET_Y);
		potentialDesiredPointArray[2] = new Coordinate(CORNER_OFFSET_X, Environment.battleFieldHeight - CORNER_OFFSET_Y);
		potentialDesiredPointArray[3] = new Coordinate(Environment.battleFieldWidth - CORNER_OFFSET_X, Environment.battleFieldHeight - CORNER_OFFSET_Y);

	}

	//any init that needs doing at the start of the turn
	public void startTurn()
	{
		robot.angularVelocity = normalRelativeAngle(Environment.getHeading() - previousHeading);
		currentPosition.set(Environment.getX(), Environment.getY());
		robot.jiggleAnalyser.update(currentPosition, Environment.getVelocity(), Environment.getHeading());
		robot.patternAnalyser.update(currentPosition, Environment.getVelocity(), Environment.getHeading(), (int)Environment.getTime());
		robot.antiDodgeAnalyser.update(currentPosition, Environment.getVelocity(), Environment.getHeading(), (int)Environment.getTime());
		
		String oldTargetName = null;
		if (currentTarget != null)
		{
			oldTargetName = currentTarget.name;
		}

		currentTarget = gameState.getBestTarget(currentPosition);

		if (currentTarget != null)
		{
	//		robot.out.println("d = " + (int)currentTarget.position.distanceFrom(currentPosition) + ", d2 = " + (int)currentTarget.closestRobotDistance + ", " + currentTarget.name);
		}


		gameState.updateImpacts
		(
			currentPosition,
			Environment.getVelocity(),
			Environment.getHeading(),
			robot.angularVelocity,
			Environment.getTime()
		);
		
		if (desiredPoint == null) desiredPoint = new Coordinate(Environment.battleFieldWidth / 2, Environment.battleFieldHeight / 2);
		desiredPoint = getDesiredPoint(currentPosition, desiredPoint);

		if (robot.getOthers() < 2)
		{
			strategyManager.setStrategyId(strategyManager.EVASION_STRATEGY_ID);
		}
		
//		if (robot.getOthers() < 4)
//		{
//			strategyManager.setStrategyId(strategyManager.MAD_HATTER_STRATEGY_ID);
//		}
	}

	//control the radar motion
	public void setScan()
	{
		// optimal scanning calls for scanning in the direction of the target that was
		// scanned the longest ago

		Target scanTarget = gameState.getMostOutDatedTarget();

		if ((scanTarget == null) || (Environment.getTime() < 10))
		{
			robot.setTurnRadarRight(720);
			return;
		}

		double dT = Environment.getTime() - scanTarget.lastTimeScanned;
		if (dT > 10)
		{
			robot.setTurnRadarRight(720);
			return;
		}


		double targetRadarAngle = currentPosition.headingTo(scanTarget.position);
		double radarTurnRightAngle = normalRelativeAngle(targetRadarAngle - Environment.getRadarHeading());

		if (radarTurnRightAngle > 0)
		{
			radarTurnRightAngle += 20;
		}
		else
		{
			radarTurnRightAngle -= 20;
		}

		robot.setTurnRadarRight(radarTurnRightAngle);
	}

	/**
	 * control the gun motion and fires if neccessary
	 */
	public void setGunRotation()
	{
		if (currentTarget == null) return;
		if (currentTarget.position == null) return;		

		Coordinate estimatedTargetPosition = currentTarget.getEstimatedPosition(Environment.getTime());
		double targetDistance = currentPosition.distanceFrom(estimatedTargetPosition);

		double bulletPower = calculateBulletPower(currentPosition, estimatedTargetPosition);

		Intercept intercept = currentTarget.gunManager.getIntercept(currentPosition, bulletPower);
		if (intercept == null) return;

		double turnAngle = normalRelativeAngle(intercept.bulletHeading_deg - Environment.getGunHeading());

		// move gun to target angle
		robot.setTurnGunRight(turnAngle);

		if (Math.abs(turnAngle) <= intercept.angleThreshold)
		{
			if 	(
					(intercept.impactPoint.x > 0) &&
					(intercept.impactPoint.x < Environment.battleFieldWidth) &&
					(intercept.impactPoint.y > 0) &&
					(intercept.impactPoint.y < Environment.battleFieldHeight)
				)
			{
				if (
					(Environment.getGunHeat() == 0) && 
					(Environment.getEnergy() > bulletPower) && 
					(currentTarget.getActiveBulletCount() < 4) 
					)
				{
					currentTarget.fire(bulletPower);
				}
			}
		}

	}

	public void endTurn()
	{
		previousHeading = Environment.getHeading();
		headingHistory.add(previousHeading);
	}

	public void onScannedRobot(ScannedRobotEvent e)
	{
//		double angle_rad = Environment.getHeadingRadians() + e.getBearingRadians();
//
//		double currentTargetPosX = Environment.getX() + e.getDistance() * Math.sin(angle_rad);
//		double currentTargetPosY = Environment.getY() + e.getDistance() * Math.cos(angle_rad);
//
//		Coordinate currentTargetPosition = new Coordinate(currentTargetPosX, currentTargetPosY);
//
//		gameState.updateTarget
//		(
//			e.getName(),
//			currentTargetPosition,
//			e.getVelocity(),
//			e.getHeading(),
//			e.getEnergy(),
//			Environment.getTime()
//		);
		
		gameState.updateTarget(e);
	}

	public void onHitByBullet(HitByBulletEvent e)
	{
		gameState.onHitByBullet(e);
		
	}

	public void onBulletHit(BulletHitEvent event)
	{
		double bulletPower = event.getBullet().getPower();
		double damageDone = bulletPower * 4;
		if (bulletPower > 1.0)
		{
			damageDone += 2.0 * (bulletPower - 1.0);
		}

		Target target = gameState.getTarget(event.getName());
		target.damageSustained += damageDone;
		target.energy = event.getEnergy();
		hitCount++;

		gameState.onBulletHit(event);
	}

	public void onBulletMissed(BulletMissedEvent event)
	{
		missCount++;
	}

	public void onHitWall(HitWallEvent e)
	{
		//robot.out.println("Ouch!");
	}

	public void onHitRobot(HitRobotEvent e)
	{
		//robot.out.println("Get out of my way!");
	}

	public void onRobotDeath(RobotDeathEvent e)
	{
		gameState.setRobotDead(e.getName());
	}

	public void onDeath(DeathEvent event)
	{
		gameState.save();
		
		if ((hitCount + missCount) == 0) return;
		int hitRatePercent = (hitCount * 100) / (hitCount + missCount);
		robot.out.println("Hit Rate is " + hitRatePercent + "%");
	}

	public void onWin(WinEvent event)
	{
		gameState.save();

		int hitRatePercent = (hitCount * 100) / (hitCount + missCount);
		robot.out.println("Hit Rate is " + hitRatePercent + "%");
	}


	public double evaluateGoodness(Coordinate robotPosition, double robotHeading, double robotVelocity)
	{
		Coordinate centrePosition = new Coordinate(Environment.battleFieldWidth / 2, Environment.battleFieldHeight / 2);
		double minDimension = Math.min(Environment.battleFieldWidth, Environment.battleFieldHeight) / 2;
		double centreGoodness = CENTER_DISTANCE_WIEGHT * Math.abs(centrePosition.distanceFrom(robotPosition) - minDimension);
		
		double desiredDistance = desiredPoint.distanceFrom(robotPosition);
		desiredDistance = Math.max(DESIRED_POINT_BOUNDARY, desiredDistance);
		double desiredPointGoodness = -DESIRED_POINT_WEIGHT * desiredDistance;
		
		return evaluatePositionGoodness(robotPosition)
				+ evaluateHeadingGoodness(robotHeading)
				+ centreGoodness + desiredPointGoodness; 
	}
	
	private double evaluateHeadingGoodness(double heading)
	{
		double oldHeading = headingHistory.get(HEADING_OFFSET - 1);
		double headingDifference = Math.abs(Strategy.normalRelativeAngle(heading - oldHeading));
		return HEADING_DIFFERENCE_WIEGHT * Math.abs(headingDifference - 90.0);
	}

	private double evaluatePositionGoodness(Coordinate robotPosition)
	{
		return gameState.getGoodness(robotPosition, Environment.getTime());
	}

	private double calculateBulletPower
	(
		Coordinate robotPosition,
		Coordinate targetPosition
	)
	{
	//	double hitPercent = currentTarget.getBulletHitPercentage();
	//	double HIT_RATE_THRESHOLD = robot.parameters.meleeHitRateThreshold.getValue();

	//	double distance = robotPosition.distanceFrom(targetPosition);
	//	boolean isEasyToHit = ((hitPercent > HIT_RATE_THRESHOLD) || (distance < 100));
	//	double power = (isEasyToHit) ? 3.0 : 0.2501;
		double power = 3.0;
		power = Math.min(Strategy.getPowerFromDamage(currentTarget.energy) +0.001, power);
		power = Math.min(Environment.getEnergy() / 2, power);
		power = Math.max(0.1, power);
		return power;


//		double power = 2 + Math.random();
//	
//		final double MAX_BULLET_POWER = 3.0;
//		final double MIN_BULLET_POWER = 0.25;
//		power = Strategy.getPowerFromDamage((double)Math.round(Strategy.getDamageFromPower(power)) + 0.01);
//		power = Math.max(MIN_BULLET_POWER, power);
//		power = Math.min(MAX_BULLET_POWER, power);
//		
//		power = Math.min(currentTarget.energy / 4.0, power);
//		power = Math.max(0.1, power);
//
//		return power;
	}

	private Coordinate getDesiredPoint(Coordinate robotPosition, Coordinate currentDesiredPoint)
	{
		Coordinate bestPosition = currentDesiredPoint;
		double bestGoodness = evaluatePositionGoodness(currentDesiredPoint);
		
		for (int i = 0; i < 10; i++)
		{
			double dx = Math.random() * 100.0 - 50.0;
			double dy = Math.random() * 100.0 - 50.0;
			Coordinate offsetPoint = new Coordinate(dx, dy);
			
			Coordinate trialPoint = Strategy.limitToBattleField(bestPosition.plus(offsetPoint));
			double goodness = evaluatePositionGoodness(trialPoint);
			if (goodness > bestGoodness)
			{
				bestPosition = new Coordinate(trialPoint); //should copy it
				bestGoodness = goodness;
			}
		}
		
	//	System.out.println(bestPosition);
		
		return bestPosition;
		
		
/*		
		Coordinate bestPosition = potentialDesiredPointArray[0];
		
		double bestGoodness = evaluatePositionGoodness(potentialDesiredPointArray[0]);
		
		for (int i = 1; i < potentialDesiredPointArray.length; i++)
		{
			double goodness = evaluatePositionGoodness(potentialDesiredPointArray[i]);
			if (goodness > bestGoodness)
			{
				bestPosition = potentialDesiredPointArray[i];
				bestGoodness = goodness;
			}
		}
		
		Coordinate[] offsetPoint = new Coordinate[4];
		double OFFSET_DISTANCE = 100;
		offsetPoint[0] = new Coordinate(0, OFFSET_DISTANCE);
		offsetPoint[1] = new Coordinate(0, -OFFSET_DISTANCE);
		offsetPoint[2] = new Coordinate(OFFSET_DISTANCE, 0);
		offsetPoint[3] = new Coordinate(-OFFSET_DISTANCE, 0);
		
		for (int i = 0; i < offsetPoint.length; i++)
		{
			Coordinate trialPoint = Strategy.limitToBattleField(robotPosition.plus(offsetPoint[i]));
			double goodness = evaluatePositionGoodness(trialPoint);
			if (goodness > bestGoodness)
			{
				bestPosition = new Coordinate(trialPoint); //should copy it
				bestGoodness = goodness;
			}
		}
		
		System.out.println(bestPosition);
		
		return bestPosition;
*/

//		//gradient descent technique
//		//starting point is center of the field
//		final double DELTA = 20.0;
//		Coordinate positionPlusDx;
//		Coordinate positionPlusDy;
//		Coordinate gradient = new Coordinate();
//		Coordinate step, bestPosition;
//		double bestGoodness, ddX, ddY, newGoodness;
//		Coordinate searchEndPoint = new Coordinate();
//		
//		Coordinate newPosition = new Coordinate(Environment.battleFieldWidth / 2, Environment.battleFieldHeight / 2);
//		newGoodness = evaluatePositionGoodness(newPosition);
//		bestPosition = newPosition;
//		
//		int i;
//		for (i = 0; i < 10; i++)
//		{
//			bestPosition = newPosition;
//			bestGoodness = newGoodness;
//			positionPlusDx = bestPosition.plus(DELTA, 0.0);
//			positionPlusDy = bestPosition.plus(0.0, DELTA);
//			
//			//calculate gradient
//			ddX = (bestGoodness - evaluatePositionGoodness(positionPlusDx)) / DELTA;
//			ddY = (bestGoodness - evaluatePositionGoodness(positionPlusDy)) / DELTA;
//			
//			gradient.set(ddX, ddY);
//			
//			final int NUM_SEARCH_POINTS = 5;
//			
//			for (int j = 0; j < NUM_SEARCH_POINTS; j++)
//			{
//				searchEndPoint.x = (gradient.x > 0) ? 0.0 :Environment.battleFieldWidth;
//				searchEndPoint.y = (gradient.y > 0) ? 0.0 :Environment.battleFieldHeight;
//				
//				
//				step = gradient.multiply(-0.1);
//			newPosition = bestPosition.plus(step);
//			newGoodness = evaluatePositionGoodness(bestPosition);
//
//			//System.out.println("Step " + step + ", best " + bestGoodness + ", new " + newGoodness);
//			
//			if (newGoodness < bestGoodness) 
//			{
//				break;
//			}
//			
//			//if (step.length() < 20) break;
//		}
//		
//		System.out.println(i + " " + bestPosition);
//		return bestPosition;
//		
		
				
	}

}