
package sgp;
import robocode.*;
import java.io.*;

public class HitRateArray
{
	private double[] hitRateArray = new double[InterceptManager.NUMBER_OF_INTERCEPT_TYPES];
	private double defaultHitRate = 0.1;

	/**
	 * Constructor for HitRateArray.
	 */
	public HitRateArray(double hitRate)
	{
		super();

		defaultHitRate = hitRate;
		initialise();
	}
	
	private void initialise()
	{
		for (int i = 0; i < InterceptManager.NUMBER_OF_INTERCEPT_TYPES; i++)
		{
			hitRateArray[i] = defaultHitRate;
		}
	}
	
	public String getRateString(int i)
	{
		return Double.toString(Math.round(hitRateArray[i] * 1000) / 10.0);
	}

	public void printInfo()
	{
		if (Environment.isDebugging)
		{
			for (int i = 0; i < hitRateArray.length; i++)
			{
				if (i != 0) System.out.print(", ");
				System.out.print(Math.round(hitRateArray[i] * 1000) / 10.0 );
			}
			System.out.println("");

		}
	}
	
	public int getMinimumRateIndex()
	{
		double bestRate = Double.POSITIVE_INFINITY;
		int bestRateType = InterceptManager.JIGGLE_OSC_PLUS_PATTERN_WHEN_VALID_CIRCULAR;

		for (int i = 0; i < InterceptManager.NUMBER_OF_INTERCEPT_TYPES; i++)
		{
			double comparedRate = hitRateArray[i];
			
			if (comparedRate < bestRate)
			{
				bestRateType = i;
				bestRate = comparedRate;
			}
		}
		
		return bestRateType;
	}

	public int getMaximumRateIndex()
	{
		double bestRate = Double.NEGATIVE_INFINITY;
		int bestRateType = InterceptManager.JIGGLE_OSC_PLUS_PATTERN_WHEN_VALID_CIRCULAR;

		for (int i = 0; i < InterceptManager.NUMBER_OF_INTERCEPT_TYPES; i++)
		{
			double comparedRate = hitRateArray[i];
			
			if (comparedRate > bestRate)
			{
				bestRateType = i;
				bestRate = comparedRate;
			}
		}
		
		return bestRateType;
	}
	
	/*
	 * Handles the fired bullet event
	 */
	public void handleBulletFired(int dodgeType)
	{
		updateHitRate(dodgeType, 0.0);	
	}
	
	/*
	 * Handles a bullet hit
	 */
	public void handleBulletHit(int dodgeType)
	{
		updateHitRate(dodgeType, 2.0);
	}
	
	/*
	 * Average the incoming x's
	 */
	private void updateHitRate(int dodgeType, double x)
	{
		final int N = 60;
		double oldAverage = hitRateArray[dodgeType];
		
		hitRateArray[dodgeType] = ((N - 1) * oldAverage + x) / N;
		
	}

	public void save(String moduleName, String robotName)
	{
		String dataFileName = moduleName + "-Attack-" + robotName + ".txt";
		try
		{
			PrintStream w = new PrintStream(new RobocodeFileOutputStream(Environment.getRobot().getDataFile(dataFileName)));
			w.println(InterceptManager.NUMBER_OF_INTERCEPT_TYPES);
			
			for (int i = 0; i < InterceptManager.NUMBER_OF_INTERCEPT_TYPES; i++)
			{
				w.println(hitRateArray[i]);
			}
			
			// PrintStreams don't throw IOExceptions during prints,
			// they simply set a flag.... so check it here.
			if (w.checkError())
			{
				//out.println("I could not write the count!");
			}
			w.close();
		}
		catch (IOException e)
		{
			//out.println("IOException trying to write: " + e);
		}
	}
	
	public void load(String moduleName, String robotName)
	{
		int firingTypeCount;
		String dataFileName = moduleName + "-Attack-" + robotName + ".txt";
		try
		{
			BufferedReader r = new BufferedReader(new FileReader(Environment.getRobot().getDataFile(dataFileName)));

			//get the generation count
			firingTypeCount = Integer.parseInt(r.readLine());
			if (firingTypeCount != InterceptManager.NUMBER_OF_INTERCEPT_TYPES)
			{
				r.close();
//				initialise();
				return;
			}

			for (int i = 0; i < InterceptManager.NUMBER_OF_INTERCEPT_TYPES; i++)
			{
				hitRateArray[i] = Double.parseDouble(r.readLine());
			}
		}
		catch (IOException e)
		{
			// Something went wrong reading the file, Start from scratch
//			initialise();
		}
		catch (NumberFormatException e)
		{
			// Something went wrong converting to ints or doubles, Start from scratch
//			initialise();
		}


	}



}
