package lancel.components;
import java.awt.geom.*;
import ags.utils.*; // CREDIT: the implementation of the kd-tree by Rednaxela
import java.util.*;
import robocode.*;
import robocode.util.Utils;
import static java.lang.Math.*; // By using this import we can omit the 'Math' prefix from math
                                // function calls.

/**************************************************************************************************
 Lynx - A duelist, based on Dynamic Clustering (DC) and WaveSurfing
 Author: lancel
 Current Version: 1.09
 Date: 6/12/2010
 License: Open Source under the RoboWiki Public Code License RWPCL (http://robowiki.net/wiki/RWPCL)
 Robocode Version: 1.6.1.4 (because of RoboRumble)
 Techniques this bot is based on:
    Gun: a k-nearest neighbors search with a kd-tree using a simple kernel density estimation 
         (dynamic clustering)
    Movement: basic WaveSurfing using the closest wave as a surfing wave
 CREDITS: Much of the ideas and code is based on various sources. A few to mention:
    wiki.BasicGFSurfer by Voidius, Pez and Bayen: initial structure for Lynx and the implementation
        of WaveSurfing
    Komarious by Voidius: the segmentation for WaveSurfing
    ABC: inventor of WaveSurfing
    Rednaxela: the implementation of the kd-tree
    rozu: precise prediction used in the Lynx's movement system
    DrussGT by Skilgannon: normalizing and weighting "features" that describe states for the gun
        ("segmentation" in DC) 
    Raiko by Jamougha: choosing a bullet power
    PEZ: Simple Iterative Wall Smoothing
    Voidious and others: good tutorials for getting into robocoding
**************************************************************************************************/

public class Gun {
//-------------------------------------------------------------------------------------------------
    final int DIMENSIONS = 6;  // the number of different features in the kd-tree
    final int NEIGHBORS  = 25; // the number of nearest neighbors used for determining
                               // a firing angle
    State state;
    KdTree<GunWave> featureSpace;
    double[] featurePoint = new double[DIMENSIONS]; // weighted features of the current state
                                                    // (point in kd-tree)
    // Weights for featurePoints describing current state (dist, lat, adv, distLast10,
    // dirChange, decelChange). Searching nearest neighbors is based on the weighted similarity.
    //1.0.9.3: 10, 40, 10, 10, 10, 10
    double[] featureWeights  = {10, 40, 10, 10, 10, 10};
    double distLast10 = 0; // the opponent's distance from the current location to the location at 
                           // 10 ticks before
    int direction;
    int lastDirection;
    double oppLastVelocity = 0;
    int timeSinceDecel = 0;
    int timeSinceDirChange = 0;
    int targetBotWidth = 32;
    ArrayList<Point2D.Double> oppLocations = new ArrayList<Point2D.Double>(1000);
    java.util.List<GunWave> activeWaves = new ArrayList<GunWave>();
    java.util.List<KdTree.Entry<GunWave>> nNearest =  new ArrayList<KdTree.Entry<GunWave>>();
    
    // The following public variables are needed by main Lynx class
    public double firingAngle = 0;
    public double firingPower = 0;
    // for counting hitrate
    public int bulletsFired   = 0;
    public int bulletHits     = 0;

    public Gun (State state) {
    //---------------------------------------------------------------------------------------------
        featureSpace = new KdTree.Manhattan<GunWave>(DIMENSIONS, new Integer(100000));
        this.state = state; // Attach game environment's state to the gun.
    }

    public Gun (State state, boolean isManhattan) {
    //---------------------------------------------------------------------------------------------
        if (isManhattan == true) {
            System.out.println("Setting KD-Tree as manhattan.");
            featureSpace = new KdTree.Manhattan<GunWave>(DIMENSIONS, new Integer(100000));
        }
        else {
            System.out.println("Setting KD-Tree as euclidian.");
            featureSpace = new KdTree.SqrEuclid<GunWave>(DIMENSIONS, new Integer(100000));
        }
        this.state = state; // Attach game environment's state to the movement system.
    }

    public void onScanned() {
    //---------------------------------------------------------------------------------------------
        checkWavesHits();
        updateStatesForGun();
        if ((abs(oppLastVelocity) < abs(state.oppVelocity)))
            timeSinceDecel = 0;
        else
            timeSinceDecel++;
        featurePoint = normalizedFeatures();

        // Get n nearest neighbors
        if (featureSpace.size() > 0)
                nNearest = featureSpace.nearestNeighbor(
                    featurePoint, min(NEIGHBORS, featureSpace.size()), false);

        firingAngle = bestAngleDensity();

        // Set a new gun wave
        GunWave newWave = new GunWave(
            state.myLocation, state.oppLocation, state.bearingToOpp,
            Rules.getBulletSpeed(firingPower), direction, state.time, featurePoint);
        activeWaves.add(newWave);

        oppLastVelocity = state.oppVelocity;
    }

    void updateStatesForGun() {
    //---------------------------------------------------------------------------------------------
        oppLocations.add(0, state.oppLocation);
        
        //-----------------------------------------------------------------------------------------
        // CREDIT: choosing a firing power is based on Raiko's method by Jamougha
        //-----------------------------------------------------------------------------------------
        boolean oppIsClose = state.myLocation.distance(state.oppLocation) < 140.0;
        firingPower = oppIsClose ? 3 : 2;
        double theta = min(state.myEnergy/4, min(state.oppEnergy/4, firingPower));
        firingPower = theta;
        
        distLast10 = state.oppLocation.distance(oppLocations.get(min(10, oppLocations.size() - 1)));
        lastDirection = direction;  // Save previous direction.
        direction = oppMovementDirection();

        if(direction == lastDirection)
            timeSinceDirChange++;
        else
            timeSinceDirChange = 0;
    }

    int oppMovementDirection() {
    //---------------------------------------------------------------------------------------------
    // Returns 1, if the opponent moves in clockwise direction with respect to us.
    // Returns -1, if the opponent moves in counter-clockwise with respect to us.
    // Returns lastDirection, if the opponent doesn't move at this tick.

        if (state.oppVelocity != 0)
        {
            if (sin(state.oppHeading - state.bearingToOpp) * state.oppVelocity < 0)
                return -1;
            else
                return 1;
        }
        else return lastDirection;
    }

    double bestAngleDensity() {
    //---------------------------------------------------------------------------------------------
    // CREDIT: This function is based on the pseudocode in Dynamic Clustering Tutorial on the 
    // RoboWiki (http://robowiki.net/wiki/Dynamic_Clustering_Tutorial).
    //---------------------------------------------------------------------------------------------

        double bestAngle = state.bearingToOpp;

        // Start from the nearest data point
        if (nNearest.size() > 0)
            bestAngle = direction * nNearest.get(0).value.guessFactor *
                        asin(8 / Rules.getBulletSpeed(firingPower)) + state.bearingToOpp;

        int bestDensity = 0;
        double bestGF  = 0;
        double botWidthAngle = abs(targetBotWidth / state.oppDistance);

        // Find an angle with the best density
        for (KdTree.Entry<GunWave> dataPointA : nNearest) {
            int density = 0;
            double firingAngleA =
                direction * dataPointA.value.guessFactor *
                asin(8 / Rules.getBulletSpeed(firingPower)) + state.bearingToOpp;

            for (KdTree.Entry<GunWave> dataPointB : nNearest) {
                if (dataPointA != dataPointB) {
                    double firingAngleB = direction * dataPointB.value.guessFactor *
                        asin(8 / Rules.getBulletSpeed(firingPower)) + state.bearingToOpp;
                    double ux = (firingAngleA - firingAngleB) / botWidthAngle;
                    if (abs(ux) <= 1.0)
                        density++;
                }
                if (density > bestDensity) {
                    bestAngle = firingAngleA;
                    bestGF = dataPointA.value.guessFactor;
                    bestDensity = density;
                }
            }
        }
        if (bestGF == 0 && distLast10 < 10.0) {
            // If an opponent is predicted to be near the same location at firing time and hit time, 
            // add small random variation. This is useful e.g. against bots that try to block bullets by 
            // bullets (e.g. Geomancy.BS tries to block bullets).
            
            double halfBotAngle = atan(targetBotWidth/2.0 / state.oppDistance);
            Random rand = new Random();
            int randAddOrDec = rand.nextInt(2);
            
            bestAngle += randAddOrDec==1 ? halfBotAngle / 2 : -halfBotAngle / 2;
            //System.out.print(".");
        } 
                               
        return bestAngle;
    }

    double[] normalizedFeatures() {
    //---------------------------------------------------------------------------------------------
    // CREDIT: Normalizing and weighting data points is partially based on method in DrussGT by 
    // Skilgannon. http://robowiki.net/wiki/DrussGT)
    //---------------------------------------------------------------------------------------------
        
        double[] features = new double[DIMENSIONS];
        double[] w = featureWeights; // w for shorter representation of weights

        features[0] = state.oppDistance / 900.0 * w[0];

        double angleOffset = Utils.normalAbsoluteAngle(state.oppHeading - state.bearingToOpp);
        double oppLateralVelocity = state.oppVelocity * sin(angleOffset);
        features[1] = abs(oppLateralVelocity / 8) * w[1];

        double advancingVelocityOpp = -state.oppVelocity * cos(angleOffset);
        features[2] = MyUtils.limit(0, advancingVelocityOpp/16 + 0.5, 1) * w[2];
        features[3] = MyUtils.limit(0, distLast10/(8*10), 1) * w[3];

        double BFT = state.oppDistance / Rules.getBulletSpeed(firingPower);
        features[4] = 1/(1 + 2*timeSinceDirChange/BFT) * w[4];
        features[5] = 1/(1 + 2*timeSinceDecel/BFT)*w[5];

        return features;
    }

    void checkWavesHits() {
    //---------------------------------------------------------------------------------------------
        // Check if any wave has passed opponent's tank. If so, log the wave to kd-tree and remove
        // it from the active waves.
        for (int i = 0; i < activeWaves.size(); i++) {
            GunWave currentWave = activeWaves.get(i);
            if (currentWave.checkHit(state.oppLocation, state.time)) {
                double features[] = new double[DIMENSIONS];
                features = currentWave.features;
                featureSpace.addPoint(features, currentWave);
                activeWaves.remove(currentWave);
            }
        }
    }
    
    class GunWave {
    //---------------------------------------------------------------------------------------------
        double features[] = new double[DIMENSIONS];
        Point2D.Double myStartLocation  = new Point2D.Double();
        Point2D.Double oppStartLocation = new Point2D.Double();
        double startBearingToOpp; // A bearing from the wave origin to the opponent's tank at the
                                  // time the wave was created.
        double waveVelocity;
        long   startTime;
        int    direction;
        double guessFactor;

        GunWave(Point2D.Double myStartLocation, Point2D.Double oppStartLocation,
                double startBearingToOpp, double waveVelocity, int direction, long startTime,
                double[] features) {
        //-----------------------------------------------------------------------------------------
            this.direction         = direction;
            this.features          = features;
            this.myStartLocation   = myStartLocation;
            this.oppStartLocation  = oppStartLocation;
            this.startBearingToOpp = startBearingToOpp;
            this.startTime         = startTime;
            this.waveVelocity      = waveVelocity;
        }

        public boolean checkHit(Point2D.Double oppLocation, long currentTime) {
        //-----------------------------------------------------------------------------------------
        // Check if this wave has passed opponent's tank.
        if (oppLocation.distance(myStartLocation) <= (currentTime-startTime) * waveVelocity) {
            double desiredDirection   = atan2(oppLocation.x - myStartLocation.x,
                                        oppLocation.y - myStartLocation.y);
            double angleOffset = Utils.normalRelativeAngle(desiredDirection
                                        - startBearingToOpp);
            guessFactor = max(-1, min(1, angleOffset /
                          MyUtils.maxEscapeAngle(waveVelocity))) * direction;
            return true;
        }
        return false;
        }
    }
}
