/*
 * ActivationFunction - Utility class for neural nets of Robocode bots.
 * Copyright (C) 2002  Joachim Hofer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * You can contact the author via email (qohnil@johoop.de) or write to
 * Joachim Hofer, Feldstr. 12, D-91052 Erlangen, Germany.
 */

package qohnil.neural.util;

public class ActivationFunction {
    public static final int F_TABLE_LENGTH = 100;
    public static final double F_TABLE_MAX = 10.0;

    private static ActivationFunction instance = null;

    private double factor = (double)(F_TABLE_LENGTH - 1) / F_TABLE_MAX;
    private double[] f = new double[F_TABLE_LENGTH];
    private double[] f_d = new double[F_TABLE_LENGTH];

    private ActivationFunction() {
        init();
    }

    private void init() {
        for (int i = 0; i < F_TABLE_LENGTH; i++) {
            f[i] = 1.0 / (1.0 + Math.exp(-((double)i) / factor));
            if (i > 0) {
                f_d[i] = f[i] - f[i - 1];
            }
        }
    }

    public static ActivationFunction getInstance() {
        if (instance == null) {
            instance = new ActivationFunction();
        }
        return instance;
    }

    public double actFunc(double x) {
        if (x >= 0.0) {
            double xd = x * factor;
            int idx = (int)xd;
            if (idx >= F_TABLE_LENGTH - 1) {
                return f[F_TABLE_LENGTH - 1];
            }
            return f[idx] + f_d[idx] * (xd - idx);
        } else {
            double xd = -x * factor;
            int idx = (int)xd;
            if (idx >= F_TABLE_LENGTH - 1) {
                return 1.0 - f[F_TABLE_LENGTH - 1];
            }
            return 1.0 - (f[idx] + f_d[idx] * (xd - idx));
        }
    }

    public double actDeriv(double f) {
        return f * (1.0 - f);
    }
}
