package kawigi.sbf.utils;
import java.awt.geom.*;
import robocode.*;

/**
 * WaveBullet - melee-capable Wave for statistics collection.
 * Based on MiniBullet in FloodMini and MeleeBullet in Coriantumr.
 */
public class WaveBullet
{
	private Point2D startPoint, lastPoint;
	private double startgunheading, direction, bulletspeed, bulletd, escapeAngle;
	private long lasttime;
	private int[] segment, indexes;
	public int weight;
	
	public WaveBullet(VolatileEnemy e, long time, int[] segment, int[] indexes)
	{
		Point2D myLocation = Utils.getMyLocation();
		startPoint = new Point2D.Double(myLocation.getX(), myLocation.getY());
		lastPoint = new Point2D.Double(e.getX(), e.getY());
		startgunheading = Utils.angle(lastPoint, startPoint);
		direction = e.getLateralDirection();
		bulletspeed = Utils.getBulletVelocity(Utils.getPower(e));
		escapeAngle = Math.asin(8/bulletspeed);
		bulletd = -bulletspeed;
		lasttime = time;
		this.segment = segment;
		weight = 1;
		this.indexes = indexes;
	}
	
	//alternate constructor to aim at myself:
	public WaveBullet(TeamRobot target, VolatileEnemy from, int[] segment, long time, double latdirection, double power, int[] indexes)
	{
		startPoint = new Point2D.Double(from.getX(), from.getY());
		if (from.lastMyLocation == null)
			lastPoint = new Point2D.Double(target.getX(), target.getY());
		else
			lastPoint = from.lastMyLocation;
		startgunheading = Utils.angle(lastPoint, startPoint);
		direction = latdirection;
		bulletspeed = Utils.getBulletVelocity(power);
		escapeAngle = Math.asin(8/bulletspeed);
		bulletd = -bulletspeed;
		lasttime = time;
		this.segment = segment;
		//This unweights non-bullet-aligned waves (it's set to something else
		//if they actually fired).
		weight = 0;
		this.indexes = indexes;
	}
	
	//returns true if this wave hit and should be removed
	public boolean updateEnemy(VolatileEnemy enemy, long time)
	{
		//I just linearly interpolate where they've been if I don't get an update each tick.  Should be a
		//reasonable interpolation, at least relative to firing angle, and I figure it's a must for melee:
		double ang = Utils.angle(enemy, lastPoint);
		double dist = enemy.distance(lastPoint)/(time-lasttime);
		while (lasttime < time)
		{
			if (startPoint.distance(lastPoint) <= bulletd)  	//could have hit now.
			{
				//increments the appropriate bucket
				double gf = getGF();
				segment[Math.min(segment.length-1, Math.max(0, (int)Math.round((1+gf)*(segment.length-1)/2)))] += Math.max(1, weight);
				return true;
			}
			lasttime++;
			bulletd += bulletspeed;
			lastPoint = Utils.projectPoint(lastPoint, ang, dist);
		}
		return false;
	}
	
	public double getGF()
	{
		return robocode.util.Utils.normalRelativeAngle(Utils.angle(lastPoint, startPoint)-startgunheading)*direction/escapeAngle;
	}
	
	public boolean addHitGuessFactors(Rectangle2D bounds, long time, boolean[] hits)
	{
		Point2D location = new Point2D.Double(bounds.getCenterX(), bounds.getCenterY());
		if (weight > 0 && startPoint.distance(location)-26 < bulletd+(time-lasttime)*bulletspeed && startPoint.distance(location)+26 > bulletd+(time-lasttime-1)*bulletspeed)
		{
			Line2D bulletLine = new Line2D.Double();
			boolean ret = false;
			for (int i=0; i<segment.length; i++)
			{
				double gfAngle = startgunheading+direction*escapeAngle*((double)i/(segment.length/2)-1);
				bulletLine.setLine(Utils.projectPoint(startPoint, gfAngle, bulletd+(time-lasttime)*bulletspeed), Utils.projectPoint(startPoint, gfAngle, bulletd+(time-lasttime+1)*bulletspeed));
				if (bounds.intersectsLine(bulletLine))
				{
					hits[i] = true;
					ret = true;
				}
			}
			//... Possible problem: What if I'm between GF lines?
			return ret;
		}
		else
			return false;
	}
	
	public double hitRisk(boolean[] hits)
	{
		int minIndex = -1, maxIndex = -1;
		for (int i=0; i<hits.length; i++)
		{
			if (hits[i])
			{
				if (minIndex == -1)
					minIndex = i;
				maxIndex = i;
			}
		}
		if (minIndex == -1)
			return 0;
		int highest = 0;
		for (int i=0; i<segment.length; i++)
			highest = Math.max(segment[i], highest);
		double weight = 0;
		int total = 0;
		for (int i=0; i<segment.length; i++)
		{
			if (hits[i])
			{
				//if (i == segment.length/2)
				//	weight += (highest+1)*4.0;
				//else
					weight += ((i == segment.length/2) ? 1 : 0) + segment[i]*4.0;
			}
			else	//sort of repel other parts of the curve that are high?
			{
				int hitDist = Math.min(Math.abs(i-minIndex), Math.abs(i-maxIndex));
				weight += (double)segment[i]/(hitDist*hitDist+1);
			}
			total += segment[i];
		}
		return (double)weight/(total+1);
	}
	
	public double match(Bullet b)
	{
		return Math.abs(startPoint.distance(b.getX(), b.getY())-(bulletd/bulletspeed)*b.getVelocity());
	}
	
	public void addBulletStat(Bullet b)
	{
		Point2D bulletPoint = new Point2D.Double(b.getX(), b.getY());
		double gf = robocode.util.Utils.normalRelativeAngle(Utils.angle(bulletPoint, startPoint)-startgunheading)*direction/escapeAngle;
		int index = Math.min(segment.length-1, Math.max(0, (int)Math.round((1+gf)*(segment.length-1)/2)));
		int highest = 0;
		for (int i=0; i<segment.length; i++)
			highest = Math.max(highest, segment[i]);
		segment[index] = highest+10;
		System.out.println("Hit at gf " + gf);
	}
	
	public int[] getIndexes()
	{
		return indexes;
	}
}
