package ag.movement;

import ag.battledata.DataVector;

/**
 * An elliptical GravPoint
 * @author Andree Grosse
 */
public class GravEllipse extends GravPoint {

	private DataVector mForceDirection [];
	private double mForce[];
	
	/**
	 * Constructor
	 * @param position Position of the GravEllipse
	 * @param exp Decline exponent
	 * @param maxDist Maximum distance for force
	 * @param dir1 Direction of the first axis
	 * @param force1 Force in direction of the first axis
	 * @param dir2 Direction of the first axis
	 * @param force2 Force in direction of the first axis
	 */
	public GravEllipse(DataVector position, double exp, double maxDist,
			DataVector dir1, double force1, DataVector dir2, double force2) {
		super(position, 0, exp, maxDist);
		
		mForceDirection = new DataVector[2];
		mForce = new double[]{force1, force2}; 
		mForceDirection[0] = dir1;
		mForceDirection[1] = dir2;
				
		for(int i = 0; i < 2; i++){
			mForce[i] = Math.signum(mForce[i]) * Math.pow(Math.abs(mForce[i]), exp);
		}
	}

	/**
	 * Calculates the forces emitted on a point.
	 * @param point the point for which to calculate the force
	 * @return a DataVector representing the force
	 */
	public DataVector getForceOn(DataVector point){
		DataVector diff = point.minus(this);
		double dist = diff.normalize();
		// Calculate the scalar product, i.e. the length
		// of the difference projected on mForceDirection[i]
		double temp;
		DataVector force = new DataVector();
		
		for(int i = 0; i < 2; i++){
			temp = diff.x * mForceDirection[i].x + diff.y * mForceDirection[i].y;;
			
			force = force.plus(mForceDirection[i].multiple(temp * mForce[i]));
		}	
		return calcForce(force, dist);
	}
	
	/**
	 * Applies force formula on a vector
	 * @param dir the simple difference vector
	 * @param len distance from force emitter to point
	 * @return a DataVector representing the force 
	 */
	private DataVector calcForce(DataVector dir, double len){
		if(len > mMaxDist){
			return new DataVector(.0, .0);
		}
		else if(len < 0.1)
			len = 0.1;
		len = Math.pow(len, mExp);		
		return dir.multiple(1 / len);
	}
	
}
