package tzu.intel;

//import tzu.team.*;
//import tzu.team.message.*;
import tzu.util.*;
import java.util.*;
import java.io.*;
import robocode.*;

/**
 * Provides management of enemy robot data.
 */
public class EnemyManager extends AbstractManager
    implements EnemyManagerInterface{

    Bot[]   bots;       // An array to store enemy bot info.
    boolean isSorted;   // Is array sorted by name?
    int     knownCount; // Number of bots stored in array thus far.
    int     totalCount; // Same as robot.getOthers() at begining of game.
    int     liveCount;  // Number of bots still alive.

    /**
     * Create a new EnemyManager object.
     * @param ar    Your AdvancedRobot.
     */
    public EnemyManager(AdvancedRobot ar) {
        super(ar, null);
        totalCount  = myRobot.getOthers();
        bots        = new Bot[totalCount];
        isSorted    = false;
        knownCount  = 0;
        liveCount   = totalCount;
    }

    /**
     * Reinitialize all the robots we know about in the second and subsequent
     * rounds of play.  Near the end of the battle, delete old data files
     * to make room for new data files on our current enemies.
     */
    public void reinitialize() {
        for (int i = 0; i < knownCount; i++) {
            bots[i].reinitialize();
        }
        liveCount = totalCount;

        if (myRobot.getRoundNum() == myRobot.getNumRounds() - 1) {
            makeRoomForNewFiles();
        }
    }

    /** Overridden to do nothing. */
    public void takeTurn() {}

    /** Return the number of enemies we have data on. */
    public int size() {
        return knownCount;
    }

    /**
     * Returns the bot at the specified index.
     */
    public Bot get(int index) {
        return bots[index];
    }

    /**
     * Returns the bot with the specified name.  Returns null if not found.
     */
    public Bot get(String name) {

        int i;
        if (isSorted) {
            i = Arrays.binarySearch(bots, name);
            if (i > -1) {
                return bots[i];
            }
        } else {
            for (i = 0; i < knownCount; i++) {
                if (bots[i].getName().equals(name)) {
                    return bots[i];
                }
            }
        }
        return null;
    }


    /**
     * Update the bot array with the given scan event data.
     * Once the bot array is filled, it is sorted and
     * subsequent lookups are done using binarySearch();
     */
    public void update (ScannedRobotEvent e) {

        int         i;
        String name = e.getName();

        if (e.getEnergy() < 0) liveCount--;

        if (isSorted) {

            i = Arrays.binarySearch(bots, name);
            if (bots[i].getEnergy() < 0 && e.getEnergy() > -1) liveCount++;
            bots[i].update(e);


        } else { /* Array isn't sorted; do a linear search. */

            for (i = 0; i < knownCount; i++) {

                if (bots[i].getName().equals(name)) {
                    if (bots[i].getEnergy() < 0 && e.getEnergy() > -1) {
                        liveCount++;
                    }
                    bots[i].update(e);
                    return;
                }
            }

            /*
             * The bot wasn't found in the array.  Add it now.
             * If the array is full, sort it now.
             */

            Bot b = new Bot(e, myRobot);
            b.readFromFile();
            bots[i] = b;
            knownCount++;

            if (knownCount == totalCount) {
                Arrays.sort(bots);
                isSorted = true;
            }
        }
    }


    /**
     * A convienience method for setting an enemy bot's energy to
     * to -1 (indicating it's dead).
     */
    public void updateDead (String name) {

        Bot b = get(name);
        if (b != null) {
            b.updateDead();
            liveCount--;
        }
    }

    /**
     * Calculate the total anti gravity force that all enemy bots
     * have on your bot.  Returns the relative Point (x,y) that your robot
     * is being pushed towards.  Returns null if there is no anti
     * gravity force in effect.
     */
    public Point calcAntiGravity() {

        Bot     b           = null;
        double  x           = 0.0;
        double  y           = 0.0;
        double  direction   = 0.0;
        double  force       = 0.0;
        long    time        = myRobot.getTime();

        for (int i = 0; i < knownCount; i++) {

            b = bots[i];

            if (b.getEnergy() > -1) {

                force = BUFFER_ROBOT - b.estimateDistanceAt(time);


                if (force > 0) {
                    force *= ANTI_GRAVITY_ROBOTS;
                    direction = (b.estimateBearingAt(time) + A_180) % A_360;
                    x += (BotMath.sin(direction) * force);
                    y += (BotMath.cos(direction) * force);
                }
            }
        }
        return new Point (x, y);
    }



    /**
     * Take the data we have accumulated on each enemy and write
     * it to file.  If there is not enough room in the data
     * directory to store a file for each bot, sort the
     * bot array by difficulty and store files for the most
     * difficult bots first until the directory is full.
     */
    public void writeToFile() {

        int maxFiles = DATA_DIRECTORY_SIZE / Bot.ROBOT_FILE_SIZE;

        if (knownCount > maxFiles) {
            if (myRobot.getRoundNum() == myRobot.getNumRounds() - 1) {
                Arrays.sort(bots, new CompareByDifficulty());
            }
        }

        maxFiles = Math.min(maxFiles, knownCount);

        for (int i = 0; i < maxFiles; i++) {
            bots[i].writeToFile();
        }
    }



    /** Return statistics on all bots in the array. */
    public String stats() {
        String s = "";
        for (int i = 0; i < knownCount; i++) {
            s = s + bots[i].stats();
        }
        return s;
    }


    /** Flag a robot as dead. */
    public void onRobotDeath(RobotDeathEvent e)     {
        updateDead(e.getName());
    }


    /** Yahoo - we hit something!  Update the direct hit count of the victim. */
    public void onBulletHit(BulletHitEvent e) {

        Bot b = get(e.getName());
        if (b != null) {
            b.directHit(e.getBullet().getPower());
        }
    }


    /** Update the shot me count of the robot that shot us. */
    public void onHitByBullet(HitByBulletEvent e) {

        Bot b = get(e.getName());
        if (b != null) {
            b.shotMe(e.getPower());
        }
    }


    /** Update the ram count of the robot that rammed us. */
    public void onHitRobot(HitRobotEvent e) {

        Bot b = get(e.getName());
        if (b != null && e.isMyFault() == false) {
            b.rammedMe();
        }
    }

    public void onScannedRobot(ScannedRobotEvent e) {
        this.update(e);
    }

    /** Call the gameOver() method. */
    public void onDeath(DeathEvent e)   { gameOver(); }
    /** Call the gameOver() method. */
    public void onWin(WinEvent e)       { gameOver(); }

    /**
     * If the last round was just played, save each robot's
     * reactions in a file and print stats on each robot.
     */
    public void gameOver() {

        if (myRobot.getRoundNum() == myRobot.getNumRounds() - 1) {
            writeToFile();

            if (DEBUG) {
                myRobot.out.println(stats());
            }
        }
    }

    /**
     * Remove old data files to make room for new files that will
     * be written at the end of this battle.
     */
    public void makeRoomForNewFiles() {

        int i;

        long dataToBeWritten    =
                BotMath.min(Bot.ROBOT_FILE_SIZE *
                totalCount,
                DATA_DIRECTORY_SIZE);

        long    totalBytes      = 0;
        long    bytesDeleted    = 0;

        File    directory       = null;
        File[]  robotFiles      = null;
        File    file            = null;
        int     length;
        String  fileName;

        /*
         * If the data directory does not exist, yet, then
         * it can't possibly be full.
         */
        directory  = myRobot.getDataDirectory();
        if (directory != null) {

            /*
             * If there are no files in the directory, then
             * we certainly have room.
             */
            robotFiles = directory.listFiles();
            if (robotFiles != null) {

                for (i = 0; i < robotFiles.length; i++) {
                    totalBytes += robotFiles[i].length();
                }

                if (totalBytes + (dataToBeWritten) >= DATA_DIRECTORY_SIZE) {

                    for (i = 0; i < robotFiles.length &&
                        bytesDeleted < dataToBeWritten; i++) {

                        file = robotFiles[i];
                        fileName = file.getName();
                        length = fileName.length();
                        bytesDeleted += file.length();


                        if (get(fileName.substring(0, length - 4)) == null) {

                            /*
                             * Still having problems deleting files, so
                             * overwrite existing files with zero bytes to
                             * make room for new data files.
                             */
                            try {
                                RobocodeFileOutputStream out =
                                        new RobocodeFileOutputStream(file);
                                out.close();

                                file.delete();

                            } catch (IOException e) {}
                        }
                    }
                }
            }
        }
    }
}
