package djc;
import robocode.*;
import java.util.Hashtable;
import java.util.Enumeration;

/**
 * GunManager - Based on the StrategyManager
 * class by Simon Parker in JollyNinja
 *
 *   Manages current gun strategy for the StrategyBot.
 */
public class GunManager extends Object
{
    /* ********************************************************************************** */
    /*                                   CONSTANTS                                        */
    /* ********************************************************************************** */
    public static final String AIMED     = "AIMED";                 // Works
    public static final String BUTTERFLY = "BUTTERFLY";             // Experimental
    public static final int NUM_GUNS     = 2;
    public static final int MIN_SHOTS    = 10;
    public static final int AIMEDID      = 0;
    public static final int BUTTERFLYID  = 1;

    /* ********************************************************************************** */
    /*                                MEMBER VARIABLES                                    */
    /* ********************************************************************************** */
    /** The initial strategy to use by default. */
    //public String INITIAL = AIMED;
    public String INITIAL = BUTTERFLY;
    public String MELEE = AIMED;
    public StrategyBot self = null;
    public Hashtable availableStrategies = new Hashtable();
    public static double energyLost[] = new double[NUM_GUNS];
    public static double energyGained[] = new double[NUM_GUNS];
    public static double damageDone[] = new double[NUM_GUNS];
    public static int shots[] = new int[NUM_GUNS];

    // Order is also order of preference
    private String[] strategyNames = {AIMED, BUTTERFLY};

    /**
     * GunManager constructor
     * 
     * Currently adds the following TacticalStrategies:
     * <ul>
     *   <li> AimedGun </li>
     *   <li> ButterflyGun </li>
     * </ul>
     *
     * @param theRobot - who I work for.
     */
    public GunManager(StrategyBot theRobot)
    {
	self = theRobot;
	Gun aGun;
	
	aGun = new AimedGun(self, this);
	availableStrategies.put(aGun.name, aGun);

	aGun = new ButterflyGun(self, this);
	availableStrategies.put(aGun.name, aGun);

	for(int i=0;i<NUM_GUNS;i++) {
	    energyLost[i] = 0;
	    energyGained[i] = 0;
	    damageDone[i] = 0;
	}
    }

    /**
     * Change current strategy to newStrategyName.  If this
     * is not found, go back to the INITIAL.
     *
     * @param newStrategyName
     */
    public void setGun(String newGunName)
    {
	if (newGunName != self.currentGun.name) {
	    self.out.println("Switching to gun: " + newGunName);
	    Gun newGun;
	    newGun =  (Gun)availableStrategies.get(newGunName);
	    if(newGun == null) newGun =  (Gun)availableStrategies.get(INITIAL);

	    self.currentGun = newGun;
	    self.currentGun.reset();
	}
    }

    /**
     * Change current strategy to newStrategyName.  If this
     * is not found, go back to the INITIAL.
     *
     * @param gunID
     */
    public void setGun(int gunID)
    {
	if(gunID == BUTTERFLYID) setGun(BUTTERFLY);
	else if(gunID == AIMEDID) setGun(AIMED);
	else {
	    self.out.println("setGun(int) error in gunID=" + gunID);
	    if(self.getOthers() > 4) setGun(MELEE);
	    else setGun(INITIAL);
	}
    }

    /**
     * Reset every strategy.
     */
    public void reset()
    {
	Gun aGun;
	Enumeration e = availableStrategies.elements();
	while(e.hasMoreElements()) {
	    aGun = (Gun)e.nextElement();
	    aGun.reset();
	}
	setInitialGun();
	decideGunChange();
    }

    /**
     * Initialize with default gun
     */
    public void setInitialGun()
    {
	Gun aGun;

	if(self.bOneVOne) 
	    aGun = (Gun)availableStrategies.get(INITIAL);
	else
	    aGun = (Gun)availableStrategies.get(MELEE);
	self.currentGun = aGun;
	self.out.println("Initializing current round with gun: " + self.currentGun.name);
    }

    /**
     * getGunEffectiveness
     *
     * 
     *
     *  @return goodness
     */
    public final double getGunEffectiveness(int id)
    {
	//return (energyGained[id] - energyLost[id] + damageDone[id]);
	//return (energyGained[id] - energyLost[id]);
	//return (energyGained[id] - energyLost[id] + damageDone[id]) / Math.max((double)shots[id],1);
	return (energyGained[id] - energyLost[id] + damageDone[id]) / Math.max(energyLost[id],1);
    }

    /**
     * Decides when to change strategies
     */
    public void decideGunChange()
    {
	if(self.currentGun.name == BUTTERFLY && shots[BUTTERFLYID] < MIN_SHOTS) return;
	if(self.currentGun.name == AIMED && shots[AIMEDID] < MIN_SHOTS) return;
	if(self.currentGun.name == AIMED && (self.getOthers() > 4 || self.getEnergy() < 12)) return;
	if(self.currentGun.name != AIMED && (self.getOthers() > 4 || self.getEnergy() < 12)) {
	    setGun(AIMEDID);
	    return;
	}
	
	double curEffectiveness = Double.MIN_VALUE;
	double bestEffectiveness = Double.NEGATIVE_INFINITY;
	int bestI = 0;

	for(int i=0;i<NUM_GUNS;i++) {
	    if(shots[i] < MIN_SHOTS && self.currentTarget != null) {
		if(self.currentTarget.damageInflicted + self.survivalBonus * self.currentTarget.outSurvived 
		   - self.currentTarget.damageSustained < 0) {
		    bestI = i;
		    break;
		}
	    }
	    curEffectiveness = getGunEffectiveness(i);
	    //self.out.println("i=" + i + " effectiveness=" + curEffectiveness);
	    if(curEffectiveness > bestEffectiveness) {
		bestI = i;
	    }
	}
	setGun(bestI);
    }
}
