package ags.lunartwins;

import robocode.*;
import static robocode.util.Utils.normalAbsoluteAngle;
import static robocode.util.Utils.normalRelativeAngle;
import ags.lunartwins.util.*;

/**
 * The Leader of the LunarTwins
 * 
 * @author Alexander Schultz
 */
public class LunarLeader extends LunarBase {
    private static EnemyRobot enemyleader, otherenemy;
    private static AbsolutePoint allylocation;
    
    public LunarLeader() {
        enemyleader = null;
        otherenemy = null;
        allylocation = null;
    }
    
    public void onScannedRobot(ScannedRobotEvent e) {
        String name = e.getName();
        if (!isTeammate(name)) {
            if (e.getEnergy() >= 150 || (enemyleader != null && enemyleader.name == name)) {
                enemyleader = EnemyRobot.gotScan(enemyleader, this, e);
            } else {
                otherenemy = EnemyRobot.gotScan(otherenemy, this, e);
            }
        }
    }
    
    public boolean isUpToDate(EnemyRobot enemy) {
        return (enemy != null && getTime()-enemy.time <= 1);
    }
    
    public EnemyRobot getClosest() {
        if ((getOthers()-getTeammates().length >= 2) || isUpToDate(otherenemy))
            return otherenemy;
        else if (isUpToDate(enemyleader))
            return enemyleader;
        else
            return null;
    }
    
    public AbsolutePoint getGoal() {

        // If we know where our ally is, calculate our average location
        AbsolutePoint avglocation = location;
        if (allylocation != null) {
            avglocation = AbsolutePoint.fromXY((location.x+allylocation.x)/2, (location.y+allylocation.y)/2);
        }
        
        try {
            //Calculate where the average location is relative to the enemy
            EnemyRobot predictplace = (EnemyRobot)getClosest().clone();
            predictplace.predictTick();
            RelativePoint rel = RelativePoint.fromPP(predictplace.location, avglocation);
            
            // Create goal points on each side of the enemy
            rel = RelativePoint.fromDM(rel.getDirection()+(Math.PI/2), Math.max(50, rel.magnitude-10));
            AbsolutePoint goal = predictplace.location.addRelativePoint(rel);
            AbsolutePoint goal2 = predictplace.location.addRelativePoint(rel.incrementDirMag(Math.PI, 0));
            
            // If we know where our ally is, untangle our paths
            if (allylocation != null) {
                double a1 = goal.distance(location);
                double b1 = goal2.distance(allylocation);
                double a2 = goal2.distance(location);
                double b2 = goal.distance(allylocation);
                
                if (!(a1 < a2 && a1 < b2) && !(b1 < a2 && b1 < b2))
                { AbsolutePoint tmp = goal; goal = goal2; goal2 = tmp; }


                // Adjust the goal locations to not run into the enemy            
                RelativePoint g1 = RelativePoint.fromPP(predictplace.location, goal2);
                RelativePoint g2 = RelativePoint.fromPP(predictplace.location, goal);
                RelativePoint o1 = RelativePoint.fromPP(predictplace.location, location);
                RelativePoint o2 = RelativePoint.fromPP(predictplace.location, allylocation);
                g1.direction = fixDirection(g1, o1);
                g2.direction = fixDirection(g2, o2);
                goal = predictplace.location.addRelativePoint(g1);
                goal2 = predictplace.location.addRelativePoint(g2);
            }
            
            goal = wallAdjust(goal, predictplace.location);
            goal2 = wallAdjust(goal2, predictplace.location);
            
            // Tell our ally where the enemy is, where to move relative to it, and scan data of the enemy.
            broadcastMessage(new OrbitPointMessage(goal2, (EnemyRobot)predictplace.clone()));
            //broadcastMessage(location);            
            return goal;
        } catch (Exception e) {
            return null;
        }
    }
    
    public static double fixDirection(RelativePoint g, RelativePoint o) {
        double rel = normalRelativeAngle(o.direction - g.direction);
        if (rel > Math.PI/2) rel = Math.PI/2;
        if (rel < -Math.PI/2) rel = -Math.PI/2;
        return normalAbsoluteAngle(o.direction + rel);
    }
    
    public AbsolutePoint getTarget() {
        try {
            // Fire at the location predicted by CircularTargeting
            EnemyRobot e = (EnemyRobot)getClosest().clone();
            return e.predict(location, getFirePower());
        } catch (Exception ex) {
            return null;
        }
    }
    
    public double currentDist() {
        return getClosest().relativelocation.getMagnitude();
    }
    
    private static final double turnpast = (2*Math.PI)/72;
    
    public void doRadar() {
        double radarturn;
        try {
            // Calculate the angle the radar needs to turn to point at the enemy
            radarturn = getClosest().relativelocation.getDirDist(getRadarHeadingRadians());
            // And turn past it some
            radarturn += Math.signum(radarturn)*turnpast;
        } catch (Exception ex) {
            radarturn = Double.POSITIVE_INFINITY;
        }
        setTurnRadarRightRadians(radarturn);
    }
    
    public void onMessageReceived(MessageEvent e) {
        try {
            allylocation = (AbsolutePoint)e.getMessage();
        } catch (Exception ex) {}
    }
}
