package sch;

import robocode.*;

/**
 * Functions used to calculate where the robot should be at a certain moment.<br>
 * This intercept method uses an iterative linear intercept.
 */
public class CircularIntercept extends LinearIntercept implements Constants {
	
		public String getName() {return "Precise Iterative Circular Intercept";}
		public String getKey() {return "Circular";}
		public int getComplexity() {return 4;}

	/**
	 * Calculate the position where the bot will be at a future time.<br>
	 * This use an iteration to calculate the precise position
	 * @param		myBotPos				position of my bot
	 * @param		targetPos				current position of the target
	 * @param		now						current time
	 * @param		targetPositionTiming	time of the target position
	 * @param		targetHeading			heading of the target
	 * @param		targetSpeed				speed of the target
	 * @param		bulletSpeed				speed of the bullet fired
	 * @param		targetAngSpeed			angular speed of the target
	 * @return								estimated position of the target
	 */
	 
		public Point calculate(	Point myBotPos,
							Point targetPos,
							double now,
							double targetPositionTiming,
							double targetHeading,
							double targetSpeed,
							double bulletSpeed,
							double targetAngSpeed) {

		int iterationCount=0;
		double supposedImpactTime;
		Point oldPos;
		Point newPos=new Point(targetPos);
		do {
			oldPos=newPos;
			supposedImpactTime=calcTime(myBotPos,oldPos,bulletSpeed,now);
			newPos=guessPosition(targetPos,targetHeading,targetSpeed,targetAngSpeed,targetPositionTiming,supposedImpactTime);
			iterationCount++;
			if (BotUtil.varContains(DEBUG,DEBUG_AIMING_CALC)) {
				System.out.println(iterationCount+ "|"+ BotMath.getRange(oldPos,newPos));
			}
		} while(iterationCount<INTERCEPT_ITERATIONS && BotMath.getRange(oldPos,newPos)>INTERCEPT_ACCURACY);
		return newPos;
	}
	
	public Point guessPosition(	Point targetPosition,
								double targetHeading,
								double targetSpeed,
								double targetAngSpeed,
								double targetPositionTiming,
								double estimatedTime) {
		
		double x;
		double y;
		
		if (Math.abs(targetAngSpeed)>ANG_SPEED_LIMIT) {
			double diffTime = estimatedTime - targetPositionTiming;
			double radius = targetSpeed/targetAngSpeed;
			double newHeading = targetHeading + (diffTime*targetAngSpeed);
			
			x = targetPosition.x + Math.cos(targetHeading)*radius - Math.cos(newHeading)*radius;
			y = targetPosition.y+ Math.sin(newHeading)*radius - Math.sin(targetHeading)*radius;
		}
		else {
			return super.guessPosition(targetPosition,targetHeading,targetSpeed,0,targetPositionTiming,estimatedTime);
		}

		return new Point(x,y);
	}	
}
