package sgp;
import robocode.*;
import java.util.*;
import java.awt.Color;
import java.io.*;

/**
 * TeamLeaderStrategy - a class by Simon Parker
 */
public class TeamLeaderStrategy extends Strategy
{
	protected TeamRobot robot;

	protected final int NUM_POSITIONS = 4;
	protected TargetTable targetTable = new TargetTable();
	protected TeamTarget currentTarget = null;
	protected TeamTarget teamTarget = null;

	protected Coordinate currentPosition = new Coordinate();


	public TeamLeaderStrategy (DrunkenStrategyManager theStrategyManager)
	{
		super(Environment.getRobot(), theStrategyManager);
		robot = Environment.getTeamRobot();
	}

	public void startTurn() //any init that needs doing at the start of the turn
	{
		currentPosition.set(robot.getX(), robot.getY());
		currentTarget = targetTable.getClosestEnemy(currentPosition);

//???		teamTarget = targetTable.getEnemyLeader();
//???		if ((teamTarget == null) || !teamTarget.isAlive)
//???		{
//???			teamTarget = currentTarget;
//???		}

		if (teamTarget != null)
		{
			try
			{
				// Send enemy to teammates
				robot.broadcastMessage(teamTarget.name);
			} catch (IOException ex)
			{
				System.out.println("Unable to send order: " + ex);
			}
			//robot.out.println("Target is " + currentTarget.name + "(" + (int)currentTarget.position.x + "," + (int)currentTarget.position.y + ")");
		}
/*
		try
		{
			RobotSnapshot snapshot = new RobotSnapshot(	robot.getName(), currentPosition,
				robot.getVelocity(), robot.getHeading(), 0, robot.getEnergy(), robot.getTime());

			// Send enemy position to teammates
			robot.broadcastMessage(snapshot);
		} catch (IOException ex)
		{
			System.out.println("Unable to send order: " + ex);
		}
*/
	}



	public void onScannedRobot(ScannedRobotEvent e)
	{
		TeamTarget target = targetTable.getByName(e.getName());

		double angle_rad = robot.getHeadingRadians() + e.getBearingRadians();

		double currentTargetPosX = robot.getX() + e.getDistance() * Math.sin(angle_rad);
		double currentTargetPosY = robot.getY() + e.getDistance() * Math.cos(angle_rad);

		Coordinate currentTargetPosition = new Coordinate(currentTargetPosX, currentTargetPosY);

		double dT = robot.getTime() - target.lastTimeScanned;
		double dH = normalRelativeAngle(robot.getHeadingRadians() - target.heading_deg);
		double angularVelocity = 0;

		if ((dT != 0) && (dT < 4))
		{
			angularVelocity = dH / dT;
		}

//		if (!robot.isTeammate(e.getName()))
//		{
			RobotSnapshot snapshot = new RobotSnapshot
			(
				e.getName(),
				currentTargetPosition,
				e.getVelocity(),
				e.getHeading(),
				angularVelocity,
				e.getEnergy(),
				robot.getTime()
			);
			target.set(snapshot);
/*
			try
			{
				// Send enemy position to teammates
				robot.broadcastMessage(snapshot);
			} catch (IOException ex)
			{
				System.out.println("Unable to send order: " + ex);
			}
*/
//		}

	}

	public void setGunRotation() //control the gun motion and fires if neccessary
	{
		if (currentTarget == null) return;

		// if the firing angle to hit a bullet is within a certain range of that
		//  angle required to hit the robot, then fire at the bullet otherwise fire
		//  at the target robot

		Coordinate estimatedTargetPosition = currentTarget.getEstimatedPosition(robot.getTime());
		double enemyRobotFiringPower = calculateBulletPower(currentPosition, estimatedTargetPosition);
		Intercept intercept = getEnemyFiringIntercept(enemyRobotFiringPower);

		double turnAngle = normalRelativeAngle(intercept.bulletHeading_deg - robot.getGunHeading());
		// move gun to target angle
		robot.setTurnGunRight(turnAngle);
		if 	(Math.abs(turnAngle) <= intercept.angleThreshold)
		{
			if 	(
					(intercept.impactPoint.x < 0) ||
					(intercept.impactPoint.x > robot.getBattleFieldWidth()) ||
					(intercept.impactPoint.y < 0) ||
					(intercept.impactPoint.y > robot.getBattleFieldHeight())
				)
			{
				//impact point off the field
				//robot.out.println("T " + intercept.impactTime + ", x " + intercept.impactPoint.x + ", y " + intercept.impactPoint.y);
			}
			else
			{
				if (robot.getGunHeat() == 0)
				{
			//		if (!isFriendlyFire(intercept.impactPoint))
			//		{
						Bullet bullet = robot.fireBullet(enemyRobotFiringPower);
			//		}
				}
			}
		}
		else
		{
			// gun not in position
		//	robot.out.println("ta " + turnAngle + ", th " + intercept.angleThreshold);
		}

	}

	private boolean isFriendlyFire(Coordinate impactPoint)
	{
		boolean result = false;
		for (int i = 0; i < targetTable.getNum(); i++)
		{
			TeamTarget observedTarget = targetTable.getAt(i);

			if ((observedTarget.isAlive) && !observedTarget.isEnemy())
			{
				double crossfireDistance = observedTarget.getEstimatedPosition(robot.getTime()).distanceFrom(currentPosition, impactPoint);
				if (crossfireDistance < 30)
				{
					result = true;
				}
				else
				{
//					robot.out.println("d " + (int)crossfireDistance);
				}
			}
		}
		return result;

	}

	public void endTurn(){}
	public void onHitByBullet(HitByBulletEvent e){}
	public void onDeath(DeathEvent event){}
	public void onHitWall(HitWallEvent e){}
	public void onHitRobot(HitRobotEvent e) {}


	public void onRobotDeath(RobotDeathEvent e)
	{
		targetTable.setRobotDead(e.getName());
		//Victory dance
		if (robot.getOthers() == 0)
		{
			robot.setAhead(0);
			robot.turnRight(36000);
		}
	}

	public void reset()
	{
		robot = Environment.getTeamRobot();
		targetTable.reset();
	}

	public void setScan() //control the radar motion
	{
		// optimal scanning calls for scanning in the direction of the target that was
		// scanned the longest ago

		TeamTarget scanTarget = targetTable.getMostOutDatedTarget();

		if ((scanTarget == null) || (robot.getTime() < 10))
		{
			robot.setTurnRadarRight(720);
			return;
		}

		double dT = robot.getTime() - scanTarget.lastTimeScanned;
		if (dT > 10)
		{
			robot.setTurnRadarRight(720);
			return;
		}

		double targetRadarAngle = currentPosition.headingTo(scanTarget.position);
		double radarTurnRightAngle = normalRelativeAngle(targetRadarAngle - robot.getRadarHeading());

		if (radarTurnRightAngle > 0)
		{
			radarTurnRightAngle += 20;
		}
		else
		{
			radarTurnRightAngle -= 20;
		}

		robot.setTurnRadarRight(radarTurnRightAngle);
	}

	public void onMessageReceived(MessageEvent e)
	{

		if (e.getMessage() instanceof RobotSnapshot)
		{
			RobotSnapshot robotSnapshot = (RobotSnapshot)e.getMessage();
			targetTable.update(robotSnapshot);
		}


	}

	public double evaluateGoodness(Coordinate robotPosition, double robotHeading, double robotVelocity)
	{
		final double TARGET_DISTANCE_WEIGHT = 10;
		final double OPTIMUM_TARGET_DISTANCE = 300;
		final double NON_TARGET_DISTANCE_WEIGHT = 100;
		final double AVOIDANCE_DISTANCE = 90;
		final double BEARING_AVOIDANCE_WEIGHT = 20;
		final double OPTIMUM_BEARING_DIFFERENCE = 20.0;
		double goodness = 0;
		if (currentTarget != null)
		{
			//get to the optimum distance from the target
			goodness += -TARGET_DISTANCE_WEIGHT * Math.abs(robotPosition.distanceFrom(currentTarget.position) - OPTIMUM_TARGET_DISTANCE);

			//keep out of the firing line of your teammates
			TeamTarget closestBearingTeammate = targetTable.getClosestBearingTeammate(robotPosition, currentTarget.position);
			if (closestBearingTeammate != null)
			{
				double teammateBearingToTarget = closestBearingTeammate.getEstimatedPosition(robot.getTime()).headingTo(currentTarget.position);
				double robotBearingToTarget = robotPosition.headingTo(currentTarget.position);
				double bearingDifference = Math.abs(Strategy.normalRelativeAngle(teammateBearingToTarget - robotBearingToTarget));
				goodness += -BEARING_AVOIDANCE_WEIGHT * Math.abs(bearingDifference - OPTIMUM_BEARING_DIFFERENCE);
			}
		}

		//don't hit anyone
		TeamTarget closestTarget = targetTable.getClosestTarget(robotPosition);
		if (closestTarget != null)
		{
			double avoidanceGoodness = robotPosition.distanceFrom(closestTarget.position);
			avoidanceGoodness = NON_TARGET_DISTANCE_WEIGHT * Math.min(avoidanceGoodness, AVOIDANCE_DISTANCE);
			goodness += avoidanceGoodness;
		}

		//don't hit the walls
		goodness += evaluateWallGoodness(robotPosition);

		return goodness;
	}

	private double calculateBulletPower
	(
		Coordinate robotPosition,
		Coordinate targetPosition
	)
	{
		final double ROBOT_RADIUS = 20;
		final double MAX_BULLET_POWER = 3.0;
		final double MIN_BULLET_POWER = 0.5;
		double distance = robotPosition.distanceFrom(targetPosition);

		double power = MAX_BULLET_POWER - (distance-2 * ROBOT_RADIUS)/300.0;
		power = Math.max(MIN_BULLET_POWER, power);
		power = Math.min(MAX_BULLET_POWER, power);
		//return 0.1;
		return power;
	}

	public Intercept getEnemyFiringIntercept(double power)
	{
		if (currentTarget == null) return null;
		Intercept intercept;


		if (
			(currentTarget.jiggleAnalyser.isJiggling) &&
			(currentPosition.distanceFrom(currentTarget.position) > 50) &&
			(currentTarget.jiggleAnalyser.isOscilating)
		   )
		{
			intercept = new JiggleIntercept(currentTarget.jiggleAnalyser);
		}
		else
		{
			intercept = new PatternIntercept(currentTarget.patternAnalyser);
		}

		intercept.calculate
		(
			currentPosition,
			currentTarget.getEstimatedPosition(robot.getTime()),
			currentTarget.heading_deg,
			currentTarget.velocity,
			power,
			currentTarget.angularVelocity
		);
		return intercept;
	}

}
