/*
 * Written by Kinsen Choy
 */

package kinsen.melee.Guns;
import kinsen.melee.Details;
import kinsen.melee.Utils;
import java.awt.Point;

/*
 * GuessFactor - A gun class that fires guess factor predicted bullets.
 */
public class GuessFactor extends Gun
{
	// Last enemy details
	private Details enemy = null;
	// Last my details
	private Details me = null;

	// Data
	private final static int NUMBER_GUESS_FACTORS = 40;
	private int guessFactorsCount[] = new int[NUMBER_GUESS_FACTORS];

	// For virtual bullets
	private double currentMaximumAngle = 0;

	private double randomGuessFactor = Math.random() * 2 - 1;
	private double bestGuessFactor = -2;

	public void calculateGun(double initialFirePower)
	{
		// If there is no data then it can not fire
		if (enemy != null)
		{
			double guessFactor = getBestGuessFactor();
			double enemyDistance = Utils.distance(me.getX(), me.getY(), enemy.getX(), enemy.getY());
			double maximumDistance = enemyDistance / (20 - 3 * initialFirePower) * 8;
			double maximumAngle = maximumDistance / (2 * Math.PI * enemyDistance) * 360;
			currentMaximumAngle = maximumAngle;
			double angle = guessFactor * maximumAngle;
			angle += Utils.angleFromPoint(me.getX(), me.getY(), enemy.getX(), enemy.getY());
			double fireX = me.getX() + Math.sin(Math.toRadians(angle)) * enemyDistance;
			double fireY = me.getY() + Math.cos(Math.toRadians(angle)) * enemyDistance;

			// Sets target location and fire power
			firePower = initialFirePower;
			fireAt = new Point((int) fireX, (int) fireY);
		}
	}

	public void calculateGun(double initialFirePower, boolean canChangePower)
	{
		calculateGun(initialFirePower);
	}

	public void recordData(Details enemyDetails, Details myDetails, long time)
	{
		enemy = enemyDetails;
		me = myDetails;
	}

	/*
	 * Increases count for guess factor
	 */
	public void recordGuessFactor(double guessFactor)
	{
		int index = (int) Math.round(guessFactor * (NUMBER_GUESS_FACTORS - 1) / 2) + (int) ((NUMBER_GUESS_FACTORS - 1) / 2);
		guessFactorsCount[index]++;
		if (guessFactorsCount[index] > 10)
		{
			// Decay stats
			for (int i = 0; i < NUMBER_GUESS_FACTORS; i++)
			{
				guessFactorsCount[i] /= 2;
			}
		}
	}

	/*
	 * Returns the angle from this robot to the enemy robot
	 */
	public double getCurrentAngle()
	{
		return Utils.angleFromPoint(me.getX(), me.getY(), enemy.getX(), enemy.getY());
	}

	/*
	 * Returns the maximum angle if it stayed at the same distance
	 */
	public double getMaximumAngle()
	{
		// Generate new random guess factor
		randomGuessFactor = Math.random() * 2 - 1;
		bestGuessFactor = -2;
		return currentMaximumAngle;
	}

	/*
	 * getBestGuessFactor: Returns the best guess factor based on stats or a random one
	 */
	private double getBestGuessFactor()
	{
		if (bestGuessFactor != -2)
			return bestGuessFactor;
		int maxNumber = -1;
		int bestIndex = -1;
		int totalCount = 0;
		for (int i = 0; i < NUMBER_GUESS_FACTORS; i++)
		{
			totalCount += guessFactorsCount[i];
			if (bestIndex == -1 || guessFactorsCount[i] > maxNumber)
			{
				maxNumber = guessFactorsCount[i];
				bestIndex = i;
			}
			else if (guessFactorsCount[i] == maxNumber && Math.random() > 0.9)
				bestIndex = i;
		}

		// Guess factor is not dominant enough
		if ((double) maxNumber <= (double) totalCount * .1)
		{
			return randomGuessFactor;
		}
		else
		{
			bestGuessFactor = (double) bestIndex * 2 / (double) NUMBER_GUESS_FACTORS - 1;
			return bestGuessFactor;
		}
	}
}