package ags.rougedc.gun;

import robocode.Rules;
import java.util.List;
import java.util.ArrayList;
import java.util.LinkedList;

/**
 * A node in the patternmatcher
 */
public final class PatternNode {
    private final static double MAX_TURN = Rules.MAX_TURN_RATE_RADIANS;
    private final static double MAX_ACCEL = Math.max(Rules.ACCELERATION, Rules.DECELERATION);
    private final static int TURN_SECTIONS = 11; // Must be odd
    private final static int ACCEL_SECTIONS = 5; // Must be odd
    private final static int SECTIONS = TURN_SECTIONS*ACCEL_SECTIONS;
    private final static double TURN_DIVISION = MAX_TURN / ((TURN_SECTIONS - 1) / 2);
    private final static double ACCEL_DIVISION = MAX_ACCEL / ((ACCEL_SECTIONS - 1) / 2);
    
    public static final int getTurnIndex(double turn) {
        return (int)(Math.min(MAX_TURN*2, Math.max(0, (turn + MAX_TURN))) / TURN_DIVISION);
    }
    
    public static final int getAccelIndex(double accel) {
        return (int)(Math.min(MAX_ACCEL*2, Math.max(0, (accel + MAX_ACCEL))) / ACCEL_DIVISION);
    }
    
    private final PatternNode[][] children = new PatternNode[TURN_SECTIONS][ACCEL_SECTIONS];
    private long hitcount = 0;
    public double ACCEL, TURN;
    
    public PatternNode(int turnIndex, int accelIndex) {
        TURN = turnIndex*TURN_DIVISION - MAX_TURN;
        ACCEL = accelIndex*ACCEL_DIVISION - MAX_ACCEL;
    }
    
    public final PatternNode getChild(int turnIndex, int accelIndex) {
        return children[turnIndex][accelIndex];
    }
    
    public final PatternNode makeChild(int turnIndex, int accelIndex) {;
        final PatternNode child = children[turnIndex][accelIndex];
        if (child == null) {
            final PatternNode newchild = new PatternNode(turnIndex, accelIndex);
            children[turnIndex][accelIndex] = newchild;
            return newchild;
        }
        return child;
    }
    
    public final List<PatternNode> getValidChildren(double v) {
        List<PatternNode> list = new ArrayList<PatternNode>(SECTIONS);
        for (int i=0; i<TURN_SECTIONS; i++) {
            for (int j=0; j<ACCEL_SECTIONS; j++) {
                final PatternNode node = children[i][j];
                if (node != null && Math.abs(v+node.ACCEL) <= Rules.MAX_VELOCITY && Math.abs(node.TURN) <= Rules.getTurnRate(v))
                    list.add(node);
            }
        }
        return list;
    }
    
    public final long getHitCount() {
        return hitcount;
    }
    
    public final void addHistory(LinkedList<PatternPredictor.HistoryPoint> history, double turn, double accel) {
        TURN = (TURN * hitcount + turn) / (hitcount + 1);
        ACCEL = (ACCEL * hitcount + accel) / (hitcount + 1);
        hitcount++;
        if (history.size() == 0)
            return;
        // PatternPredictor.HistoryPoint nextpoint = history.pollFirst(); Not compatible with Java 5
        PatternPredictor.HistoryPoint nextpoint = history.getFirst(); history.removeFirst();
        makeChild(nextpoint.turnIndex, nextpoint.accelIndex).addHistory(history, nextpoint.turn, nextpoint.accel);
    }
}
