package romz.component.targeting.historyanalysis;

import java.util.ArrayList;
import java.util.List;

import static romz.math.RobotMath.abs;
import romz.component.targeting.base.AbstractTargeting;
import romz.model.robot.Robot;
import romz.model.wave.Wave;

public class HistoryAnalysisTargeting extends AbstractTargeting {
	
	/** The maximum distance at which we fire with maximum power */
	public static final double MAX_POWER_DISTANCE = 300;
	
	/** The weight given to side proximity similarity */
	public static final double SIDE_PROXIMITY_WEIGHT = 1;
	
	/** The weight given to heading alignment similarity */
	public static final double HEADING_ALIGNMENT_WEIGHT = 1;
	
	/** The weight given to velocity similarity */
	public static final double VELOCITY_WEIGHT = 1;
	
	/** The weight given to angular speed similarity */
	public static final double ANGULAR_SPEED_WEIGHT = 1;
	
	/** The weight given to time similarity */
	public static final double TIME_WEIGHT = 1;
	
	/** The max lower a factor can be from the best factor to be selected when computing mean */
	public static final double SELECTION_THRESHOLD = 0.01;
	
	private List<Wave> pastWaves = new ArrayList<Wave>();
	
	@Override
	public void handleWaveHit(List<Wave> hittingWaves) {
		for (Wave wave : hittingWaves) {
			pastWaves.add(wave);
			System.out.println("--- Hittig wave ---");
			System.out.println(wave.toString());
			System.out.println("-------------------\n");
		}
	}
	
	public double selectFireAngle(double firePower) {
		double fireAngle;
		if (pastWaves.isEmpty()) {
			fireAngle = circularAcceleratedAngle(firePower);
		} else {
			fireAngle = similarityFactoredOffsetAngle(firePower);
		}
		return fireAngle;
	}
	
	private double similarityFactoredOffsetAngle(double firePower) {
		double similarityFactoredOffsetAngle = getSimilarityFactoredOffsetAngle();
		return situation.enemy.absoluteBearing + situation.enemy.direction * similarityFactoredOffsetAngle;
	}

	private double getSimilarityFactoredOffsetAngle() {
		List<Double> similarityFactors = getSimilarityFactors();
//		return getMean(similarityFactors);
		return getBestAngle(similarityFactors);
	}

	private List<Double> getSimilarityFactors() {
		System.out.println("--- Enemy ---");
		System.out.println(situation.enemy.toString());
		System.out.println("-------------\n");
		List<Double> factors = new ArrayList<Double>();
		for (Wave wave : pastWaves) {
			factors.add(getSimilarityFactor(wave));
		}
		return factors;
	}
	
	private double getBestAngle(List<Double> factors) {
		int bestIndex = getBestIndex(factors);
		double angle = pastWaves.get(bestIndex).hitOffsetAngle;
		
		System.out.println("--- Getting best ---");
		System.out.println("Current angular speed : " + situation.enemy.angularSpeed);
		System.out.println(factors.get(bestIndex) + " * " + pastWaves.get(bestIndex).startEnemyAngularSpeed + ", " + angle);
		System.out.println("--------------------");
		return angle;
	}
	
	private int getBestIndex(List<Double> factors) {
		int bestIndex = 0;
		double bestFactor = factors.get(0);
		for (int i = 0; i < factors.size(); i++) {
			double factor = factors.get(i);
			if (factor > bestFactor) {
				bestFactor = factor;
				bestIndex = i;
			}
		}
		return bestIndex;
	}

	private double getMean(List<Double> factors) {
		double factorSum = 0;
		double angleSum = 0;
		System.out.println("--- Computing mean ---");
		System.out.println("Current angular speed : " + situation.enemy.angularSpeed);
		int bestIndex = getBestIndex(factors);
		double bestFactor = factors.get(bestIndex);
		double threshold = getThreshold(bestFactor);
		for (int i = 0; i < factors.size(); i++) {
			double factor = factors.get(i);
			if (factor >= threshold) {
				double angle = pastWaves.get(i).hitOffsetAngle;
				System.out.println(
						(int)(factor * 100) + " * (" + 
						pastWaves.get(i).startEnemyAngularSpeed + ", " + 
						(int)angle + ")");
				factorSum += factor;
				angleSum += factor * angle;
			}
		}
		System.out.println("---");
		System.out.println("angleSum / factorSum : " + angleSum / factorSum);
		System.out.println("----------------------\n");
		return angleSum / factorSum;
	}

	private double getThreshold(double bestFactor) {
		double threshold = 1;
		if (bestFactor < 1) {
			threshold = bestFactor - (SELECTION_THRESHOLD * bestFactor);
		}
		return threshold;
	}
	
	private double getSimilarityFactor(Wave wave) {
		System.out.println("--- Similarity of wave ---");
		System.out.println(wave.toString());
		System.out.println("\n");
		double similarityFactor = 1;
		similarityFactor *= angularSpeedFactor(wave);
		similarityFactor *= timeFactor(wave);
		System.out.println("--------------------------\n");
		return similarityFactor;
	}
	
	private double angularSpeedFactor(Wave wave) {
		double angularSpeedDiff = abs(situation.enemy.angularSpeed - wave.startEnemyAngularSpeed);
		return ANGULAR_SPEED_WEIGHT * (1 - angularSpeedDiff / Robot.MAX_ANGULAR_SPEED);
	}
	
	private double timeFactor(Wave wave) {
		double age = situation.time - wave.startTime;
		return TIME_WEIGHT * (1 - age / situation.time / 100);
	}

	@Override
	protected boolean selectFire() {
		return true;
	}
	
}
