package djc;

import djc.util.*;

import java.awt.*;
import java.awt.geom.*;     // for Point2D's
import java.util.*; 	    // for collection of waves

import robocode.*;
import robocode.util.*;

/**
 * Enemy Management
 */
public class EnemyManager
{
	/********************************************************************************
	 *                        Member Variables                                      *
	 *******************************************************************************/
	public static AbstractDynaBot myrobot = null;
	public static Hashtable enemyList;
	public Enemy currentEnemy;
		
	// Each Enemy has an entry in these Hashtables
	public static Hashtable funkyChickenPatterns    = new Hashtable(); // For pattern matcher
	public static Hashtable moebiusPatterns         = new Hashtable(); // For pattern matcher
	public static Hashtable guessFactors            = new Hashtable(); // For GF Gun - GameSegment X DistSeg X VelocityIndex X Velocity Index X GuessFactorBins
	public static Hashtable surfStats               = new Hashtable(); // For WaveSurfing; Array of size GuessFactorBins
	
	public static Hashtable enemyTargettedHits      = new Hashtable(); // GameSegment X DistSeg X NumGuns for targeting performance
	public static Hashtable enemyShotsFiredAt       = new Hashtable(); // GameSegment X DistSeg X NumGuns for targeting performance
	public static Hashtable enemyDamageTo           = new Hashtable(); // GameSegment X DistSeg - enemy performance
	public static Hashtable enemyDamageToMe         = new Hashtable(); // GameSegment X DistSeg - enemy performance
	
	// For Sandbox Flattener - based on MakoHT - http://robowiki.net/cgi-bin/robowiki?MakoHT/Code
	public static Hashtable enemyDuelShotsAtMeSF    = new Hashtable(); // By SandBoxFlattener_DistSeg - enemy performance
	public static Hashtable enemyDuelHitsSF         = new Hashtable(); // By SandBoxFlattener_DistSeg - enemy performance
	public static Hashtable enemyDuelEnergyGainedSF = new Hashtable(); // By SandBoxFlattener_DistSeg - enemy performance
	public static Hashtable enemyDuelEnergyLostSF   = new Hashtable(); // By SandBoxFlattener_DistSeg - enemy performance
	public static Hashtable myDuelEnergyLostSF      = new Hashtable(); // By SandBoxFlattener_DistSeg - enemy performance
	public static Hashtable myDuelEnergyGainedSF    = new Hashtable(); // By SandBoxFlattener_DistSeg - enemy performance		
	public static Hashtable myDuelShotsSF           = new Hashtable(); // By SandBoxFlattener_DistSeg - enemy performance
	public static Hashtable myDuelHitsSF            = new Hashtable(); // By SandBoxFlattener_DistSeg - enemy performance
		
	public static Hashtable enemyAccidentalHits  = new Hashtable();    // GameSegment X NumGuns for targeting performance
	public static Hashtable consecutiveMisses    = new Hashtable();

	// Support for Virtual Gun tracking stats....
	public static Hashtable enemyVirtualHits        = new Hashtable();    // GameSegment X DistSeg X NumGuns for targeting performance
	public static Hashtable enemyVirtualShots       = new Hashtable();    // GameSegment X DistSeg X NumGuns for targeting performance
	public static Hashtable enemyVirtualHitRateRoll = new Hashtable();    // GameSegment X DistSeg X NumGuns for targeting performance
	
	// For Angular Gun...
	public static Hashtable meanAimFactorLeft    = new Hashtable();
	public static Hashtable meanAimFactorMiddle  = new Hashtable();
	public static Hashtable meanAimFactorRight   = new Hashtable();
	public static Hashtable meanAimFactor        = new Hashtable();
					
	public EnemyManager(AbstractDynaBot themyrobot) {
		myrobot = themyrobot;
		enemyList = new Hashtable();
	}

	/**
	 * what to do before each battle
	 */
	public void reset() {
		Enumeration en = enemyList.elements();
		Enemy e;
		// Reset each enemy
		while(en.hasMoreElements()) {
			e = (Enemy)en.nextElement();
			e.reset();
			Integer cm = new Integer(100);
			consecutiveMisses.put(e.name, cm);
			
			// Age the GuessFactors
			if(myrobot.getRoundNum() > 1) {
				ageGuessFactors(e.name, .75);
			}
		}
	}

	protected void ageGuessFactors(String enemyName, double ageRate) {
		Enemy e = (Enemy)enemyList.get(enemyName);
		double [][][][][]gf;
		gf = (double[][][][][])guessFactors.get(enemyName);
		for(int i=0;i<DynaBotConstants.GAMESTAGES;i++) {
			for(int j=0;j<DynaBotConstants.NUM_DIST_SEG;j++) {
				for(int k=0;k<DynaBotConstants.VELOCITY_INDEXES;k++) {
					for(int m=0;m<DynaBotConstants.VELOCITY_INDEXES;m++) {
						for(int n=0;n<DynaBotConstants.GUESSFACTORBINS;n++) {
							gf[i][j][k][m][n] *= ageRate;
						}
					}
				}
			}
		}
	}

	/**
	 * what to do each tick
	 */
	public void doWork() {
		Enemy newEnemy = pickCurrentEnemy();
		Enemy oldEnemy = currentEnemy;
		currentEnemy = newEnemy;
		if(currentEnemy != oldEnemy && currentEnemy != null) {
			myrobot.theGunManager.switchToFitestExcludingID(-1);
		}
		if(currentEnemy != null) {
			// Age GF roughly every 150 ticks
			if(((int)myrobot.getTime() % 150 ==0) && myrobot.getTime() > 0) {
				ageGuessFactors(currentEnemy.name, .9);
				myrobot.out.println("Aging guessfactors complete");
			}
		}
	}

	/**
	 * Tries to pick the best enemy to target:
	 *   If any disabled enemies exist, target them
	 *   Otherwise, target the closest enemy if it is
	 *   less than 90% the currentEnemy distance.
	 */
	protected Enemy pickCurrentEnemy() {
		Enemy closestEnemy = findClosestEnemy();
		Enemy oneShotKillEnemy = findOneShotKillEnemy();
		if(oneShotKillEnemy != null) return oneShotKillEnemy;
		if (currentEnemy == null) return closestEnemy;
		else {
			if(currentEnemy == closestEnemy) return currentEnemy;
			
			/*
			myrobot.out.println("closestEnemy is " + closestEnemy.name + " is " + closestEnemy.lastDistance +
								" away; currentEnemy is " + currentEnemy.name + " is " + currentEnemy.lastDistance);
			*/
			// TODO - also factor in Bearing direction and Gun direction
			double dGunBearing = myrobot.getGunHeading(); // Degrees
			double dEnemyBearing = Math.toDegrees(closestEnemy.absBearing);
			myrobot.out.println("CurEnemy: " + currentEnemy.name + " but found " + closestEnemy.name + " at " + dEnemyBearing + " Gun: " + dGunBearing);
			
			if(closestEnemy.lastDistance < .9 * currentEnemy.lastDistance){
				//myrobot.out.println("new enemy");
				return closestEnemy;
			} else {
				return currentEnemy;
			}
		}
	}

	public void enemyFired(double shotPower, String enemyName, Point2D.Double enemyShotFromLoc) {
		myrobot.theBattleManager.enemyFired(shotPower, enemyName, enemyShotFromLoc);
		
		// Update stats for Sandbox Flattener
		if(myrobot.getOthers() == 1) {
			// Add for Sandbox Flattener
			int []enemyShotsAtMeSF;
			double []enemyEnergyLostSF;
			double dist = myrobot.location.distance(enemyShotFromLoc);
			int distBin = (int)(Math.min(dist,1000) / 50.0);
			
			enemyShotsAtMeSF = (int[])enemyDuelShotsAtMeSF.get(enemyName);
			enemyEnergyLostSF = (double[])enemyDuelEnergyLostSF.get(enemyName);
			
			enemyShotsAtMeSF[distBin]++;
			enemyEnergyLostSF[distBin] += shotPower;
			
			enemyDuelShotsAtMeSF.put(enemyName, enemyShotsAtMeSF);
			enemyDuelEnergyLostSF.put(enemyName, enemyEnergyLostSF);
		}
	}

	protected Enemy findClosestEnemy() {
		Enumeration en = enemyList.elements();
		Enemy e = null;
		Enemy retval = null;
		double closestDist = Double.POSITIVE_INFINITY;
		// Reset each enemy
		while(en.hasMoreElements()) {
			e = (Enemy)en.nextElement();
			if(e.isAlive && e.lastDistance < closestDist) {
				retval = e;
				closestDist = e.lastDistance;
			}
		}
		return retval;
	}

	protected Enemy findOneShotKillEnemy() {
		Enumeration en = enemyList.elements();
		Enemy e;
		// Reset each enemy
		while(en.hasMoreElements()) {
			e = (Enemy)en.nextElement();
			if(e.isAlive && e.energy < .01) return e;
		}
		return null;
	}

	public Enemy findLeastRecentlyScannedEnemy(String excludeName) {
		Enumeration en = enemyList.elements();
		Enemy e = null;
		Enemy retval = null;
		double timeSinceLastScan = myrobot.getTime();
		// Reset each enemy
		while(en.hasMoreElements()) {
			e = (Enemy)en.nextElement();
			if(e.isAlive && excludeName.indexOf(e.name + ",") == -1 && e.lastTimePosition < timeSinceLastScan) {
				retval = e;
				timeSinceLastScan = e.lastTimePosition;
			}
		}
		return retval;
	}

	/**
	 * onScannedRobot: Update the Enemy
	 */
	public void onScannedRobot(ScannedRobotEvent e) {
		String enemyName = e.getName();
		Enemy scannedEnemy = (Enemy)enemyList.get(enemyName);
		// We may not have this enemy yet.
		if(scannedEnemy == null) {
			scannedEnemy = new Enemy(myrobot, enemyName);
			addNewEnemy(enemyName);
		}
		// Update the data about this enemy
		scannedEnemy.updateFromScannedRobotEvent(e);
		enemyList.put(enemyName, scannedEnemy);
	}

	protected void addNewEnemy(String enemyName) {
		if(!enemyTargettedHits.containsKey(enemyName)) {
			int [][][]timesHit              = new int[DynaBotConstants.GAMESTAGES][DynaBotConstants.NUM_DIST_SEG][DynaBotConstants.NUMGUNS];
			int [][][]timesFiredAt          = new int[DynaBotConstants.GAMESTAGES][DynaBotConstants.NUM_DIST_SEG][DynaBotConstants.NUMGUNS];

			int [][][]virtualHit            = new int[DynaBotConstants.GAMESTAGES][DynaBotConstants.NUM_DIST_SEG][DynaBotConstants.NUMGUNS];
			int [][][]virtualShots          = new int[DynaBotConstants.GAMESTAGES][DynaBotConstants.NUM_DIST_SEG][DynaBotConstants.NUMGUNS];
			double [][][]virtualHitRateRoll = new double[DynaBotConstants.GAMESTAGES][DynaBotConstants.NUM_DIST_SEG][DynaBotConstants.NUMGUNS];
			int [][]accidentalHits          = new int[DynaBotConstants.GAMESTAGES][DynaBotConstants.NUMGUNS];

			double [][]damageTo             = new double[DynaBotConstants.GAMESTAGES][DynaBotConstants.NUM_DIST_SEG];
			double [][]damageToMe           = new double[DynaBotConstants.GAMESTAGES][DynaBotConstants.NUM_DIST_SEG];
						
			// For GF Gun - GameSegment X DistSeg X VelocityIndex (cur) X Velocity Index (prev) X GuessFactorBins 
			double [][][][][]gf           = new double[DynaBotConstants.GAMESTAGES][DynaBotConstants.NUM_DIST_SEG][DynaBotConstants.VELOCITY_INDEXES][DynaBotConstants.VELOCITY_INDEXES][DynaBotConstants.GUESSFACTORBINS];

			enemyTargettedHits.put(enemyName, timesHit);
			enemyShotsFiredAt.put(enemyName, timesFiredAt);

			enemyVirtualHits.put(enemyName, virtualHit);
			enemyVirtualShots.put(enemyName, virtualShots);
			enemyVirtualHitRateRoll.put(enemyName, virtualHitRateRoll);
			
			enemyDamageToMe.put(enemyName, damageToMe);
			enemyDamageTo.put(enemyName, damageTo);
			enemyAccidentalHits.put(enemyName, accidentalHits);
			
			// For Guess Factors; initial them off to default to head-on
			for(int i=0;i<DynaBotConstants.GAMESTAGES;i++) {
				for(int j=0;j<DynaBotConstants.NUM_DIST_SEG;j++) {
					for(int k=0;k<DynaBotConstants.VELOCITY_INDEXES;k++) {
						for(int m=0;m<DynaBotConstants.VELOCITY_INDEXES;m++) {
							gf[i][j][k][m][DynaBotConstants.GUESSFACTORMIDDLE_BIN] = .5;
						}
					}
				}
			}
			guessFactors.put(enemyName, gf);
			
			// Add for Sandbox Flattener
			int []enemyShotsAtMeSF       = new int[DynaBotConstants.SF_NUM_SEGMENTS];
			int []enemyHitsSF            = new int[DynaBotConstants.SF_NUM_SEGMENTS];
			double []enemyEnergyGainSF   = new double[DynaBotConstants.SF_NUM_SEGMENTS];
			double []enemyEnergyLostSF   = new double[DynaBotConstants.SF_NUM_SEGMENTS];
			int []myShotsSF              = new int[DynaBotConstants.SF_NUM_SEGMENTS];
			int []myHitsSF               = new int[DynaBotConstants.SF_NUM_SEGMENTS];
			double []myEnergyGainSF      = new double[DynaBotConstants.SF_NUM_SEGMENTS];
			double []myEnergyLostSF      = new double[DynaBotConstants.SF_NUM_SEGMENTS];
			
			enemyDuelShotsAtMeSF.put(enemyName, enemyShotsAtMeSF);
			enemyDuelHitsSF.put(enemyName, enemyHitsSF);
			enemyDuelEnergyGainedSF.put(enemyName, enemyEnergyGainSF);
			enemyDuelEnergyLostSF.put(enemyName, enemyEnergyLostSF);
			
			myDuelEnergyLostSF.put(enemyName, myEnergyLostSF);
			myDuelEnergyGainedSF.put(enemyName, myEnergyGainSF);
			myDuelShotsSF.put(enemyName, myShotsSF);
			myDuelHitsSF.put(enemyName, myHitsSF);

			meanAimFactor.put(enemyName, new Double(0));
			meanAimFactorLeft.put(enemyName, new Double(0));
			meanAimFactorMiddle.put(enemyName, new Double(0));
			meanAimFactorRight.put(enemyName, new Double(0));
		}
	}

	/**
	 * onHitByBullet: What to do when you're hit by a bullet
	 */
	public void onHitByBullet(HitByBulletEvent e) {
		double energyGained = MyUtils.getShotEnergyGain(e.getPower());
		double damageCaused = MyUtils.getBulletDamage(e.getPower());
		String enemyName = e.getName();
		Enemy enemy = (Enemy)enemyList.get(enemyName);
		if(enemy == null) return;
		double lastDist = Math.min(enemy.lastDistance, 1000.0);
		enemy.lastHitMe = myrobot.getTime();
		
		double [][]damageToMe = (double[][])enemyDamageToMe.get(enemyName);
		int segmentBin = MyUtils.getDistanceBin(enemy.lastDistance);
		damageToMe[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin] += damageCaused;
		damageToMe[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL] += damageCaused;
		enemyDamageToMe.put(enemyName, damageToMe);
		
		// Find the enemy
		int distBin = (int)(lastDist / 50.0);
		
		// Add for Sandbox Flattener
		int []enemyHitsSF            = new int[DynaBotConstants.SF_NUM_SEGMENTS];
		double []enemyEnergyGainSF   = new double[DynaBotConstants.SF_NUM_SEGMENTS];
		double []myEnergyLostSF      = new double[DynaBotConstants.SF_NUM_SEGMENTS];
		
		if(enemyDuelHitsSF.containsKey(enemyName)) {
			enemyHitsSF = (int[])enemyDuelHitsSF.get(enemyName);
			enemyHitsSF[distBin]++;
		
			enemyEnergyGainSF = (double[])enemyDuelEnergyGainedSF.get(enemyName);
			enemyEnergyGainSF[distBin] += energyGained;
		
			myEnergyLostSF = (double[])myDuelEnergyLostSF.get(enemyName);
			myEnergyLostSF[distBin] += damageCaused;
						
			enemyDuelHitsSF.put(enemyName, enemyHitsSF);
			enemyDuelEnergyGainedSF.put(enemyName, enemyEnergyGainSF);
			myDuelEnergyLostSF.put(enemyName, myEnergyLostSF);
		}
	}

	public void onBulletHit(BulletHitEvent e) {
		if(currentEnemy != null) {
			Bullet b = e.getBullet();
			AdvancedBullet ab = myrobot.theBattleManager.getAdvancedBulletFromBullet(b);
			if(ab != null) {
				//myrobot.out.println("Found bullet");
				if(ab.enemyName == e.getName()) {
					String hitEnemy = e.getName();
					Integer cm = (Integer)consecutiveMisses.get(hitEnemy);
					cm = new Integer(0);
					consecutiveMisses.put(hitEnemy, cm);		
					scoreHit(ab);
				} else {
					myrobot.out.println("Accidental hit...");
					scoreAccidentalHit(ab, e);
				}
			} else {
				myrobot.out.println("Could not find bullet in onBulletHit");
			}
		}
	}

	public void onBulletHitBullet(BulletHitBulletEvent e) {
		// TODO - figure out how to count this
	}

	public void onBulletMissed(BulletMissedEvent e) {
		AdvancedBullet ab = myrobot.theBattleManager.getAdvancedBulletFromBullet(e.getBullet());
		if(ab != null) {
			int c;
			if(consecutiveMisses.containsKey(ab.enemyName)) {
				c = ((Integer)consecutiveMisses.get(ab.enemyName)).intValue() + 1;
			} else {
				c = 0;
			}
			if(c >= DynaBotConstants.CONSECUTIVE_MISS_THRESHOLD) {
				myrobot.theGunManager.switchToFitestExcludingID(ab.gunID);
				c = 0;
			}
			Integer cm = new Integer(c);
			consecutiveMisses.put(ab.enemyName, cm);
		} else {
			myrobot.out.println("Could not find bullet in onBulletMissed");
		}
	}

	public void onDeath(DeathEvent e) {
		printStats();
	}

	public void onHitRobot(HitRobotEvent e) {
	}

	public void onPaint(Graphics2D g) {
	}

	public void onRobotDeath(RobotDeathEvent event) {
		String deadRobot = event.getName();
		Enemy e = (Enemy)enemyList.get(deadRobot);
		e.isAlive = false;
		enemyList.put(deadRobot,e);
		currentEnemy = null;          // Make sure to get a new target
	}

	public void onWin(WinEvent e) {
		printStats();
	}

	public void printStats() {
		if(enemyList.size() < 5) {
			Enumeration en = enemyList.elements();
			while(en.hasMoreElements()) {
				Enemy e = (Enemy)en.nextElement();
				//StringBuffer fcPattern = (StringBuffer)funkyChickenPatterns.get(e.name);
				//myrobot.out.println("Enemy: " + e.name + " fcPattern.length= " + fcPattern.length());
				//for(int k=0;k<DynaBotConstants.GAMESTAGES;k++) {
					int k = DynaBotConstants.GAMESTAGE_ALLSTAGES;
					int i = DynaBotConstants.DIST_SEG_ALL;
					//for(int i=0;i<DynaBotConstants.NUM_DIST_SEG;i++) {
						for(int j=0;j<DynaBotConstants.NUMGUNS;j++) {
							int [][][]timesHit;
							int [][][]timesFiredAt;
							timesHit     = (int[][][])enemyTargettedHits.get(e.name);
							timesFiredAt = (int[][][])enemyShotsFiredAt.get(e.name);
							int [][][]vHits;
							int [][][]vShots;
							vHits = (int[][][])enemyVirtualHits.get(e.name);
							vShots = (int[][][])enemyVirtualShots.get(e.name);	
							if(timesFiredAt[k][i][j] != 0 || vShots[k][i][j] != 0) {
								myrobot.out.println("Enemy: " + e.name + " [" + k + "][" + i + "][" + j + "] = " 
										+ timesHit[k][i][j] + " / " + timesFiredAt[k][i][j] + " gun: " + myrobot.theGunManager.gunList[j].name
										+ " VG: " + vHits[k][i][j] + " / " + vShots[k][i][j]);
							}
						}
					//}
				//}
				int []hisShots = (int[])enemyDuelShotsAtMeSF.get(e.name);
				int []hisHits = (int[])enemyDuelHitsSF.get(e.name);
				int totalHits =0;
				int totalShots=0;
				for(int j=0;j<DynaBotConstants.SF_NUM_SEGMENTS;j++) {
					//myrobot.out.println("    Enemy: " + e.name + " [" + j + "] = " + hisHits[j] + " / " + hisShots[j]);				
					totalHits += hisHits[j];
					totalShots += hisShots[j];
				}
				myrobot.out.println(e.name + " Totals: " + totalHits + " / " + totalShots);
			}
		}		
	}

	public double getDuelOpponentHitRate() {
		if(currentEnemy==null) return .1;
		
		int []hisShots = (int[])enemyDuelShotsAtMeSF.get(currentEnemy.name);
		int []hisHits = (int[])enemyDuelHitsSF.get(currentEnemy.name);
		int totalHits =0;
		int totalShots=0;
		for(int j=0;j<DynaBotConstants.SF_NUM_SEGMENTS;j++) {
			totalHits += hisHits[j];
			totalShots += hisShots[j];
		}
		if(totalShots > 0) {
			return ((double)totalHits / (double)totalShots);
		} else {
			return .1;
		}
	}		
		
	public void doFireGun(double shotPower) {
		if(currentEnemy != null) {
			double dist = currentEnemy.lastDistance;
			int gameStage = MyUtils.getGameStage(myrobot.getOthers());
			int segmentBin = MyUtils.getDistanceBin(dist);
			int theGunID = myrobot.theGunManager.currentGun().gunID;
			int [][][]timesFiredAt = (int[][][])enemyShotsFiredAt.get(currentEnemy.name);
			//myrobot.out.println("firedAt: " + currentEnemy.name + " with gun " + myrobot.theGunManager.currentGun().name);
						
			timesFiredAt[gameStage][segmentBin][theGunID]++;                			// Update this distance
			timesFiredAt[gameStage][DynaBotConstants.DIST_SEG_ALL][theGunID]++;			// And all distances
			// Update this distance for all game stages
			timesFiredAt[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][theGunID]++;                			
			timesFiredAt[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][theGunID]++;
			
			enemyShotsFiredAt.put(currentEnemy.name, timesFiredAt);
			
			// Increase SF myEnergyGained, enemyEnergyLost, myHits
			if(myrobot.getOthers() == 1) {
				// only for duels
				// Add for Sandbox Flattener
				int []myShotsSF;
				double []myEnergyLostSF;
			
				int distBin = (int)(Math.min(dist, 1000) / 50.0);
						
				myShotsSF = (int[])myDuelShotsSF.get(currentEnemy.name);
				myEnergyLostSF = (double[])myDuelEnergyLostSF.get(currentEnemy.name);
			
				myShotsSF[distBin]++;
				myEnergyLostSF[distBin] += shotPower;
			
				myDuelShotsSF.put(currentEnemy.name, myShotsSF);
				myDuelEnergyLostSF.put(currentEnemy.name, myEnergyLostSF);
			}
		}
	}

	public void scoreHit(AdvancedBullet ab) {
		//myrobot.out.println("ScoredHit: " + ab.enemyName);
		Enemy hitEnemy = (Enemy)myrobot.theEnemyManager.enemyList.get(ab.enemyName);
		double dist = hitEnemy.lastDistance;
		int segmentBin = MyUtils.getDistanceBin(dist);
		int theGunID = ab.gunID;
		int gameStage = MyUtils.getGameStage(myrobot.getOthers());
		int [][][]timesHit = (int[][][])enemyTargettedHits.get(ab.enemyName);
								
		timesHit[gameStage][segmentBin][theGunID]++;                			// Update this distance
		timesHit[gameStage][DynaBotConstants.DIST_SEG_ALL][theGunID]++;			// And all distances
		// Update this distance for all game stages
		timesHit[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][theGunID]++;                			
		timesHit[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][theGunID]++;
			
		enemyTargettedHits.put(ab.enemyName, timesHit);

		// TODO increase damageTo
		double energyGained = MyUtils.getShotEnergyGain(ab.b.getPower());
		double damageCaused = MyUtils.getBulletDamage(ab.b.getPower());
		
		double [][]damage = (double[][])enemyDamageTo.get(hitEnemy.name);
		damage[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin] += damageCaused;
		damage[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL] += damageCaused;
		enemyDamageTo.put(hitEnemy.name, damage);
						
		// TODO increase energy gained
		
		// Increase SF myEnergyGained, enemyEnergyLost, myHits
		if(myrobot.getOthers() == 1) {
			// only for duels
			// Add for Sandbox Flattener
			int []myHitsSF;
			double []enemyEnergyLostSF;
			double []myEnergyGainedSF;
			
			int distBin = (int)(Math.min(hitEnemy.lastDistance, 1000) / 50.0);
						
			enemyEnergyLostSF = (double[])enemyDuelEnergyLostSF.get(hitEnemy.name);
			myHitsSF = (int[])myDuelHitsSF.get(hitEnemy.name);
			myEnergyGainedSF = (double[])myDuelEnergyGainedSF.get(hitEnemy.name);
			
			myHitsSF[distBin]++;
			enemyEnergyLostSF[distBin] += damageCaused;
			myEnergyGainedSF[distBin] += energyGained;
			
			enemyDuelEnergyLostSF.put(hitEnemy.name, enemyEnergyLostSF);
			myDuelHitsSF.put(hitEnemy.name, myHitsSF);
			myDuelEnergyGainedSF.put(hitEnemy, myEnergyGainedSF);
		}
	}

	public void scoreAccidentalHit(AdvancedBullet ab, BulletHitEvent e) {
		//myrobot.out.println("ScoredAccidentalHit: was for " + ab.enemyName + " but hit " + e.getName());
		Enemy hitEnemy = (Enemy)myrobot.theEnemyManager.enemyList.get(e.getName());
		Enemy targettedEnemy = (Enemy)myrobot.theEnemyManager.enemyList.get(ab.enemyName);
								
		double dist = currentEnemy.lastDistance;
		int segmentBin = MyUtils.getDistanceBin(dist);
		int theGunID = ab.gunID;
		int gameStage = MyUtils.getGameStage(myrobot.getOthers());
				
		// TODO - update the accidentally hit enemy
		// TODO increase damageTo
				
		// TODO - decrement the firedAt count for the targetted enemy if target is farther than hit enemy
		//int [][][]timesFiredAt = (int[][][])enemyShotsFiredAt.get(ab.enemyName);
		
	}
	/********************************************************************************
	 *                        For Virtual Gun Tracking                              *
	 *******************************************************************************/
	public void fireVirtualShots(String enemyName, double dist) {
		int segmentBin = MyUtils.getDistanceBin(dist);
		int gameStage = MyUtils.getGameStage(myrobot.getOthers());
		int [][][]virtualShots = (int[][][])enemyVirtualShots.get(enemyName);
		
		// int[DynaBotConstants.GAMESTAGES][DynaBotConstants.NUM_DIST_SEG][DynaBotConstants.NUMGUNS];
		for(int i=0;i<DynaBotConstants.NUMGUNS;i++) {
			virtualShots[gameStage][segmentBin][i]++;
			virtualShots[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][i]++;
			virtualShots[gameStage][DynaBotConstants.DIST_SEG_ALL][i]++;
			virtualShots[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][i]++;
		}
	
		enemyVirtualShots.put(enemyName, virtualShots);
	}

	public void recordVirtualHit(String enemyName, double dist, int gunID) {
		int segmentBin = MyUtils.getDistanceBin(dist);
		int gameStage = MyUtils.getGameStage(myrobot.getOthers());
		int [][][]virtualHits = (int[][][])enemyVirtualHits.get(enemyName);
		double [][][]virtualHitRate = (double[][][])enemyVirtualHitRateRoll.get(enemyName);
		
		virtualHits[gameStage][segmentBin][gunID]++;
		virtualHits[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][gunID]++;
		virtualHits[gameStage][DynaBotConstants.DIST_SEG_ALL][gunID]++;
		virtualHits[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][gunID]++;
	
		virtualHitRate[gameStage][segmentBin][gunID] = MyUtils.rollingAvg(virtualHitRate[gameStage][segmentBin][gunID], 1.0, 75, 25);
		virtualHitRate[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][gunID] = MyUtils.rollingAvg(virtualHitRate[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][gunID], 1.0, 75, 25);
		virtualHitRate[gameStage][DynaBotConstants.DIST_SEG_ALL][gunID] = MyUtils.rollingAvg(virtualHitRate[gameStage][DynaBotConstants.DIST_SEG_ALL][gunID], 1.0, 75, 25);
		virtualHitRate[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][gunID] = MyUtils.rollingAvg(virtualHitRate[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][gunID], 1.0, 75, 25);
			
		enemyVirtualHits.put(enemyName, virtualHits);
		enemyVirtualHitRateRoll.put(enemyName, virtualHitRate);
	}

	public void recordVirtualMiss(String enemyName, double dist, int gunID) {
		int segmentBin = MyUtils.getDistanceBin(dist);
		int gameStage = MyUtils.getGameStage(myrobot.getOthers());
		double [][][]virtualHitRate = (double[][][])enemyVirtualHitRateRoll.get(enemyName);

		// int[DynaBotConstants.GAMESTAGES][DynaBotConstants.NUM_DIST_SEG][DynaBotConstants.NUMGUNS];
		virtualHitRate[gameStage][segmentBin][gunID] = MyUtils.rollingAvg(virtualHitRate[gameStage][segmentBin][gunID], 1.0, 75, 25);
		virtualHitRate[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][gunID] = MyUtils.rollingAvg(virtualHitRate[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][gunID], 1.0, 75, 25);
		virtualHitRate[gameStage][DynaBotConstants.DIST_SEG_ALL][gunID] = MyUtils.rollingAvg(virtualHitRate[gameStage][DynaBotConstants.DIST_SEG_ALL][gunID], 1.0, 75, 25);
		virtualHitRate[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][gunID] = MyUtils.rollingAvg(virtualHitRate[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][gunID], 1.0, 75, 25);
			
		enemyVirtualHitRateRoll.put(enemyName, virtualHitRate);
	}
		
	/********************************************************************************
	 *                        For Gun Fitness                                      *
	 *******************************************************************************/
	public double getVirtualHitRate(int theGunID) {
		if(currentEnemy == null) return 0;
		else return getVirtualHitRateRolling(currentEnemy.name, theGunID);
	}

	public double getVirtualHitRate(String enemyName, int theGunID) {
		Enemy e = (Enemy)myrobot.theEnemyManager.enemyList.get(enemyName);
		double dist = e.lastDistance;
		int segmentBin = MyUtils.getDistanceBin(dist);
		int gameStage = MyUtils.getGameStage(myrobot.getOthers());
		int [][][]timesFiredAt = (int[][][])enemyVirtualShots.get(enemyName);
		int [][][]timesHit = (int[][][])enemyVirtualHits.get(enemyName);
		
		// Check current GameStage and Distance bin
		if(timesFiredAt[gameStage][segmentBin][theGunID] >= DynaBotConstants.MIN_NUM_SHOT) {
			return ((double)timesHit[gameStage][segmentBin][theGunID]) / ((double)timesFiredAt[gameStage][segmentBin][theGunID]) ;
		} else {
			// Check all Distances then all GameStages
			if(timesFiredAt[gameStage][DynaBotConstants.DIST_SEG_ALL][theGunID] >= DynaBotConstants.MIN_NUM_SHOT) {
				return ((double)timesHit[gameStage][DynaBotConstants.DIST_SEG_ALL][theGunID]) / ((double)timesFiredAt[gameStage][DynaBotConstants.DIST_SEG_ALL][theGunID]) ;
			} else {
				// Check all GameStages for this distance
				if(timesFiredAt[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][theGunID] >= DynaBotConstants.MIN_NUM_SHOT) {
					return ((double)timesHit[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][theGunID]) / ((double)timesFiredAt[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][theGunID]) ;
				} else {
					// Check all GameStages and all distances
					if(timesFiredAt[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][theGunID] >= DynaBotConstants.MIN_NUM_SHOT) {
						return ((double)timesHit[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][theGunID]) / ((double)timesFiredAt[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][theGunID]) ;
					}
				}
			}
		}
		return .99999; // Something unreal
	}
	
	public double getVirtualHitRateRolling(int theGunID) {
		if(currentEnemy == null) return 0;
		else return getVirtualHitRateRolling(currentEnemy.name, theGunID);
	}
		
	// Switch over to rolling average....
	public double getVirtualHitRateRolling(String enemyName, int theGunID) {
		Enemy e = (Enemy)myrobot.theEnemyManager.enemyList.get(enemyName);
		double dist = e.lastDistance;
		int segmentBin = MyUtils.getDistanceBin(dist);
		int gameStage = MyUtils.getGameStage(myrobot.getOthers());
		int [][][]timesFiredAt = (int[][][])enemyVirtualShots.get(enemyName);
		double [][][]rollingHitRate = (double[][][])enemyVirtualHitRateRoll.get(enemyName);
		
		// Check current GameStage and Distance bin
		if(timesFiredAt[gameStage][segmentBin][theGunID] >= DynaBotConstants.MIN_NUM_SHOT) {
			return rollingHitRate[gameStage][segmentBin][theGunID];
		} else {
			// Check all Distances then all GameStages
			if(timesFiredAt[gameStage][DynaBotConstants.DIST_SEG_ALL][theGunID] >= DynaBotConstants.MIN_NUM_SHOT) {
				return rollingHitRate[gameStage][DynaBotConstants.DIST_SEG_ALL][theGunID];
			} else {
				// Check all GameStages for this distance
				if(timesFiredAt[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][theGunID] >= DynaBotConstants.MIN_NUM_SHOT) {
					return rollingHitRate[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][theGunID];
				} else {
					// Check all GameStages and all distances
					if(timesFiredAt[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][theGunID] >= DynaBotConstants.MIN_NUM_SHOT) {
						return rollingHitRate[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][theGunID];
					}
				}
			}
		}
		return .999; // Something unreal
	}

	public double getHitRate(int theGunID) {
		if(currentEnemy == null) return 0;
		else return getHitRate(currentEnemy.name, theGunID);
	}

	public double getHitRate(String enemyName, int theGunID) {
		Enemy e = (Enemy)myrobot.theEnemyManager.enemyList.get(enemyName);
		double dist = e.lastDistance;
		int segmentBin = MyUtils.getDistanceBin(dist);
		int gameStage = MyUtils.getGameStage(myrobot.getOthers());
		int [][][]timesFiredAt = (int[][][])enemyShotsFiredAt.get(enemyName);
		int [][][]timesHit = (int[][][])enemyTargettedHits.get(enemyName);
		
		// Check current GameStage and Distance bin
		if(timesFiredAt[gameStage][segmentBin][theGunID] >= DynaBotConstants.MIN_NUM_SHOT) {
			return ((double)timesHit[gameStage][segmentBin][theGunID]) / ((double)timesFiredAt[gameStage][segmentBin][theGunID]) ;
		} else {
			// Check all Distances then all GameStages
			if(timesFiredAt[gameStage][DynaBotConstants.DIST_SEG_ALL][theGunID] >= DynaBotConstants.MIN_NUM_SHOT) {
				return ((double)timesHit[gameStage][DynaBotConstants.DIST_SEG_ALL][theGunID]) / ((double)timesFiredAt[gameStage][DynaBotConstants.DIST_SEG_ALL][theGunID]) ;
			} else {
				// Check all GameStages for this distance
				if(timesFiredAt[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][theGunID] >= DynaBotConstants.MIN_NUM_SHOT) {
					return ((double)timesHit[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][theGunID]) / ((double)timesFiredAt[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][theGunID]) ;
				} else {
					// Check all GameStages and all distances
					if(timesFiredAt[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][theGunID] >= DynaBotConstants.MIN_NUM_SHOT) {
						return ((double)timesHit[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][theGunID]) / ((double)timesFiredAt[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][theGunID]) ;
					}
				}
			}
		}
		return .75; // Don't bother switching guns if better than 75% hit rate
	}		

	public int getShotsFired(int theGunID) {
		if(currentEnemy == null) return 0;
		else return getShotsFired(currentEnemy.name, theGunID);
	}

	public int getShotsFired(String enemyName, int theGunID) {
		Enemy e = (Enemy)myrobot.theEnemyManager.enemyList.get(enemyName);
		double dist = e.lastDistance;
		int segmentBin = MyUtils.getDistanceBin(dist);
		int gameStage = MyUtils.getGameStage(myrobot.getOthers());
		int [][][]timesFiredAt = (int[][][])enemyShotsFiredAt.get(enemyName);
		
		// Check current GameStage and Distance bin
		if(timesFiredAt[gameStage][segmentBin][theGunID] >=  DynaBotConstants.MIN_NUM_SHOT) {
			return timesFiredAt[gameStage][segmentBin][theGunID];
		} else if(timesFiredAt[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][theGunID] >=  DynaBotConstants.MIN_NUM_SHOT) {
			return timesFiredAt[DynaBotConstants.GAMESTAGE_ALLSTAGES][segmentBin][theGunID];
		} else if (timesFiredAt[gameStage][DynaBotConstants.DIST_SEG_ALL][theGunID] >=  DynaBotConstants.MIN_NUM_SHOT) {
			return timesFiredAt[gameStage][DynaBotConstants.DIST_SEG_ALL][theGunID];
		} else {
			return timesFiredAt[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][theGunID];
		}
	}		

	/********************************************************************************
	 *                        For GuessFactor                                       *
	 *******************************************************************************/
	
	public void waveHitEnemy(String enemyName, EnemyWave ew) {
		// For GF Gun - GameSegment X DistSeg X VelocityIndex (cur) X Velocity Index (prev) X GuessFactorBins 
		Enemy e = (Enemy)enemyList.get(enemyName);
		double [][][][][]gf;
		if(guessFactors.containsKey(enemyName)) {
			gf = (double[][][][][])guessFactors.get(enemyName);
		} else {
			gf = new double[DynaBotConstants.GAMESTAGES][DynaBotConstants.NUM_DIST_SEG][DynaBotConstants.VELOCITY_INDEXES][DynaBotConstants.VELOCITY_INDEXES][DynaBotConstants.GUESSFACTORBINS];
			guessFactors.put(enemyName, gf);
		}
		if(!ew.lastDistances.containsKey(enemyName)) {
			return;
		}
		double lastDist   = ((Double)ew.lastDistances.get(enemyName)).doubleValue();
		double lastVel    = ((Double)ew.prevVelocity.get(enemyName)).doubleValue();
		double curVel     = ((Double)ew.curVelocity.get(enemyName)).doubleValue();
		double oriBearing = ((Double)ew.directAngles.get(enemyName)).doubleValue();
		int gameStage = MyUtils.getGameStage(myrobot.getOthers());
		int distSeg = MyUtils.getDistanceBin(lastDist);
		//int curVelIndex  = MyUtils.getVelocityIndex(curVel);
		//int prevVelIndex = MyUtils.getVelocityIndex(lastVel);
		int curVelIndex  = 0;
		int prevVelIndex = 0;
		int gfIndex = ew.getFactorIndex(e.location , enemyName);
		
		// Update GuessFactors
		for(int i=0;i<DynaBotConstants.GUESSFACTORBINS;i++) {
            // for the spot bin that we were hit on, add 1;
            // for the bins next to it, add 1 / 2;
            // the next one, add 1 / 5; and so on...
	        gf[gameStage][distSeg][curVelIndex][prevVelIndex][i] += 1.0 / (Math.pow(gfIndex - i, 2) + 1);
	        gf[DynaBotConstants.GAMESTAGE_ALLSTAGES][distSeg][curVelIndex][prevVelIndex][i] += 1.0 / (Math.pow(gfIndex - i, 2) + 1);
	        gf[gameStage][DynaBotConstants.DIST_SEG_ALL][curVelIndex][prevVelIndex][i] += 1.0 / (Math.pow(gfIndex - i, 2) + 1);
	        gf[DynaBotConstants.GAMESTAGE_ALLSTAGES][DynaBotConstants.DIST_SEG_ALL][curVelIndex][prevVelIndex][i] += 1.0 / (Math.pow(gfIndex - i, 2) + 1);
		}
	
		guessFactors.put(enemyName, gf);
		
		// Update for Angular Gun as well
		double bulletPower = ew.shotPower;
		double meanOffsetFactor = ((Double)myrobot.theEnemyManager.meanAimFactor.get(enemyName)).doubleValue();
		double meanAFL = ((Double)myrobot.theEnemyManager.meanAimFactorLeft.get(enemyName)).doubleValue();
		double meanAFM = ((Double)myrobot.theEnemyManager.meanAimFactorMiddle.get(enemyName)).doubleValue();
		double meanAFR = ((Double)myrobot.theEnemyManager.meanAimFactorRight.get(enemyName)).doubleValue();		
		double impactBearing = MyUtils.absoluteBearing(ew.fireLocation, e.location);
		double bearingDiff = Utils.normalRelativeAngle(impactBearing - oriBearing);
		
		meanOffsetFactor = MyUtils.rollingAvg(meanOffsetFactor, bearingDiff, 20, bulletPower);
		if(Math.abs(bearingDiff) > 0.05) {
			double factor = bearingDiff / e.deltaBearing;
			if(e.deltaBearing < -0.3) {
				meanAFL = MyUtils.rollingAvg(meanAFL, factor, 75, bulletPower);
			} else if (e.deltaBearing > 0.3) {
				meanAFR = MyUtils.rollingAvg(meanAFR, factor, 75, bulletPower);
			} else {
				meanAFM = MyUtils.rollingAvg(meanAFM, factor, 75, bulletPower);
			}
		}
		myrobot.theEnemyManager.meanAimFactor.put(enemyName, new Double(meanOffsetFactor));
		myrobot.theEnemyManager.meanAimFactorLeft.put(enemyName, new Double(meanAFL));
		myrobot.theEnemyManager.meanAimFactorMiddle.put(enemyName, new Double(meanAFM));
		myrobot.theEnemyManager.meanAimFactorRight.put(enemyName, new Double(meanAFR));				
	}
}
