/*
 * Created on 2004-9-25
 */

package tide;

import java.io.IOException;
import java.io.Serializable;

import robocode.Bullet;
import tide.util.MathUtils;
import tide.util.Utils;


/**
 * robot console, you can get the robot information and 
 * control your robot via this console.
 * all angle is in radians, all method is return immediately.
 * 
 * @author xiemin
 */
public class RobotConsole
{
	//the condition uesed by execute the fire operate
	private class FireCondition extends robocode.Condition
	{
		//the heading you want to fire
		private double heading; 
		//the power you want to fire
		private double power;
		
		public FireCondition(double heading, double power)
		{
			this.heading = heading;
			this.power = power;
		}
		
		public boolean test()
		{
			double bearing = MathUtils.bearing(heading, getGunHeading());
			turnGun(bearing);
			if( Math.abs(bearing)<0.00001 )
			{
				//execute fire, and remove this condition
				robot.removeCustomEvent(this);
				turnGun(0);
				robot.setFire(power);
			}
			return false;
		}
	}
	
	//a big double value, make sure robot can move or turn a long time
	private static final double BIG_VALE = 1000000;
	//the robot be operrated
	private robocode.TeamRobot robot;
	
	/**
	 * reset this console. because the robot object will be 
	 * re-created per round by robocode, so the console will be 
	 * reset when the robot be created.
	 */
	public void reset(robocode.TeamRobot robot)
	{
		this.robot = robot;
	}
	
	/////////////////////////////////////////////////////////////
	//body
	/**
	 * set the move velocity of the robot, since you call this method, 
	 * the robot will move in this velocity until it hit wall or others,
	 * or you call this method again.
	 * 
	 * @param velocity the velocity you want the robot to move
	 */
	public void move(double velocity)
	{
		robot.setMaxVelocity(Math.abs(velocity));
		if( velocity>0 ) robot.setAhead(BIG_VALE);
		else robot.setAhead(-BIG_VALE);
	}

	/**
	 * set the turn velocity of the robot, since you call this method, 
	 * the robot will turn in this velocity until you call this method 
	 * again.
	 * 
	 * @param velocity the velocity you want the robot to turn
	 */
	public void turn(double velocity)
	{
		robot.setMaxTurnRate(Math.toDegrees(Math.abs(velocity)));
		if( velocity>0 ) robot.setTurnLeftRadians(BIG_VALE);
		else robot.setTurnLeftRadians(-BIG_VALE);
	}
	
	public robocode.TeamRobot getRobocodeRobot(){
		return robot;
	}
	
	/**
	 * Return the direction the robot is facing.
	 * @return heading 0<=heading<2*PI.
	 */
	public double getHeading()
	{
		return Utils.transform( robot.getHeadingRadians());
	}

	/**
	 * Return the velocity of the robot.
	 * @return velocity the actually velocity the robot move.
	 */
	public double getVelocity()
	{
		return robot.getVelocity();
	}
	
	/**
	 * Return the X position of the robot.
	 * @return the X position of the robot.
	 */
	public double getX()
	{
		return robot.getX();
	}

	/**
	 * Return the Y position of the robot.
	 * @return the Y position of the robot.
	 */
	public double getY()
	{
		return robot.getY();
	}
	
	//////////////////////////////////////////////////////////
	//gun	
	/**
	 * execute the fire operate. if last fire operate has not finish,
	 * this operate is no actually execute. is may be cost several tick
	 * to turn the gun to the heading you want to fire, so it may be several
	 * tick behind you call this method when the bullet is actually shooted.
	 *  
	 * @param heading the heading you want to fire
	 * @param power the power you want to fire
	 */
	public void fire(double heading, double power)
	{
		if( robot.getGunTurnRemaining() == 0 )
			robot.addCustomEvent(new FireCondition(heading, power));
	}
	
	/**
	 * execute the fire operate. 
	 * is as same as robocode.AdvancedRobot.setFire().
	 */
	public void fire(double power)
	{
		Bullet bullet = robot.setFireBullet(power);
		if(bullet != null){
			//Pear.log("fire bullet--------");
			//Pear.log("bullet : " + new Point2D.Double(bullet.getX(), bullet.getY()));
			//Pear.log("My pos : " + new Point2D.Double(robot.getX(), robot.getY()));
		}
	}
	
	/**
	 * rotates your robot's gun.
	 * @param d the angle you want to rotate the gun.
	 */
	public void turnGun(double d)
	{
		robot.setTurnGunLeftRadians(d);
	}

	/**
	 * Return gun heading.
	 * @return gun heading.
	 */
	public double getGunHeading()
	{
		return Utils.transform( robot.getGunHeadingRadians());
	}
	
	/**
	 * get the prepare time for fire, You cannot fire unless this is 0. 
	 * (Calls to fire will succeed, but will not actually fire unless 
	 * getFirePrepareTime()==0.
	 * 
	 * @return time the prepare time for fire
	 */
	public long getFirePrepareTime()
	{
		return (long)(Math.ceil(robot.getGunHeat()/robot.getGunCoolingRate()));
	}
	
	/////////////////////////////////////////////////////////
	//radar
	/**
	 * rotates your robot's radar.
	 * @param d the angle you want to rotate the radar.
	 */
	public void turnRader(double d)
	{
		robot.setTurnRadarLeftRadians(d);
	}
	
	/**
	 * get radar heading.
	 * @return radar heading.
	 */
	public double getRadarHeading()
	{
		return Utils.transform(robot.getRadarHeadingRadians());
	}
	
	/////////////////////////////////////////////////////////
	/**
	 * print information to the console.
	 * @param info information you want to print
	 */
	public void println(String info)
	{
		robot.out.println(info);
	}
	
	/**
	 * Returns the number of the current round in the battle.
	 * @return number of the current round (0 to getTotalRounds()-1). 
	 */
	public int getCurrentRound()
	{
		return robot.getRoundNum();
	}

	/**
	 * Returns the robot's current energy.
	 * @return current energy
	 */
	public double getEnergy()
	{
		return robot.getEnergy();
	}

	/**
	 * Returns the robot's name.
	 * @return robot's name.
	 */
	public String getName()
	{
		return robot.getName();
	}

	/**
	 * Returns the current time.
	 * @return current time.
	 */
	public long getTime()
	{
		return robot.getTime();
	}

	/**
	 * Returns the total rounds in the battle.
	 * @return total rounds in the battle.
	 */
	public int getTotalRounds()
	{
		return robot.getNumRounds();
	}

	/**
	 * Return the width of the current battlefield.
	 * @return the width of the current battlefield.
	 */
	public double getMapWidth()
	{
		return robot.getBattleFieldWidth();
	}
	
	/**
	 * Return the height of the current battlefield.
	 * @return the height of the current battlefield.
	 */
	public double getMapHeight()
	{
		return robot.getBattleFieldHeight();
	}
	
	/**
	 * Returns how many opponents are left
	 * @return how many opponents are left
	 */
	public int getOthers()
	{
		return robot.getOthers();
	}
	
	/////////////////////////////////////////////////////////////
	//for team
	
	/**
	 * Checks if a given robot name is one of your teammates.
	 * @param name the robot name
	 */
	public boolean isTeammate(String name)
	{
		return robot.isTeammate(name);
	}
	/**
	 * Return all teammates name.
	 * @return all teammates name.
	 */
	public String[] getTeammates()
	{
		return robot.getTeammates();
	}
	
	/**
	 * Broadcasts a message to all teammates.
	 * @param message the message you want to broadcast.
	 * @throws IOException
	 */
	public void broadcastMessage(Serializable message) throws IOException
	{
		robot.broadcastMessage(message);
	}
	
	/**
	 * Sends a message to one (or more) teammates.
	 * @param name
	 * @param message
	 * @throws IOException
	 */
	public void sendMessage(String name, Serializable message) throws IOException
	{
		robot.sendMessage(name, message);
	}
	
}












