package kc.micro;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

import robocode.AdvancedRobot;
import robocode.BulletHitEvent;
import robocode.Condition;
import robocode.HitByBulletEvent;
import robocode.Rules;
import robocode.ScannedRobotEvent;
import robocode.StatusEvent;
import robocode.util.Utils;

/**
* WaveShark, by Kevin Clark (Kev).
* A micro-sized bot with wave surfing.
* See https://robowiki.net/wiki/WaveShark for more information.
*/

public class WaveShark extends AdvancedRobot {
  // Movement constants
  private static final int HIT_BINS = 95;
  private static final int MIDDLE_BIN = (HIT_BINS - 1) / 2;
  private static final int BIN_SMOOTHING_FACTOR = 3;
  private static final double STICK_LENGTH = 160;

  // Gun constants
  private static final int MAX_BFT = 70;
  private static final int MAX_MATCH_LENGTH = 50;
  private static final double WALL_FEATURE = 0.5;
  private static final double ANTI_BULLET_SHIELD_FACTOR = 0.0005;

  // Bullet power constants
  private static final int BULLET_POWER = 2;
  private static final double BULLET_DAMAGE = 10;
  private static final int BULLET_SPEED = 14;
  private static final int MIN_FIRE_ENERGY = 2;
  private static final int FULL_POWER_DISTANCE = 120;

  // Game state
  private static double myX;
  private static double myY;
  private static double lastVelocity;
  private static double lastLastVelocity;
  private static double enemyEnergy;

  // Wave surfing
  private static Wave surfWave;
  private static double[][] hits = new double[5][HIT_BINS];

  @Override
  public void run() {
    setAdjustRadarForGunTurn(true);
    setAdjustGunForRobotTurn(true);
    enemyEnergy = 102;  // create an initial power-2 wave so we start moving immediately
  }

  @Override
  public void onStatus(StatusEvent e) {
    setTurnRadarRightRadians(1);
  }

  @Override
  public void onScannedRobot(ScannedRobotEvent e) {
    // Register variables
    int dir_matchLen_bulletDist = -1;
    double energyDrop_absBearing_aimAngle;

    // Other variables
    int matchPos;
    double distance;
    double targetHeading;
    double risk;
    double maxRisk = Double.POSITIVE_INFINITY;
    Point2D.Double projectedLocation;

    // Create a new enemy wave on energy drop
    Wave w = new Wave();
    if ((energyDrop_absBearing_aimAngle = enemyEnergy - (enemyEnergy = e.getEnergy())) > 0) {
      addCustomEvent(w);
      w.speed = Rules.getBulletSpeed(energyDrop_absBearing_aimAngle);
      // use lastLastVelocity because our opponent aims 2 ticks before we see the energy drop
      w.hits = hits[(int)(Math.abs(lastLastVelocity)) >> 1];
      w.orbitDirection = Math.copySign(MIDDLE_BIN, lastLastVelocity);
    }
    lastLastVelocity = lastVelocity;

    // Update bot locations
    myX = getX();
    myY = getY();
    w.source = project(energyDrop_absBearing_aimAngle = w.absoluteBearingPlusPi =
        getHeadingRadians() + e.getBearingRadians(), distance = e.getDistance());

    // Radar lock
    setTurnRadarRightRadians(2 * Utils.normalRelativeAngle(
        energyDrop_absBearing_aimAngle - getRadarHeadingRadians()));

    // Predict going forward or backward until closest wave hits and pick the least dangerous option
    try {
      while (dir_matchLen_bulletDist <= 1) {
        targetHeading = energyDrop_absBearing_aimAngle + Math.toRadians(
            TARGET_HEADING_TABLE.charAt(3 * (int)distance + dir_matchLen_bulletDist));
        while (fieldContains(targetHeading -= (dir_matchLen_bulletDist * 0.01),
            STICK_LENGTH * dir_matchLen_bulletDist) > 0);
        if ((risk =
            (1 + surfWave.hits[surfWave.hitBin(
                Math.atan2(surfWave.source.x - (projectedLocation = project(
                    targetHeading,
                    dir_matchLen_bulletDist * (short)MOVE_DISTANCE_TABLE.charAt(
                      ((surfWave.ticksUntilBreak + 3) << 5) +
                      (dir_matchLen_bulletDist * (int)(lastVelocity = getVelocity()))))).x,
                    surfWave.source.y - projectedLocation.y))]) /
            projectedLocation.distanceSq(w.source)) < maxRisk) {
          maxRisk = risk;
          setAhead(dir_matchLen_bulletDist << 8);
          setTurnRightRadians(Utils.normalRelativeAngle(targetHeading - getHeadingRadians()));
        }
        dir_matchLen_bulletDist += 2;
      }
    } catch (Exception ex) {}

    // Update log of enemy movements and pattern match
    enemyHistory = String.valueOf(
        (char)(
            ((int)(e.getVelocity() *
                Math.sin(e.getHeadingRadians() - energyDrop_absBearing_aimAngle)) << 2)
            | (fieldContains(energyDrop_absBearing_aimAngle + WALL_FEATURE, distance) << 1)
            | fieldContains(energyDrop_absBearing_aimAngle - WALL_FEATURE, distance)
          )
        ).concat(enemyHistory);
    dir_matchLen_bulletDist = MAX_MATCH_LENGTH;
    while ((matchPos = enemyHistory.indexOf(
        enemyHistory.substring(0, (--dir_matchLen_bulletDist)), MAX_BFT)) < 0);

    // Fire and aim by playing forward enemy movements
    if ((int)(getEnergy()) > MIN_FIRE_ENERGY) {
      setFire(BULLET_POWER + FULL_POWER_DISTANCE / (dir_matchLen_bulletDist = (int)distance));
    }
    do {
       energyDrop_absBearing_aimAngle += ((short)enemyHistory.charAt(--matchPos) >> 2) / distance;
    } while ((dir_matchLen_bulletDist -= BULLET_SPEED) > 0);
    setTurnGunRightRadians(Utils.normalRelativeAngle(
        ANTI_BULLET_SHIELD_FACTOR + energyDrop_absBearing_aimAngle - getGunHeadingRadians()));

    // Each tick we re-select the oldest active wave to surf
    surfWave = null;
  }

  @Override
  public void onBulletHit(BulletHitEvent e) {
    enemyEnergy -= BULLET_DAMAGE;
  }

  @Override
  public void onHitByBullet(HitByBulletEvent e) {
    try {
      int i = 0;
      while(true) {
        surfWave.hits[i] += 1000.0 / (BIN_SMOOTHING_FACTOR + Math.abs(i++ - surfWave.hitBin(
            e.getHeadingRadians() + Math.PI)));
      }
    } catch (Exception ex) {}
  }

  private static Point2D.Double project(double heading, double distance) {
    return new Point2D.Double(
        myX + (distance * Math.sin(heading)),
        myY + (distance * Math.cos(heading)));
  }

  private static int fieldContains(double heading, double distance) {
    return Integer.signum(new Rectangle2D.Double(18.0, 18.0, 764.0, 564.0).outcode(
        project(heading, distance)));
  }

  public static class Wave extends Condition {
    public Point2D.Double source;
    public double absoluteBearingPlusPi;
    public double orbitDirection;
    public double speed;
    public int ticksSinceFire;
    public int ticksUntilBreak;
    public double[] hits;

    @Override
    public boolean test() {
      if ((ticksUntilBreak = (int)(source.distance(myX, myY) / speed) - (++ticksSinceFire)) > 0
          && surfWave == null) {
        // sets surfWave to the oldest active wave (because older waves call test() earlier)
        surfWave = this;
      }
      return false;
    }

    public int hitBin(double bulletHeadingPlusPi) {
      return MIDDLE_BIN + (int)(orbitDirection * Utils.normalRelativeAngle(
          absoluteBearingPlusPi - bulletHeadingPlusPi) / Math.asin(8 / speed));
    }
  }

  // log of enemy movement, initialize to something to avoid out-of-bounds
  static String enemyHistory = "" +
      (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 +
      (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 +
      (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 +
      (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 +
      (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 +
      (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 +
      (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 +
      (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 +
      (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 +
      (char)0 + (char)3 + (char)0 + (char)-3 + (char)0 + (char)3 + (char)0 + (char)-3 + (char)0;

  // how far we can move forward given our current velocity and ticks until the next wave hits us
  private static final String MOVE_DISTANCE_TABLE = "" +
      (char)-6 + (char)-5 + (char)-4 + (char)-3 + (char)-2 + (char)-1 + (char)0 + (char)0 + (char)1 + (char)2 + (char)3 + (char)4 + (char)5 + (char)6 + (char)7 + (char)8 + (char)8 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)-6 + (char)-5 + (char)-4 + (char)-3 + (char)-2 + (char)-1 + (char)0 + (char)0 + (char)1 + (char)2 + (char)3 + (char)4 + (char)5 + (char)6 + (char)7 + (char)8 + (char)8 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)-6 + (char)-5 + (char)-4 + (char)-3 + (char)-2 + (char)-1 + (char)0 + (char)0 + (char)1 + (char)2 + (char)3 + (char)4 + (char)5 + (char)6 + (char)7 + (char)8 + (char)8 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)-6 + (char)-5 + (char)-4 + (char)-3 + (char)-2 + (char)-1 + (char)0 + (char)0 + (char)1 + (char)2 + (char)3 + (char)4 + (char)5 + (char)6 + (char)7 + (char)8 + (char)8 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)-6 + (char)-5 + (char)-4 + (char)-3 + (char)-2 + (char)-1 + (char)0 + (char)0 + (char)1 + (char)2 + (char)3 + (char)4 + (char)5 + (char)6 + (char)7 + (char)8 + (char)8 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)-6 + (char)-5 + (char)-4 + (char)-3 + (char)-2 + (char)-1 + (char)0 + (char)0 + (char)1 + (char)2 + (char)3 + (char)4 + (char)5 + (char)6 + (char)7 + (char)8 + (char)8 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)-10 + (char)-8 + (char)-6 + (char)-4 + (char)-2 + (char)0 + (char)1 + (char)2 + (char)3 + (char)5 + (char)7 + (char)9 + (char)11 + (char)13 + (char)15 + (char)16 + (char)16 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)-12 + (char)-9 + (char)-6 + (char)-3 + (char)-1 + (char)1 + (char)3 + (char)4 + (char)6 + (char)9 + (char)12 + (char)15 + (char)18 + (char)21 + (char)23 + (char)24 + (char)24 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)-12 + (char)-8 + (char)-5 + (char)-2 + (char)1 + (char)3 + (char)6 + (char)8 + (char)10 + (char)14 + (char)18 + (char)22 + (char)26 + (char)29 + (char)31 + (char)32 + (char)32 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)-11 + (char)-7 + (char)-3 + (char)0 + (char)4 + (char)7 + (char)10 + (char)12 + (char)15 + (char)20 + (char)25 + (char)30 + (char)34 + (char)37 + (char)39 + (char)40 + (char)40 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)-9 + (char)-4 + (char)0 + (char)4 + (char)8 + (char)11 + (char)15 + (char)18 + (char)21 + (char)27 + (char)33 + (char)38 + (char)42 + (char)45 + (char)47 + (char)48 + (char)48 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)-6 + (char)-1 + (char)4 + (char)8 + (char)13 + (char)17 + (char)21 + (char)24 + (char)28 + (char)35 + (char)41 + (char)46 + (char)50 + (char)53 + (char)55 + (char)56 + (char)56 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)-2 + (char)3 + (char)9 + (char)14 + (char)19 + (char)23 + (char)28 + (char)32 + (char)36 + (char)43 + (char)49 + (char)54 + (char)58 + (char)61 + (char)63 + (char)64 + (char)64 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)3 + (char)9 + (char)15 + (char)20 + (char)26 + (char)31 + (char)36 + (char)40 + (char)44 + (char)51 + (char)57 + (char)62 + (char)66 + (char)69 + (char)71 + (char)72 + (char)72 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)9 + (char)15 + (char)22 + (char)28 + (char)34 + (char)39 + (char)44 + (char)48 + (char)52 + (char)59 + (char)65 + (char)70 + (char)74 + (char)77 + (char)79 + (char)80 + (char)80 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)16 + (char)23 + (char)30 + (char)36 + (char)42 + (char)47 + (char)52 + (char)56 + (char)60 + (char)67 + (char)73 + (char)78 + (char)82 + (char)85 + (char)87 + (char)88 + (char)88 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)24 + (char)31 + (char)38 + (char)44 + (char)50 + (char)55 + (char)60 + (char)64 + (char)68 + (char)75 + (char)81 + (char)86 + (char)90 + (char)93 + (char)95 + (char)96 + (char)96 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)32 + (char)39 + (char)46 + (char)52 + (char)58 + (char)63 + (char)68 + (char)72 + (char)76 + (char)83 + (char)89 + (char)94 + (char)98 + (char)101 + (char)103 + (char)104 + (char)104 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)40 + (char)47 + (char)54 + (char)60 + (char)66 + (char)71 + (char)76 + (char)80 + (char)84 + (char)91 + (char)97 + (char)102 + (char)106 + (char)109 + (char)111 + (char)112 + (char)112 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)48 + (char)55 + (char)62 + (char)68 + (char)74 + (char)79 + (char)84 + (char)88 + (char)92 + (char)99 + (char)105 + (char)110 + (char)114 + (char)117 + (char)119 + (char)120 + (char)120 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)56 + (char)63 + (char)70 + (char)76 + (char)82 + (char)87 + (char)92 + (char)96 + (char)100 + (char)107 + (char)113 + (char)118 + (char)122 + (char)125 + (char)127 + (char)128 + (char)128 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)64 + (char)71 + (char)78 + (char)84 + (char)90 + (char)95 + (char)100 + (char)104 + (char)108 + (char)115 + (char)121 + (char)126 + (char)130 + (char)133 + (char)135 + (char)136 + (char)136 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)72 + (char)79 + (char)86 + (char)92 + (char)98 + (char)103 + (char)108 + (char)112 + (char)116 + (char)123 + (char)129 + (char)134 + (char)138 + (char)141 + (char)143 + (char)144 + (char)144 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)80 + (char)87 + (char)94 + (char)100 + (char)106 + (char)111 + (char)116 + (char)120 + (char)124 + (char)131 + (char)137 + (char)142 + (char)146 + (char)149 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)88 + (char)95 + (char)102 + (char)108 + (char)114 + (char)119 + (char)124 + (char)128 + (char)132 + (char)139 + (char)145 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)96 + (char)103 + (char)110 + (char)116 + (char)122 + (char)127 + (char)132 + (char)136 + (char)140 + (char)147 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)104 + (char)111 + (char)118 + (char)124 + (char)130 + (char)135 + (char)140 + (char)144 + (char)148 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)112 + (char)119 + (char)126 + (char)132 + (char)138 + (char)143 + (char)148 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)120 + (char)127 + (char)134 + (char)140 + (char)146 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)128 + (char)135 + (char)142 + (char)148 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)136 + (char)143 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)144 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 +
      (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)150 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0 + (char)0;

  // what our heading should be given our direction and distance to enemy
  public static final String TARGET_HEADING_TABLE = "" +
      (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 +
      (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 +
      (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 +
      (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 +
      (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 +
      (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 +
      (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 +
      (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)15 + (char)0 + (char)165 + (char)23 + (char)0 + (char)158 +
      (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 +
      (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 +
      (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 +
      (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 +
      (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 +
      (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 +
      (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 +
      (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)23 + (char)0 + (char)158 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 +
      (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 +
      (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 +
      (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 +
      (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 +
      (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 +
      (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 +
      (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 +
      (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)45 + (char)0 + (char)135 + (char)46 + (char)0 + (char)134 + (char)46 + (char)0 + (char)134 + (char)46 +
      (char)0 + (char)134 + (char)46 + (char)0 + (char)134 + (char)46 + (char)0 + (char)134 + (char)46 + (char)0 + (char)134 + (char)46 + (char)0 + (char)134 + (char)47 + (char)0 + (char)133 + (char)47 + (char)0 +
      (char)133 + (char)47 + (char)0 + (char)133 + (char)47 + (char)0 + (char)133 + (char)47 + (char)0 + (char)133 + (char)47 + (char)0 + (char)133 + (char)47 + (char)0 + (char)133 + (char)47 + (char)0 + (char)133 +
      (char)47 + (char)0 + (char)133 + (char)47 + (char)0 + (char)133 + (char)47 + (char)0 + (char)133 + (char)48 + (char)0 + (char)132 + (char)48 + (char)0 + (char)132 + (char)48 + (char)0 + (char)132 + (char)48 +
      (char)0 + (char)132 + (char)48 + (char)0 + (char)132 + (char)48 + (char)0 + (char)132 + (char)48 + (char)0 + (char)132 + (char)48 + (char)0 + (char)132 + (char)48 + (char)0 + (char)132 + (char)48 + (char)0 +
      (char)132 + (char)48 + (char)0 + (char)132 + (char)48 + (char)0 + (char)132 + (char)49 + (char)0 + (char)131 + (char)49 + (char)0 + (char)131 + (char)49 + (char)0 + (char)131 + (char)49 + (char)0 + (char)131 +
      (char)49 + (char)0 + (char)131 + (char)49 + (char)0 + (char)131 + (char)49 + (char)0 + (char)131 + (char)49 + (char)0 + (char)131 + (char)49 + (char)0 + (char)131 + (char)49 + (char)0 + (char)131 + (char)49 +
      (char)0 + (char)131 + (char)50 + (char)0 + (char)130 + (char)50 + (char)0 + (char)130 + (char)50 + (char)0 + (char)130 + (char)50 + (char)0 + (char)130 + (char)50 + (char)0 + (char)130 + (char)50 + (char)0 +
      (char)130 + (char)50 + (char)0 + (char)130 + (char)50 + (char)0 + (char)130 + (char)50 + (char)0 + (char)130 + (char)50 + (char)0 + (char)130 + (char)50 + (char)0 + (char)130 + (char)51 + (char)0 + (char)129 +
      (char)51 + (char)0 + (char)129 + (char)51 + (char)0 + (char)129 + (char)51 + (char)0 + (char)129 + (char)51 + (char)0 + (char)129 + (char)51 + (char)0 + (char)129 + (char)51 + (char)0 + (char)129 + (char)51 +
      (char)0 + (char)129 + (char)51 + (char)0 + (char)129 + (char)51 + (char)0 + (char)129 + (char)51 + (char)0 + (char)129 + (char)51 + (char)0 + (char)129 + (char)52 + (char)0 + (char)128 + (char)52 + (char)0 +
      (char)128 + (char)52 + (char)0 + (char)128 + (char)52 + (char)0 + (char)128 + (char)52 + (char)0 + (char)128 + (char)52 + (char)0 + (char)128 + (char)52 + (char)0 + (char)128 + (char)52 + (char)0 + (char)128 +
      (char)52 + (char)0 + (char)128 + (char)52 + (char)0 + (char)128 + (char)52 + (char)0 + (char)128 + (char)53 + (char)0 + (char)127 + (char)53 + (char)0 + (char)127 + (char)53 + (char)0 + (char)127 + (char)53 +
      (char)0 + (char)127 + (char)53 + (char)0 + (char)127 + (char)53 + (char)0 + (char)127 + (char)53 + (char)0 + (char)127 + (char)53 + (char)0 + (char)127 + (char)53 + (char)0 + (char)127 + (char)53 + (char)0 +
      (char)127 + (char)53 + (char)0 + (char)127 + (char)54 + (char)0 + (char)126 + (char)54 + (char)0 + (char)126 + (char)54 + (char)0 + (char)126 + (char)54 + (char)0 + (char)126 + (char)54 + (char)0 + (char)126 +
      (char)54 + (char)0 + (char)126 + (char)54 + (char)0 + (char)126 + (char)54 + (char)0 + (char)126 + (char)54 + (char)0 + (char)126 + (char)54 + (char)0 + (char)126 + (char)54 + (char)0 + (char)126 + (char)54 +
      (char)0 + (char)126 + (char)55 + (char)0 + (char)125 + (char)55 + (char)0 + (char)125 + (char)55 + (char)0 + (char)125 + (char)55 + (char)0 + (char)125 + (char)55 + (char)0 + (char)125 + (char)55 + (char)0 +
      (char)125 + (char)55 + (char)0 + (char)125 + (char)55 + (char)0 + (char)125 + (char)55 + (char)0 + (char)125 + (char)55 + (char)0 + (char)125 + (char)55 + (char)0 + (char)125 + (char)56 + (char)0 + (char)124 +
      (char)56 + (char)0 + (char)124 + (char)56 + (char)0 + (char)124 + (char)56 + (char)0 + (char)124 + (char)56 + (char)0 + (char)124 + (char)56 + (char)0 + (char)124 + (char)56 + (char)0 + (char)124 + (char)56 +
      (char)0 + (char)124 + (char)56 + (char)0 + (char)124 + (char)56 + (char)0 + (char)124 + (char)56 + (char)0 + (char)124 + (char)57 + (char)0 + (char)123 + (char)57 + (char)0 + (char)123 + (char)57 + (char)0 +
      (char)123 + (char)57 + (char)0 + (char)123 + (char)57 + (char)0 + (char)123 + (char)57 + (char)0 + (char)123 + (char)57 + (char)0 + (char)123 + (char)57 + (char)0 + (char)123 + (char)57 + (char)0 + (char)123 +
      (char)57 + (char)0 + (char)123 + (char)57 + (char)0 + (char)123 + (char)57 + (char)0 + (char)123 + (char)58 + (char)0 + (char)122 + (char)58 + (char)0 + (char)122 + (char)58 + (char)0 + (char)122 + (char)58 +
      (char)0 + (char)122 + (char)58 + (char)0 + (char)122 + (char)58 + (char)0 + (char)122 + (char)58 + (char)0 + (char)122 + (char)58 + (char)0 + (char)122 + (char)58 + (char)0 + (char)122 + (char)58 + (char)0 +
      (char)122 + (char)58 + (char)0 + (char)122 + (char)59 + (char)0 + (char)121 + (char)59 + (char)0 + (char)121 + (char)59 + (char)0 + (char)121 + (char)59 + (char)0 + (char)121 + (char)59 + (char)0 + (char)121 +
      (char)59 + (char)0 + (char)121 + (char)59 + (char)0 + (char)121 + (char)59 + (char)0 + (char)121 + (char)59 + (char)0 + (char)121 + (char)59 + (char)0 + (char)121 + (char)59 + (char)0 + (char)121 + (char)60 +
      (char)0 + (char)120 + (char)60 + (char)0 + (char)120 + (char)60 + (char)0 + (char)120 + (char)60 + (char)0 + (char)120 + (char)60 + (char)0 + (char)120 + (char)60 + (char)0 + (char)120 + (char)60 + (char)0 +
      (char)120 + (char)60 + (char)0 + (char)120 + (char)60 + (char)0 + (char)120 + (char)60 + (char)0 + (char)120 + (char)60 + (char)0 + (char)120 + (char)60 + (char)0 + (char)120 + (char)61 + (char)0 + (char)119 +
      (char)61 + (char)0 + (char)119 + (char)61 + (char)0 + (char)119 + (char)61 + (char)0 + (char)119 + (char)61 + (char)0 + (char)119 + (char)61 + (char)0 + (char)119 + (char)61 + (char)0 + (char)119 + (char)61 +
      (char)0 + (char)119 + (char)61 + (char)0 + (char)119 + (char)61 + (char)0 + (char)119 + (char)61 + (char)0 + (char)119 + (char)62 + (char)0 + (char)118 + (char)62 + (char)0 + (char)118 + (char)62 + (char)0 +
      (char)118 + (char)62 + (char)0 + (char)118 + (char)62 + (char)0 + (char)118 + (char)62 + (char)0 + (char)118 + (char)62 + (char)0 + (char)118 + (char)62 + (char)0 + (char)118 + (char)62 + (char)0 + (char)118 +
      (char)62 + (char)0 + (char)118 + (char)62 + (char)0 + (char)118 + (char)62 + (char)0 + (char)118 + (char)63 + (char)0 + (char)117 + (char)63 + (char)0 + (char)117 + (char)63 + (char)0 + (char)117 + (char)63 +
      (char)0 + (char)117 + (char)63 + (char)0 + (char)117 + (char)63 + (char)0 + (char)117 + (char)63 + (char)0 + (char)117 + (char)63 + (char)0 + (char)117 + (char)63 + (char)0 + (char)117 + (char)63 + (char)0 +
      (char)117 + (char)63 + (char)0 + (char)117 + (char)64 + (char)0 + (char)116 + (char)64 + (char)0 + (char)116 + (char)64 + (char)0 + (char)116 + (char)64 + (char)0 + (char)116 + (char)64 + (char)0 + (char)116 +
      (char)64 + (char)0 + (char)116 + (char)64 + (char)0 + (char)116 + (char)64 + (char)0 + (char)116 + (char)64 + (char)0 + (char)116 + (char)64 + (char)0 + (char)116 + (char)64 + (char)0 + (char)116 + (char)65 +
      (char)0 + (char)115 + (char)65 + (char)0 + (char)115 + (char)65 + (char)0 + (char)115 + (char)65 + (char)0 + (char)115 + (char)65 + (char)0 + (char)115 + (char)65 + (char)0 + (char)115 + (char)65 + (char)0 +
      (char)115 + (char)65 + (char)0 + (char)115 + (char)65 + (char)0 + (char)115 + (char)65 + (char)0 + (char)115 + (char)65 + (char)0 + (char)115 + (char)65 + (char)0 + (char)115 + (char)66 + (char)0 + (char)114 +
      (char)66 + (char)0 + (char)114 + (char)66 + (char)0 + (char)114 + (char)66 + (char)0 + (char)114 + (char)66 + (char)0 + (char)114 + (char)66 + (char)0 + (char)114 + (char)66 + (char)0 + (char)114 + (char)66 +
      (char)0 + (char)114 + (char)66 + (char)0 + (char)114 + (char)66 + (char)0 + (char)114 + (char)66 + (char)0 + (char)114 + (char)67 + (char)0 + (char)113 + (char)67 + (char)0 + (char)113 + (char)67 + (char)0 +
      (char)113 + (char)67 + (char)0 + (char)113 + (char)67 + (char)0 + (char)113 + (char)67 + (char)0 + (char)113 + (char)67 + (char)0 + (char)113 + (char)67 + (char)0 + (char)113 + (char)67 + (char)0 + (char)113 +
      (char)67 + (char)0 + (char)113 + (char)67 + (char)0 + (char)113 + (char)68 + (char)0 + (char)112 + (char)68 + (char)0 + (char)112 + (char)68 + (char)0 + (char)112 + (char)68 + (char)0 + (char)112 + (char)68 +
      (char)0 + (char)112 + (char)68 + (char)0 + (char)112 + (char)68 + (char)0 + (char)112 + (char)68 + (char)0 + (char)112 + (char)68 + (char)0 + (char)112 + (char)68 + (char)0 + (char)112 + (char)68 + (char)0 +
      (char)112 + (char)68 + (char)0 + (char)112 + (char)69 + (char)0 + (char)111 + (char)69 + (char)0 + (char)111 + (char)69 + (char)0 + (char)111 + (char)69 + (char)0 + (char)111 + (char)69 + (char)0 + (char)111 +
      (char)69 + (char)0 + (char)111 + (char)69 + (char)0 + (char)111 + (char)69 + (char)0 + (char)111 + (char)69 + (char)0 + (char)111 + (char)69 + (char)0 + (char)111 + (char)69 + (char)0 + (char)111 + (char)70 +
      (char)0 + (char)110 + (char)70 + (char)0 + (char)110 + (char)70 + (char)0 + (char)110 + (char)70 + (char)0 + (char)110 + (char)70 + (char)0 + (char)110 + (char)70 + (char)0 + (char)110 + (char)70 + (char)0 +
      (char)110 + (char)70 + (char)0 + (char)110 + (char)70 + (char)0 + (char)110 + (char)70 + (char)0 + (char)110 + (char)70 + (char)0 + (char)110 + (char)71 + (char)0 + (char)109 + (char)71 + (char)0 + (char)109 +
      (char)71 + (char)0 + (char)109 + (char)71 + (char)0 + (char)109 + (char)71 + (char)0 + (char)109 + (char)71 + (char)0 + (char)109 + (char)71 + (char)0 + (char)109 + (char)71 + (char)0 + (char)109 + (char)71 +
      (char)0 + (char)109 + (char)71 + (char)0 + (char)109 + (char)71 + (char)0 + (char)109 + (char)71 + (char)0 + (char)109 + (char)72 + (char)0 + (char)108 + (char)72 + (char)0 + (char)108 + (char)72 + (char)0 +
      (char)108 + (char)72 + (char)0 + (char)108 + (char)72 + (char)0 + (char)108 + (char)72 + (char)0 + (char)108 + (char)72 + (char)0 + (char)108 + (char)72 + (char)0 + (char)108 + (char)72 + (char)0 + (char)108 +
      (char)72 + (char)0 + (char)108 + (char)72 + (char)0 + (char)108 + (char)73 + (char)0 + (char)107 + (char)73 + (char)0 + (char)107 + (char)73 + (char)0 + (char)107 + (char)73 + (char)0 + (char)107 + (char)73 +
      (char)0 + (char)107 + (char)73 + (char)0 + (char)107 + (char)73 + (char)0 + (char)107 + (char)73 + (char)0 + (char)107 + (char)73 + (char)0 + (char)107 + (char)73 + (char)0 + (char)107 + (char)73 + (char)0 +
      (char)107 + (char)74 + (char)0 + (char)106 + (char)74 + (char)0 + (char)106 + (char)74 + (char)0 + (char)106 + (char)74 + (char)0 + (char)106 + (char)74 + (char)0 + (char)106 + (char)74 + (char)0 + (char)106 +
      (char)74 + (char)0 + (char)106 + (char)74 + (char)0 + (char)106 + (char)74 + (char)0 + (char)106 + (char)74 + (char)0 + (char)106 + (char)74 + (char)0 + (char)106 + (char)74 + (char)0 + (char)106 + (char)75 +
      (char)0 + (char)105 + (char)75 + (char)0 + (char)105 + (char)75 + (char)0 + (char)105 + (char)75 + (char)0 + (char)105 + (char)75 + (char)0 + (char)105 + (char)75 + (char)0 + (char)105 + (char)75 + (char)0 +
      (char)105 + (char)75 + (char)0 + (char)105 + (char)75 + (char)0 + (char)105 + (char)75 + (char)0 + (char)105 + (char)75 + (char)0 + (char)105 + (char)76 + (char)0 + (char)104 + (char)76 + (char)0 + (char)104 +
      (char)76 + (char)0 + (char)104 + (char)76 + (char)0 + (char)104 + (char)76 + (char)0 + (char)104 + (char)76 + (char)0 + (char)104 + (char)76 + (char)0 + (char)104 + (char)76 + (char)0 + (char)104 + (char)76 +
      (char)0 + (char)104 + (char)76 + (char)0 + (char)104 + (char)76 + (char)0 + (char)104 + (char)77 + (char)0 + (char)103 + (char)77 + (char)0 + (char)103 + (char)77 + (char)0 + (char)103 + (char)77 + (char)0 +
      (char)103 + (char)77 + (char)0 + (char)103 + (char)77 + (char)0 + (char)103 + (char)77 + (char)0 + (char)103 + (char)77 + (char)0 + (char)103 + (char)77 + (char)0 + (char)103 + (char)77 + (char)0 + (char)103 +
      (char)77 + (char)0 + (char)103 + (char)77 + (char)0 + (char)103 + (char)78 + (char)0 + (char)102 + (char)78 + (char)0 + (char)102 + (char)78 + (char)0 + (char)102 + (char)78 + (char)0 + (char)102 + (char)78 +
      (char)0 + (char)102 + (char)78 + (char)0 + (char)102 + (char)78 + (char)0 + (char)102 + (char)78 + (char)0 + (char)102 + (char)78 + (char)0 + (char)102 + (char)78 + (char)0 + (char)102 + (char)78 + (char)0 +
      (char)102 + (char)79 + (char)0 + (char)101 + (char)79 + (char)0 + (char)101 + (char)79 + (char)0 + (char)101 + (char)79 + (char)0 + (char)101 + (char)79 + (char)0 + (char)101 + (char)79 + (char)0 + (char)101 +
      (char)79 + (char)0 + (char)101 + (char)79 + (char)0 + (char)101 + (char)79 + (char)0 + (char)101 + (char)79 + (char)0 + (char)101 + (char)79 + (char)0 + (char)101 + (char)80 + (char)0 + (char)100 + (char)80 +
      (char)0 + (char)100 + (char)80 + (char)0 + (char)100 + (char)80 + (char)0 + (char)100 + (char)80 + (char)0 + (char)100 + (char)80 + (char)0 + (char)100 + (char)80 + (char)0 + (char)100 + (char)80 + (char)0 +
      (char)100 + (char)80 + (char)0 + (char)100 + (char)80 + (char)0 + (char)100 + (char)80 + (char)0 + (char)100 + (char)80 + (char)0 + (char)100 + (char)81 + (char)0 + (char)99 + (char)81 + (char)0 + (char)99 +
      (char)81 + (char)0 + (char)99 + (char)81 + (char)0 + (char)99 + (char)81 + (char)0 + (char)99 + (char)81 + (char)0 + (char)99 + (char)81 + (char)0 + (char)99 + (char)81 + (char)0 + (char)99 + (char)81 +
      (char)0 + (char)99 + (char)81 + (char)0 + (char)99 + (char)81 + (char)0 + (char)99 + (char)82 + (char)0 + (char)98 + (char)82 + (char)0 + (char)98 + (char)82 + (char)0 + (char)98 + (char)82 + (char)0 +
      (char)98 + (char)82 + (char)0 + (char)98 + (char)82 + (char)0 + (char)98 + (char)82 + (char)0 + (char)98 + (char)82 + (char)0 + (char)98 + (char)82 + (char)0 + (char)98 + (char)82 + (char)0 + (char)98 +
      (char)82 + (char)0 + (char)98 + (char)83 + (char)0 + (char)97 + (char)83 + (char)0 + (char)97 + (char)83 + (char)0 + (char)97 + (char)83 + (char)0 + (char)97 + (char)83 + (char)0 + (char)97 + (char)83 +
      (char)0 + (char)97 + (char)83 + (char)0 + (char)97 + (char)83 + (char)0 + (char)97 + (char)83 + (char)0 + (char)97 + (char)83 + (char)0 + (char)97 + (char)83 + (char)0 + (char)97 + (char)83 + (char)0 +
      (char)97 + (char)84 + (char)0 + (char)96 + (char)84 + (char)0 + (char)96 + (char)84 + (char)0 + (char)96 + (char)84 + (char)0 + (char)96 + (char)84 + (char)0 + (char)96 + (char)84 + (char)0 + (char)96 +
      (char)84 + (char)0 + (char)96 + (char)84 + (char)0 + (char)96 + (char)84 + (char)0 + (char)96 + (char)84 + (char)0 + (char)96 + (char)84 + (char)0 + (char)96 + (char)85 + (char)0 + (char)95 + (char)85 +
      (char)0 + (char)95 + (char)85 + (char)0 + (char)95 + (char)85 + (char)0 + (char)95 + (char)85 + (char)0 + (char)95 + (char)85 + (char)0 + (char)95 + (char)85 + (char)0 + (char)95 + (char)85 + (char)0 +
      (char)95 + (char)85 + (char)0 + (char)95 + (char)85 + (char)0 + (char)95 + (char)85 + (char)0 + (char)95 + (char)86 + (char)0 + (char)94 + (char)86 + (char)0 + (char)94 + (char)86 + (char)0 + (char)94 +
      (char)86 + (char)0 + (char)94 + (char)86 + (char)0 + (char)94 + (char)86 + (char)0 + (char)94 + (char)86 + (char)0 + (char)94 + (char)86 + (char)0 + (char)94 + (char)86 + (char)0 + (char)94 + (char)86 +
      (char)0 + (char)94 + (char)86 + (char)0 + (char)94 + (char)86 + (char)0 + (char)94 + (char)87 + (char)0 + (char)93 + (char)87 + (char)0 + (char)93 + (char)87 + (char)0 + (char)93 + (char)87 + (char)0 +
      (char)93 + (char)87 + (char)0 + (char)93 + (char)87 + (char)0 + (char)93 + (char)87 + (char)0 + (char)93 + (char)87 + (char)0 + (char)93 + (char)87 + (char)0 + (char)93 + (char)87 + (char)0 + (char)93 +
      (char)87 + (char)0 + (char)93 + (char)88 + (char)0 + (char)92 + (char)88 + (char)0 + (char)92 + (char)88 + (char)0 + (char)92 + (char)88 + (char)0 + (char)92 + (char)88 + (char)0 + (char)92 + (char)88 +
      (char)0 + (char)92 + (char)88 + (char)0 + (char)92 + (char)88 + (char)0 + (char)92 + (char)88 + (char)0 + (char)92 + (char)88 + (char)0 + (char)92 + (char)88 + (char)0 + (char)92 + (char)89 + (char)0 +
      (char)91 + (char)89 + (char)0 + (char)91 + (char)89 + (char)0 + (char)91 + (char)89 + (char)0 + (char)91 + (char)89 + (char)0 + (char)91 + (char)89 + (char)0 + (char)91 + (char)89 + (char)0 + (char)91 +
      (char)89 + (char)0 + (char)91 + (char)89 + (char)0 + (char)91 + (char)89 + (char)0 + (char)91 + (char)89 + (char)0 + (char)91 + (char)89 + (char)0 + (char)91 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 +
      (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 +
      (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 +
      (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90 + (char)90 + (char)0 + (char)90;
}
