package pl.Patton.Tanks;

import robocode.*;

import java.awt.geom.*;
import java.util.ArrayList;

import pl.Enemy.AdvancedEnemy;
import pl.Utilities.*;

/*******************************************************************************
 * A tank that does Wave Surfing.
 * 
 * Since: 1.0
 * 
 * Last changed: 1.53
 * 
 * Originally created by basically reversing GFTGun; rewritten in v1.52 using
 * snippets taken from wiki.BasicSurfer found at
 * http://robowiki.net/cgi-bin/robowiki?BasicSurfer/Code
 ******************************************************************************/
public class WaveSurfingTank {
	// Keep track of the robot using this
	public static AdvancedRobot robot;
	// Keep track of my enemy
	public static AdvancedEnemy enemy;

	public static double MAX_DISTANCE;
	public static int DISTANCE_INDEXES;

	// Indexes are 0&+-1, +-2&+-3, +-4&+-5, +-6&+-7, +-8
	public final static int VELOCITY_INDEXES = 5;
	public final static int PREVIOUS_VELOCITY_INDEXES = VELOCITY_INDEXES;
	public final static int MIDDLE_VELOCITY_INDEX = (VELOCITY_INDEXES - 1) / 2;

	// One bin every two angles should be sufficient
	public final static int BINS = 47;
	public final static int MIDDLE_BIN = (BINS - 1) / 2;
	// This angle is derived from Math.asin(8D / (20D - (3D * power))), with
	// power being 3.0, the max bullet power. In degrees, it is
	// 46.658241772777615D, the source of the number of bins I use.
	public final static double MAX_ESC_ANGLE = 0.8143399421265254D;
	public final static double BIN_WIDTH = MAX_ESC_ANGLE / MIDDLE_BIN;

	public static Rectangle2D.Double fieldRect;

	// Segmented stats - 4 dimensional array - this gives me 11,750 bins to work
	// with :P
	public static double[][][][] stats;
	// Non-segmented stats for fast learning
	public static double[] statsFast;

	// Keep track of the waves
	public static ArrayList<WSWave> waves = new ArrayList<WSWave>();
	// Keep track of my directions, bearings, distances, and velocities
	public static ArrayList<Integer> signs = new ArrayList<Integer>();
	public static ArrayList<Double> bearings = new ArrayList<Double>();
	public static ArrayList<Double> distances = new ArrayList<Double>();
	public static ArrayList<Double> velocities = new ArrayList<Double>();

	// Remember which direction I want move in
	public byte moveDir = 1;

	// Keep track of how many times tank was hit and total times fired
	public int hits = 0;
	public int totalFired = 0;

	/**
	 * Constructor with AdvancedRobot
	 * 
	 * @param robot Robot using wavesurfing
	 */
	public WaveSurfingTank(AdvancedRobot robot) {
		WaveSurfingTank.robot = robot;
		fieldRect = new Rectangle2D.Double(18, 18, robot.getBattleFieldWidth() - 36, robot
				.getBattleFieldHeight() - 36);
		MAX_DISTANCE = Point2D.distance(0, 0, robot.getBattleFieldWidth(), robot
				.getBattleFieldHeight());
		// Each distance index is 100 large.
		DISTANCE_INDEXES = (int) (MAX_DISTANCE / 100);
		stats = new double[DISTANCE_INDEXES][VELOCITY_INDEXES][PREVIOUS_VELOCITY_INDEXES][BINS];
		statsFast = new double[BINS];
	}

	/**
	 * Called by whatever is using WaveSurfing
	 * 
	 */
	public void run(AdvancedEnemy enemy0) {
		// If I did not already have an enemy
		if (enemy == null) {
			// I do now.
			enemy = enemy0;
			hits = 0;
			totalFired = 0;
		}
		// If I have a new enemy
		else if (enemy != enemy0) {
			// Switch to the new one
			enemy = enemy0;
			hits = 0;
			totalFired = 0;
		}

		// Figure out which direction I'm moving in, CW or CCW -
		// Math.sin determines if I'm facing CW or CCW, but my
		// velocity might be negative
		signs.add(0, MiscUtils.sign(robot.getVelocity() * Math.sin(enemy.bearing)));
		// Find my absolut bearing to the opponent
		bearings.add(0, robot.getHeadingRadians() + enemy.bearing + Math.PI);
		distances.add(0, Point2D.distance(robot.getX(), robot.getY(), enemy.x, enemy.y));
		velocities.add(0, robot.getVelocity());

		// Add a wave if enemy fired a bullet
		if (enemy.firedBullet() && signs.size() > 2) {
			totalFired++;
			double firePower = enemy.getEnergyChange();
			WSWave w = new WSWave();
			// They fired the shot one tick before I detect it
			w.fireTime = enemy.lastUpdated - 1;
			// They fired from their location one tick before I detect
			// the shot
			w.firedLocation = new Point2D.Double(enemy.pX, enemy.pY);
			// The direction they're using is from two ticks ago
			w.sign = signs.get(2);
			// The bearing they're using is from two ticks ago
			w.bearing = bearings.get(2);
			// Find the velocity of the wave
			w.velocity = BulletUtils.bulletVelocity(firePower);
			// One tick after it fired, which means it's already moved
			w.distanceTraveled = BulletUtils.bulletVelocity(firePower);
			// Set the segmentations
			int distanceIndex = (int) (distances.get(2) / (MAX_DISTANCE / DISTANCE_INDEXES));
			int velocityIndex = (int) Math.abs(velocities.get(2) / 2);
			int previousVelocityIndex = (int) Math.abs(velocities.get(3) / 2);
			w.segment = stats[distanceIndex][velocityIndex][previousVelocityIndex];
			// Only add the wave if we're still active
			if (robot.getEnergy() >= 0.1)
				waves.add(w);
		}

		// Always update waves
		updateWaves();
		// I am a Southern Californian (or perhaps a just wannabe)
		doSurfing();
	}

	/**
	 * Does wave surfing
	 */
	public void doSurfing() {
		// Find the closest wave
		WSWave w = getClosestWave();
		// If it exists, surf it
		if (w != null) {
			double dangerLeft = getDanger(w, -1);
			double dangerRight = getDanger(w, 1);

			Point2D.Double myLocation = new Point2D.Double(robot.getX(), robot.getY());
			double angle = AngleUtils.absoluteBearing(w.firedLocation, myLocation);
			if (dangerLeft < dangerRight)
				angle = wallSmooth(myLocation, angle - AngleUtils.HALF_PI, -1);
			else
				angle = wallSmooth(myLocation, angle + AngleUtils.HALF_PI, 1);

			// Get the final angle we'll turn
			angle = AngleUtils.normalizeBearing(angle - robot.getHeadingRadians());
			// If we need to turn more than 90 degrees either way, go back
			// instead
			if (Math.abs(angle) > AngleUtils.HALF_PI) {
				// If we're turning left originally, turn right plus 180 degrees
				if (angle < 0)
					robot.setTurnRightRadians(angle + Math.PI);
				// Otherwise, we were turning right originally, so turn left
				// minus 180 degrees
				else
					robot.setTurnLeftRadians(angle - Math.PI);
				// We want to move backwards
				moveDir = -1;
			}
			// Otherwise, we move forward
			else {
				// Turn - if angle is negative, we'll turn left instead
				robot.setTurnRightRadians(angle);
				// We want to move forwards
				moveDir = 1;
			}
			// And move, because that's the point
			robot.setAhead(100 * moveDir);
		}
	}

	/**
	 * Finds the closest Wave
	 * 
	 * @return closest BulletWave
	 */
	public WSWave getClosestWave() {
		// Start with a huge distance
		double distance = Double.POSITIVE_INFINITY;
		WSWave wave = null;

		for (int i = 0; i < waves.size(); i++) {
			WSWave w = waves.get(i);
			double dist = w.firedLocation.distance(robot.getX(), robot.getY()) - w.distanceTraveled;

			if (distance > w.velocity && dist < distance) {
				wave = w;
				distance = dist;
			}
		}
		return wave;
	}

	/**
	 * Predicts the future position we will be at the next time we get an
	 * intercept
	 * 
	 * CREDIT: rozu
	 * 
	 * @param w the WSWave to predict intercept for
	 * @param sign direction we move in
	 * @return predicted position
	 */
	public Point2D.Double predictFutureLocation(WSWave w, int sign) {
		// The current number of ticks into the future is 0, meaning now
		int futureTicks = 0;

		// Set the future location as the current one because future time is 0
		Point2D.Double predictedLocation = new Point2D.Double(robot.getX(), robot.getY());
		// Set the future velocity as the current one because future time is 0
		double predictedVelocity = robot.getVelocity();
		// Set the future heading as the current one because future time is 0
		double predictedHeading = robot.getHeadingRadians();
		// Stuff to calculate later
		double maxTurnRate, turnAngle, moveDir;

		boolean intercepted = false;

		do {
			turnAngle = wallSmooth(predictedLocation, AngleUtils
					.absoluteBearing(w.firedLocation, predictedLocation)
					+ (sign * (Math.PI / 2)), sign)
					- predictedHeading;
			// Let's move clockwise
			moveDir = 1;

			// If we're turning backwards, let's turn and move the other way
			if (Math.cos(turnAngle) < 0) {
				turnAngle += Math.PI;
				moveDir = -1;
			}

			// And of course, normalize the angle
			turnAngle = AngleUtils.normalizeBearing(turnAngle);

			// Max turn rate is built into robocode
			maxTurnRate = Math.PI / 720D * (40D - 3D * Math.abs(predictedVelocity));
			// Add the angle we will turn at to the predicted heading
			predictedHeading = AngleUtils.normalizeBearing(predictedHeading
					+ MiscUtils.limit(-maxTurnRate, maxTurnRate, turnAngle));

			// If predictedVelocity and moveDir have different signs, break,
			// otherwise, accelerate
			predictedVelocity += (predictedVelocity * moveDir < 0 ? 2 * moveDir : moveDir);
			// Limit the velocity to possible velocities
			predictedVelocity = MiscUtils.limit(-8, 8, predictedVelocity);

			// calculate the new predicted location
			predictedLocation = MiscUtils
					.project(predictedLocation, predictedHeading, predictedVelocity);

			// Increment the ticks into the future
			futureTicks++;

			// If our future location's distance from the wave's fired location
			// is less than the distance it will have traveled when we get
			// there, we have intercepted it
			if (predictedLocation.distance(w.firedLocation) < w.distanceTraveled
					+ (futureTicks * w.velocity) + w.velocity)
				intercepted = true;
		}
		// Keep predicting until we find an intercept or we have predicted 500
		// ticks ahead
		while (!intercepted && futureTicks < 500);

		return predictedLocation;
	}

	/**
	 * Gets the danger of the index we will be at
	 * 
	 * @param w wave to check the danger of
	 * @param sign direction we're moving
	 * @return visit count of the bin we'll be at
	 */
	public double getDanger(WSWave w, int sign) {
		Point2D.Double futureLocation = predictFutureLocation(w, sign);

		// Find bin of the predicted location
		int bin = (int) Math.round((AngleUtils.normalizeBearing(
		// Absolute bearing to me minus original absolute bearing
				AngleUtils.absoluteBearing(w.firedLocation, futureLocation) - w.bearing))
		// divided by the size of each bin, but not always positive
				/ (w.sign * BIN_WIDTH))
		// And put it back in the array
				+ MIDDLE_BIN;

		// Limit my bins to the stats
		bin = (int) MiscUtils.limit(0, BINS - 1, bin);

		// Use the fast stats if the current segment is empty
		if (!isEmpty(w.segment))
			return w.segment[bin];
		else
			return statsFast[bin];
	}

	/**
	 * Iterative Wall Smoothing originally by PEZ
	 * 
	 * CREDIT: PEZ
	 * 
	 * @param source source location
	 * @param angle angle to move at
	 * @param sign direction to move
	 * @return smoothed angle
	 */
	public double wallSmooth(Point2D.Double source, double angle, int sign) {
		while (!fieldRect.contains(MiscUtils.project(source, angle, 160))) {
			angle += 0.05 * sign;
		}
		return angle;
	}

	/**
	 * Loops through a segment to see if it is empty
	 * 
	 * @param segment segment to check
	 * @return true if it's empty, false if it's not
	 */
	public boolean isEmpty(double[] segment) {
		for (int i = 0; i < segment.length; i++)
			if (segment[i] != 0D)
				return false;
		return true;
	}

	/**
	 * Logs a hit because we were hit by a bullet
	 * 
	 * @param e HitByBulletEvent
	 */
	public void logHitByBullet(HitByBulletEvent e) {
		// If there aren't any waves, we must have missed one
		if (!waves.isEmpty()) {
			Point2D.Double hitLocation = new Point2D.Double(e.getBullet().getX(), e.getBullet()
					.getY());
			WSWave hitWave = null;

			// look through the waves, and find one that probably hit us.
			for (int i = 0; i < waves.size(); i++) {
				WSWave w = waves.get(i);

				if (Math.abs(w.distanceTraveled
						- w.firedLocation.distance(robot.getX(), robot.getY())) < 50
						&& Math.round(BulletUtils.bulletVelocity(e.getBullet().getPower()) * 10) == Math
								.round(w.velocity * 10)) {
					hitWave = w;
					break;
				}
			}

			if (hitWave != null) {
				// Find bin of the hit location
				int bin = (int) Math.round((AngleUtils.normalizeBearing(
				// Absolute bearing to me minus original absolute bearing
						AngleUtils.absoluteBearing(hitWave.firedLocation, hitLocation)
								- hitWave.bearing))
						// divided by the size of each bin, but not always
						// positive
						/ (hitWave.sign * BIN_WIDTH))
						// And put it back in the array
						+ MIDDLE_BIN;

				for (int i = 0; i < BINS; i++) {
					// Bin smoothing using inverse power
					hitWave.segment[i] += 1D / (Math.pow(bin - i, 2) + 1);
					statsFast[i] += 1D / (Math.pow(bin - i, 2) + 1);
				}

				// We can remove this wave now, of course.
				waves.remove(waves.lastIndexOf(hitWave));
			}
		}
	}

	/**
	 * Logs a hit because we hit a bullet
	 * 
	 * @param e BulletHitBulletEvent
	 */
	public void logBulletHitBullet(BulletHitBulletEvent e) {
		// If there aren't any waves, we must have missed one
		if (!waves.isEmpty()) {
			Point2D.Double hitLocation = new Point2D.Double(e.getBullet().getX(), e.getBullet()
					.getY());
			WSWave hitWave = null;

			// look through the waves, and find one that probably hit us.
			for (int i = 0; i < waves.size(); i++) {
				WSWave w = waves.get(i);

				if (Math.abs(w.distanceTraveled
						- w.firedLocation.distance(robot.getX(), robot.getY())) < 50
						&& Math.round(BulletUtils.bulletVelocity(e.getBullet().getPower()) * 10) == Math
								.round(w.velocity * 10)) {
					hitWave = w;
					break;
				}
			}

			if (hitWave != null) {
				// Find bin of the hit location
				int bin = (int) Math.round((AngleUtils.normalizeBearing(
				// Absolute bearing to me minus original absolute bearing
						AngleUtils.absoluteBearing(hitWave.firedLocation, hitLocation)
								- hitWave.bearing))
						// divided by the size of each bin, but not always
						// positive
						/ (hitWave.sign * BIN_WIDTH))
						// And put it back in the array
						+ MIDDLE_BIN;

				for (int i = 0; i < BINS; i++) {
					// Bin smoothing using inverse power
					hitWave.segment[i] += 1D / (Math.pow(bin - i, 2) + 1);
					statsFast[i] += 1D / (Math.pow(bin - i, 2) + 1);
				}

				// We can remove this wave now, of course.
				waves.remove(waves.lastIndexOf(hitWave));
			}
		}
	}

	/**
	 * Updates all the BulletWaves, removes the ones that have passed, and
	 * updates the stats from the passed waves.
	 */
	public void updateWaves() {
		for (int i = 0; i < waves.size(); i++) {
			WSWave w = waves.get(i);

			// The bullet travels
			w.distanceTraveled = (robot.getTime() - w.fireTime) * w.velocity;
			// If the bullet has reached me
			if (w.distanceTraveled > w.firedLocation.distance(robot.getX(), robot.getY()) + 50) {
				// Since it's done, no point updating anymore
				waves.remove(i);
				i--;
			}
		}
	}

	/**
	 * A class for tracking bullet waves.
	 */
	public class WSWave {
		public double[] segment;
		public Point2D.Double firedLocation;
		public double velocity, distanceTraveled, bearing, sign;
		public long fireTime;
	}

}