package ags.rougedc.gun.util;

import java.awt.Polygon;
import java.util.List;
import java.util.ArrayList;
import ags.utils.Range;

/**
 * Represents a hypothetical movement profile with discrete increases and decreases
 * 
 * @author Alexander Schultz
 */
public class MovementProfile {
    private final List<double[]> data = new ArrayList<double[]>();
    
    /**
     * Add another movement profile, with a certain weight
     */
    public void add(MovementProfile profile, double weight) {
        for (double[] d : profile.data) {
            sortedEntryInsert(d[0], d[1]*weight);
        }
    }
    
    /**
     * Add a guessfactor range with a certain weight
     */
    public void add(Range range, double weight) {
        sortedEntryInsert(range.getLower(), weight);
        sortedEntryInsert(range.getUpper(), -weight);
    }
    
    public void meld(MovementProfile[] profiles, double[] weights) {
        if (profiles.length != weights.length)
            throw new IllegalArgumentException("Profiles and weights must be the same length.");
        
        final List<List<double[]>> valuetable = new ArrayList<List<double[]>>(profiles.length);
        for (MovementProfile profile : profiles) {
            final List<double[]> values = new ArrayList<double[]>(profile.data.size());
            double value = 0;
            for (int i = 0; i < profile.data.size(); i++) {
                value += data.get(i)[1];
                values.add(new double[] {data.get(i)[1], value});
            }
            valuetable.add(values);
        }
        
        final int[] indexes = new int[profiles.length];
        while(true) {
            final double[] startgf = new double[profiles.length];
            final double[] endgf = new double[profiles.length];
            for (int p=0; p<profiles.length; p++) {
                startgf[p] = valuetable.get(p).get(indexes[p])[0];
                endgf[p] = valuetable.get(p).get(indexes[p]+1)[0];
            }
            int first=0;
            for (int p=0; p<profiles.length; p++) {
                if (startgf[p] < startgf[first])
                    first = p;
            }
            
            
        }
    }
    
    /**
     * Do a sorted insert in the list
     */
    private void sortedEntryInsert(double guessfactor, double change) {
        for (int i = 0; i < data.size(); i++) {
            if (data.get(i)[0] > guessfactor) {
                data.add(i, new double[]{guessfactor, change});
                return;
            }
        }
        data.add(new double[]{guessfactor, change});
    }
    
    /**
     * Normalize the profile to peak at 1
     */
    public void normalize() {
        double value = 0;
        double maxValue = 0;
        for (int i = 0; i < data.size(); i++) {
            value += data.get(i)[1];
            if (value > maxValue) {
                maxValue = value;
            }
        }
        
        for (int i = 0; i < data.size(); i++) {
            final double[] d = data.get(i);
            d[1] = d[1] / maxValue;
        }
    }
    
    /**
     * Normalize the area at 1
     */
    public void normalizeArea() {
        double value = 0;
        double totalArea = 0;
        for (int i = 0; i < data.size()-1; i++) {
            value += data.get(i)[1];
            final double gf1 = data.get(i)[0];
            final double gf2 = data.get(i+1)[0];
            totalArea += value*Math.max(0, gf2-gf1);
        }
        for (int i = 0; i < data.size(); i++) {
            final double[] d = data.get(i);
            d[1] = d[1] / totalArea;
        }
    }
    
    /**
     * Returns the GuessFactor at the center of the peak region
     */
    public double getBestGF() {
        if (data.size() == 0)
            return 0;
        
        int maxIndex = 0;
        double maxValue = 0;
        double value = 0;
        for (int i = 0; i < data.size(); i++) {
            value += data.get(i)[1];
            if (value > maxValue) {
                maxValue = value;
                maxIndex = i;
            }
        }
        
        return (data.get(maxIndex)[0] + data.get(maxIndex+1)[0])/2;
    }
    
    /**
     * Returns the portion of the profile whose area is within a certain range
     */
    public double getAreaWithin(Range range) {
        double value = 0;
        double rangeArea = 0;
        double totalArea = 0;
        for (int i = 0; i < data.size()-1; i++) {
            value += data.get(i)[1];
            final double gf1 = data.get(i)[0];
            final double gf2 = data.get(i+1)[0];
            
            totalArea += value*Math.max(0, gf2-gf1);
            
            if (gf2 > range.getLower() && gf1 < range.getUpper()) {
                final double r1 = Math.max(gf1, range.getLower());
                final double r2 = Math.min(gf2, range.getUpper());
                rangeArea += value*Math.max(0, r2-r1);
            }
        }
        
        if (totalArea == 0)
            return 0;
        
        return rangeArea/totalArea;
    }
    
    // Get peak in range
    public double getPeakInRange(Range range) {
        double value = 0;
        double peak = 0;
        for (int i = 0; i < data.size()-1; i++) {
            value += data.get(i)[1];
            final double gf1 = data.get(i)[0];
            final double gf2 = data.get(i+1)[0];
            if (gf2 > range.getLower() && gf1 < range.getUpper()) {
                peak = Math.max(peak, value);
            }
        }
        return peak;
    }
    
    public Polygon getGraphPolygon(int x, int y, int width, int height) {
        final Polygon poly = new Polygon();
        
        final List<double[]> values = new ArrayList<double[]>(data.size());
        double value = 0;
        for (int i = 0; i < data.size(); i++) {
            value += data.get(i)[1];
            values.add(new double[]{data.get(i)[0], value});
        }
        
        double maxv=0; 
        for (double[] d : values) {
            if (d[1] > maxv)
                maxv = d[1];
        }
        poly.addPoint(x, y);
        int lasty = 0;
        for (double[] d : values) {
            final double nx = width*((d[0]/1.2+1.0)/2.0);
            final double ny = height*d[1]/maxv;
            poly.addPoint((int)nx+x, lasty+y);
            poly.addPoint((int)nx+x, (int)ny+y);
            lasty = (int)ny;
        }
        poly.addPoint(width+x, y);
        return poly;
    }
}
