package ar;

import java.awt.Color;
import java.util.*;
import robocode.*;

/**
 * QuantumChromodynamics - a robot by Aaron Rotenberg
 *
 * Movement: A complicated pattern that attempts to move into the optimal firing
 *           range around an enemy robot. Supports wall avoidance.
 * Gun:      A modified linear targeting system that can effectively target all
 *           simple bots and some stop-and-go and wave surfing bots. Least
 *           effective against walls-based bots that stop before they get all the
 *           way to the next wall.
 * Radar:    Infinity lock.
 */
public class QuantumChromodynamics extends AdvancedRobot {
	private Map<String, Double> lastEnergy = new HashMap<String, Double>();

	private Point2D target = new Point2D(0, 0);
	private int targeted = 0;

	public void run() {
		setColors(Color.blue, Color.gray, Color.black);

		setAdjustGunForRobotTurn(true);
		setTurnRadarRight(Double.POSITIVE_INFINITY);
	}

	public void onScannedRobot(ScannedRobotEvent e) {
		setTurnRadarLeft(getRadarTurnRemaining());

		boolean hasFired = false;
		if (lastEnergy.containsKey(e.getName())) {
			double energyDifference = e.getEnergy()
					- lastEnergy.get(e.getName());
			if (energyDifference > 0 && energyDifference <= 3) {
				hasFired = true;
			}
		}
		lastEnergy.put(e.getName(), e.getEnergy());

		double absoluteBearing = getHeadingRadians() + e.getBearingRadians();
		double distance = e.getDistance();
		if (e.getEnergy() > 0) {
			distance -= ((Math.random() * 200) + 100);
		}
		if ((distance < 300 || hasFired) && e.getEnergy() > 0) {
			absoluteBearing += (Math.random() < 0.5 ? 1 : -1) * Math.PI / 2;
		}
		double enemyX = getX() + distance * Math.sin(absoluteBearing);
		double enemyY = getY() + distance * Math.cos(absoluteBearing);

		if (targeted == 0) {
			target = new Point2D(enemyX, enemyY);
			targeted = 10;
		} else {
			targeted--;
		}

		distance = e.getDistance();
		double bulletPower = Math.min(Rules.MAX_BULLET_POWER, getEnergy());
		double myX = getX();
		double myY = getY();
		absoluteBearing = getHeadingRadians() + e.getBearingRadians();
		enemyX = getX() + distance * Math.sin(absoluteBearing);
		enemyY = getY() + distance * Math.cos(absoluteBearing);
		double enemyHeading = e.getHeadingRadians();
		double enemyVelocity = e.getVelocity();

		// Defense against stop-and-go and wavesurfing bots.
		if (e.getEnergy() > 0) {
			if (enemyVelocity < 2) {
				enemyVelocity += 0.5;
			} else if (enemyVelocity < 6) {
				enemyVelocity -= 0.5;
			} else {
				enemyVelocity *= 1.8;
			}
		}

		double deltaTime = 0;
		double battleFieldHeight = getBattleFieldHeight(), battleFieldWidth = getBattleFieldWidth();
		double predictedX = enemyX, predictedY = enemyY;
		while ((++deltaTime) * (20.0 - Rules.MAX_BULLET_POWER * bulletPower) < Util
				.distance(myX, myY, predictedX, predictedY)) {
			predictedX += Math.sin(enemyHeading) * enemyVelocity;
			predictedY += Math.cos(enemyHeading) * enemyVelocity;
			if (predictedX < Util.ROBOT_CENTER
					|| predictedY < Util.ROBOT_CENTER
					|| predictedX > battleFieldWidth - Util.ROBOT_CENTER
					|| predictedY > battleFieldHeight - Util.ROBOT_CENTER) {
				predictedX = Math.min(Math.max(Util.ROBOT_CENTER, predictedX),
						battleFieldWidth - Util.ROBOT_CENTER);
				predictedY = Math.min(Math.max(Util.ROBOT_CENTER, predictedY),
						battleFieldHeight - Util.ROBOT_CENTER);
				break;
			}
		}
		double theta = Util.normalAbsoluteAngleRadians(Math.atan2(predictedX
				- getX(), predictedY - getY()));

		setTurnGunRightRadians(Util.normalRelativeAngleRadians(theta
				- getGunHeadingRadians()));
		if (e.getEnergy() > 0) {
			setFire(bulletPower);
		}

		goTo(target);
	}

	private void goTo(Point2D point) {
		translateInsideField(point, Util.ROBOT_WIDTH + 10);

		double distance = Math.sqrt(Math.pow(getX() - point.x, 2)
				+ Math.pow(getY() - point.y, 2));
		double angle = Util.normalRelativeAngle(absoluteBearing(new Point2D(
				getX(), getY()), point)
				- getHeading());
		if (Math.abs(angle) > 90.0) {
			distance *= -1.0;
			if (angle > 0.0) {
				angle -= 180.0;
			} else {
				angle += 180.0;
			}
		}
		setTurnRight(angle);
		setMaxVelocity((Math.random() * 20) + 3);
		setAhead(distance);
	}

	private void translateInsideField(Point2D point, double margin) {
		point.x = Math.max(margin, Math.min(getBattleFieldWidth() - margin,
				point.x));
		point.y = Math.max(margin, Math.min(getBattleFieldHeight() - margin,
				point.y));
	}

	private double absoluteBearing(Point2D source, Point2D target) {
		return Math.toDegrees(Math.atan2(target.x - source.x, target.y
				- source.y));
	}
}
