package pe.minimelee;
import robocode.*;
import java.awt.Color;
import java.util.*;
import java.io.*;

/**
 * SandboxMiniMelee - a robot by Paul Evans
 * Please acknowledge any code/ideas used.
 * for more information visit www.sandbox.dial.pipex.com/robocode
 */
public class SandboxMiniMelee extends TeamRobot
{

	//***********************        Declarations      ***********************//

		//statics us less space - I don't know why - initilise key variables at the start of each battle to prevent values from the last battle being used.

		static Enemy target;				//our present target robot
		static HashMap enemies;				//a mapping (bot name is key) of all our opponents and fake antigravity bots

		static double offset;				//the last direction the radar was going before we set it spinning (saves doubleing back)

		static double direction;			//direction we are heading...0 = forward, 1 = backwards
		static double nextTurn;				//the time we will select another co-ordinate to move to
		static double nextX;				//co-ordinates of where we are moving to at the moment
		static double nextY;
		static double leftRight;			//set to be a 50/50 chance of left or right (-1.0,1.0)
						
		static double [][] corners;			//an array of 4 corners and factors to apply to the corner moves
		static double [][] cornerMoves;		//an array of 16 coordinates for moving in the corners
		static int inCorner;				//indicates which corner we are in (0 to 3) or not in a corner (-1)
		
		static double bulletPower;			//The bullet power in use							
		
	//***********************          run            ***********************//		
	public void run() {

		setColors(Color.pink,Color.pink,Color.white);
		setAdjustGunForRobotTurn(true);
		setAdjustRadarForGunTurn(true);
		nextTurn = 0;						// make sure we are due to set next move coordinates
//		do_buildObjects();					// to create new objects activate this line (and the section at the bottom of this class)
		enemies = (HashMap) PEReadObject("agPoints.data");				//read in an empty list of enemies and 5 anti-gravity points
		corners = (double[][]) PEReadObject("corners.data");			//read in 4 corners and factors for corner moves
		cornerMoves = (double[][]) PEReadObject("cornerMoves.data");	//read in corner move data
		inCorner = -1;

		while(true) {
			do_setTarget();					//set target to our nearest enemy
			if(target==null) {
				do_radar();					//if we dont have a target just scan to avoid null references on the target
				execute();
			} else{
				bulletPower = Math.min(3.0,Math.max(0.1,-20.0/3.0/750.0*target.distance + 20.0/3.0));		//this calculation sets a 3 power bullet when the enemy is closer than 55% of range (750.0) - after that bullet power decreaces at a rate to keep bullet time constant
				do_movement();			//move our robot
				do_radar();				//point the radar at the enemy
				do_gun();				//point the gun in the same direction as the radar, adjust for guess factor, and fire if possible
				execute();
			}
		}
	}

	//***********************      Movement         ***********************//
	void do_movement() {

		if(getTime()>=nextTurn) {	//has the time come to set new nextX and nextY co-ordinates and a new nextTurn value
			leftRight = sign(Math.random()-0.5);   	//returns -1.0 or 1.0 used in do_dualMove and do_AGmove
			for (int i=0; i<4; i++) {			//test to se if we are withing 250 units of a corner - if so set inCorner
				if (getRange(getX(),getY(),corners[i][0]*getBattleFieldWidth(),corners[i][1]*getBattleFieldHeight()) < 250.0) inCorner=i;
			}
			if(getOthers() == 1) {				// if it is now one on one do standard 1v1 movement
				do_duelMove();
			} else if (inCorner >= 0 && target.distance >100.0) {		//if we are in a corner by ourselves do our corner move
				//******* Corner Movement *******//
				nextTurn = getTime()+11;				//move on every multiple of 11
				int moveNum = (int)(getTime()/11)%16;	//we have 16 coordinates that make up our corner move - select the one for this turn
				nextX = getBattleFieldWidth() *corners[inCorner][0] + cornerMoves[moveNum][0]*corners[inCorner][2];		//ensures the corner move works for any battle field size
				nextY = getBattleFieldHeight()*corners[inCorner][1] + cornerMoves[moveNum][1]*corners[inCorner][3];
			} else {  //otherwise do our antigravity move
				inCorner = -1;  //set inCorner to represent not on a corner (this won't be set again unless we get near the corner again)
				do_AGMove();
			}
			nextX=Math.min(getBattleFieldWidth()-50,Math.max(50,nextX))*2.0 - nextX;	//we don't want to crash into walls so if our co-ordinates cross a 50 unit margin inside the walls - reflect the co-ordinate about the margin line
			nextY=Math.min(getBattleFieldHeight()-50,Math.max(50,nextY))*2.0 - nextY;
		}
		
		
		// The rest of this method gets us to the nextX, nextY coordinate and stops there (or near there) until a new nextX, nextY is set when nextMove==getTime()
		double reqOffset = normaliseBearing(Math.atan2(nextX-getX(),nextY-getY())+direction*Math.PI - getHeadingRadians());  //calculate the change of angle required to get to the target co-ordianates
		if (Math.abs(reqOffset) > Math.PI/2.0) {  //if it is more than 180 degrees
			reqOffset += Math.PI;				//add 180 degrees to the offset
			if (direction == 1) direction=0; else direction=1;  //and change the direction.
		}
		setTurnRightRadians(normaliseBearing(reqOffset));
		double distToGo=getRange(getX(),getY(),nextX,nextY);	//calculate how far to out target point
		setAhead((-2.0*direction+1.0)*distToGo);				//tell the engine the distance (it will do all the speed calcs to stop on the point
		if(Math.abs(getTurnRemaining()) > 65) setAhead(0);  	// if we need to turn sharply slow down/stop for a faster, tighter turn
		
		if (distToGo < 5.0) {		//to ensure we do not wobble and spin over the target point - be stationary if within 5 units of the point
			setAhead(0.0);
			setTurnRightRadians(0.0);
		}
	}
	
	//***********************      1v1 Movement            ***********************//
	void do_duelMove() {
		
		nextTurn = getTime() + target.distance/15;		//bullets travel at between 11 and 19.7 units per tick - set new nextX,Y co-ordinates a little before a 3.0 bullet would reach us
		double distFactor = Math.random();				//distance factor is roughly how far to travel until nextTurn (1.0 a long way, 0.0 nowhere)
		double adjustInOut = Math.max(-Math.PI/4,(target.distance-300)/-200.0*Math.PI/4);  //negative -get nearer the opponent, +ve further away
		double angleAwayFromTarget = (Math.PI/2.0 + adjustInOut) * leftRight;		//basically move at 90 degrees to the opponent, adjust if we are too far or too near the opponent - and choose either clockwise or anticlockwise at random.
		nextX=getX() + Math.sin(target.bearingAbs+angleAwayFromTarget) * target.distance*0.8*distFactor;		//in the time allocated we could travel a maximum of around 0.8 times the distance to the enemy
		nextY=getY() + Math.cos(target.bearingAbs+angleAwayFromTarget) * target.distance*0.8*distFactor;
	}
				
	//***********************      Anti Gravity Movement            ***********************//
	void do_AGMove() {
		double bestG = Double.MAX_VALUE;
		double testAngle;
		double bestAngle = 0.0;
		double testX;
		double testY;
		double distance;
		double testG;
				
		nextTurn=getTime()+16;
		for (testAngle=0.0872665 ; testAngle<2.0*Math.PI ; testAngle+=0.174533) {   //test angles all around our present position  starting at 5 degrees and increment by 10 degrees
			testG=0.0;
			for (double adjustAngle=-0.26180; adjustAngle<0.27; adjustAngle+=0.26180*2.0) {  // sum the gravity for two angles 15 degrees either side of the test angle
				testX=getX()+8.0*6.0*Math.sin(testAngle+adjustAngle);					//the two test points are at a distance of a 6 tick full speed move from our present position
				testY=getY()+8.0*6.0*Math.cos(testAngle+adjustAngle);
				for (Iterator m = enemies.entrySet().iterator(); m.hasNext(); ) {		//loop through all our enemies and the fake AG bots and calculate a gravity value for each test point
					Enemy iteratedBot = (Enemy) ((Map.Entry) m.next()).getValue();
					distance = getRange(testX,testY,iteratedBot.x+iteratedBot.agX*getBattleFieldWidth() , iteratedBot.y+iteratedBot.agY*getBattleFieldHeight());
					testG += 1.0/(distance*distance);
				}
			}
			if (testG < bestG) {		//keep details of the best test angle 
				bestG = testG;
				bestAngle=testAngle;
			}
		}
		nextX=getX()+8.0*16.0*1.5*Math.sin(bestAngle+leftRight*0.0872665);		//set nextX,nextY to one of two angles 15% either side of the best angle
		nextY=getY()+8.0*16.0*1.5*Math.cos(bestAngle+leftRight*0.0872665);
	}
	


				
	//***********************       Radar            ***********************//
	void do_radar() {
		
		if (target==null || getGunHeat()>0.5) {
			setTurnRadarRightRadians(1 + 2*sign(offset));		//no target or the gun is hot - spin the radar (in the same direction as the last purposeful radar movement) (if 
		} else {
			offset = normaliseBearing(target.bearingAbs - getRadarHeadingRadians());		//the new offset is basically the difference between the direction of the target now/was and the where the radar is presently pointing 
			setTurnRadarRightRadians(offset + sign(offset)*0.4);  //overshoot by approx 1/16th of a circle
		}
	}
	
	//***********************         Gun            ***********************//
	void do_gun() {
		
		setTurnGunRightRadians(normaliseBearing(target.bearingAbs - getGunHeadingRadians() + (Math.random()*0.3 - 0.15 + 0.9 - target.distance/1000.0)*target.linearShootAngle));	// point the gun in the same direction as the enemy with an offset dicatated by linear targeting and a factor wich decreases with distance
		if(getEnergy()>bulletPower) {		//don't fire if it will disable us
			setFire(bulletPower);
		}
	}
	
	//***********************    onScannedRobot        ***********************//
	public void onScannedRobot(ScannedRobotEvent e) {
		
		Enemy scannedRobot = (Enemy) enemies.get(e.getName());		//locate the scanned nbot in our map of enemies
		if (scannedRobot==null) {									//if there is no bot there create one
			scannedRobot = new Enemy();
			enemies.put(e.getName(),scannedRobot);
		}

		scannedRobot.bearingAbs = e.getBearingRadians()+getHeadingRadians();		//set important parameteters about the scanned bot
		scannedRobot.distance = e.getDistance();
		scannedRobot.energy = e.getEnergy();
		scannedRobot.x = getX()+Math.sin(scannedRobot.bearingAbs)*scannedRobot.distance;
		scannedRobot.y = getY()+Math.cos(scannedRobot.bearingAbs)*scannedRobot.distance;
		scannedRobot.linearShootAngle = Math.asin(e.getVelocity()/(20.0-bulletPower*3.0) * Math.sin(e.getHeadingRadians()-scannedRobot.bearingAbs));  //use sine rule to work out the gun offset angle - the two lengths are proportional to our bullet velocity (11.0) and the enemy velocity - the sign of the result gives us clockwise +ve

	}

	//***********************   Set Target   ***********************//
	void do_setTarget() {
		
		target=null;
		for (Iterator m = enemies.entrySet().iterator(); m.hasNext(); ) {		//loop over all enemies and anti gravity bots
			Map.Entry entry = (Map.Entry) m.next();
			Enemy iteratedBot = (Enemy) entry.getValue();
			if (!isTeammate((String)entry.getKey()) && !iteratedBot.isAgBot && (target == null || iteratedBot.distance<target.distance)) target=iteratedBot;		//ignore any agbots and find the nearest bot - target is null if no bots have been scanned 
		}
	}				

	
	//***********************   onRobotDeath   ***********************//
	public void onRobotDeath(RobotDeathEvent e) {
		
			enemies.remove(e.getName());		//the robot has been killed by someone - remove it from our mapping
	}
	
	
	//***********************        Helpers        ***********************//
	double normaliseBearing(double ang) {
		while (ang > Math.PI) ang -= 2*Math.PI;
		while (ang <= -Math.PI) ang += 2*Math.PI;
		return ang;
	}
    
	double getRange( double x1,double y1, double x2,double y2 ) {
		double xo = x2-x1;
		double yo = y2-y1;
		return Math.sqrt( ( (xo)*(xo) ) + ( (yo)*(yo) ) );
	}
	
	double sign(double value) {
		return (value == 0.0) ? 0.0 : value/Math.abs(value);
	}
		
	Object PEReadObject(String fileName) {		//returns an object read from file (could be an array, collection, anything other than a primative) from the specified file
		try {
			ObjectInputStream ois = new ObjectInputStream(new FileInputStream(getDataFile(fileName)));
			Object o = ois.readObject();
			ois.close();
			return o;
		} catch (IOException e) {
//			out.println("IOException trying to read: "+e);
		} catch (ClassNotFoundException e) {
//			out.println("error reading object from file: "+fileName+"   exception: "+e);
		}
		return null;
	}

// This next block is normally commented out - if it is un-commented and the do_buildObjects code is uncommented in the run() method new data files will be generated at the beginning of each round
/*
	void do_buildObjects() {
		// Five AG points - 100 units off the center of each edge and one in the middle of the battle field (x and y aka agX and agY are eventually multiplied by getBattleFieldHeight() and ...Width())
		//                        x      y      offset x   offset y
		double [][] agPoints = {{0.5,  0.0,       0.0,      -100.0},  //bottom
		                        {0.5,  1.0,       0.0,       100.0},  //top
		                        {0.0,  0.5,    -100.0,         0.0},  //left
					            {1.0,  0.5,     100.0,         0.0},  //right
		                        {0.5,  0.5,       0.0,         0.0},  //middle
		                       };

		enemies = new HashMap();
		for (int i=0; i<5; i++) {
			Enemy enemy = new Enemy();
			enemy.isAgBot=true;
			enemy.agX=agPoints[i][0];
			enemy.agY=agPoints[i][1];
			enemy.x=agPoints[i][2];
			enemy.y=agPoints[i][3];
			enemies.put("dummy"+i,enemy);
		}		
		PEWriteObject(enemies,"agPoints.data");
		
		// each of 4 corners (x and y mutiplied by getBattleFieldWidth() and ...Height()) and the scale factors for offset of corner moves
		//                       x         y        scalex     scaley
		double [][] corners = {{0.0,      0.0,      50.0,     50.0},
		                       {1.0,      0.0,     -50.0,     50.0},
					           {0.0,      1.0,      50.0,    -50.0},
		                       {1.0,      1.0,     -50.0,    -50.0}
		                      };
		PEWriteObject(corners,"corners.data");
		
		//16 corner moves , presently multiplied by 50
		//                          x    y
		double[][] cornerMoves = {{2.0,2.0},
		                          {1.0,3.0}, 
		                          {2.0,4.0},
		                          {1.0,5.0},
		                          {1.0,6.0},
		                          {1.0,5.0},
		                          {2.0,4.0},
		                          {1.0,3.0},
		                          {2.0,2.0},
		                          {3.0,1.0},
		                          {4.0,2.0},
		                          {5.0,1.0},
		                          {6.0,1.0},
		                          {5.0,1.0},
		                          {4.0,2.0},
		                          {3.0,1.0}
		                         };
		PEWriteObject(cornerMoves,"cornerMoves.data");
	}				
		
		
	void PEWriteObject(Object o, String fileName) { 
		try {
			ObjectOutputStream oos = new ObjectOutputStream(new RobocodeFileOutputStream(getDataFile(fileName)));
			oos.writeObject(o);
			oos.close();
		} catch (IOException e) {
			out.println("IOException trying to write: " + e);
		}
	}
*/

}

	//***********************     CLASS Enemy      ***********************//
	
class Enemy implements Serializable{	//Serializable allows us to use objectInput/OutputStream
		double bearingAbs;				//Heading of the target from us (+/- PI)
		double distance;				//distance target is away from us
		double energy;					//energy level of the target
		double x;						//coordinates of the target (or an offset for ag bots)
		double y;
		double linearShootAngle;		//gun offset (relative to present direction of target) if target was to continue in a straight line at it's present velocity with a power 3 bullet (+ve clockwise, -ve anticlockwise)
		boolean isAgBot;				//true if an ag bot
		double agX;						// 0.0 for left side 1.0 for right side etc
		double agY;						// 0.0 for bottom, 1.0 for top etc
}
		

				

								