package ags.rougedc.gun;

import static robocode.util.Utils.normalRelativeAngle;
import ags.rougedc.base.*;
import ags.rougedc.base.actors.GunActor;
import ags.rougedc.gun.util.MovementProfile;
import ags.rougedc.gun.util.Predictor;
import ags.rougedc.waves.SelfWave;
import ags.rougedc.robots.*;
import ags.utils.points.*;
import robocode.Bullet;

import java.util.ArrayList;
import java.util.List;

public class FlammeRouge {
    private final Rules rules;
    private final EnemyList enemies;
    private final StatusRobot status;    
    private final List<SelfWave> waves = new ArrayList<SelfWave>();
    private final Predictor predictor;
    private final static BulletPowerSelector bps = new BulletPowerSelector();
    
    public FlammeRouge(Rules rules, StatusRobot status, EnemyList enemies) {
        this.rules = rules;
        this.status = status;
        this.enemies = enemies;
        predictor = new CrowdPredictor(
                new CrowdPredictor2TMP(
                        new DCPredictor(this.rules, this.status),
                        new DCPredictor2TMP(this.rules, this.status)),
                new PatternPredictor(this.rules));
    }
    
    private MovementProfile profile = null;
    private SelfWave possibleWave = null; 
    private double bulletpower;
    public void run(GunActor actor) {
        processWaves();
        
        EnemyRobot target = enemies.getTarget();
        if (target == null)
            return;
        
        // Fire bullet/wave
        boolean fired = false;
        if (possibleWave != null) {
            Bullet bullet = actor.setFire(bulletpower);
            if (bullet != null) {
                predictor.waveFired(possibleWave);
                waves.add(possibleWave);
                fired = true;
            }
        }
        
        predictor.remember(target);

        // If the gunheat is too high or we just fired, then don't bother aiming yet
        if (status.getGunHeat() > 0.5 || fired) {
            actor.setTurnGun(normalRelativeAngle(target.getRelativeLocation().direction - status.getGunHeading()));
            return;
        }
        
        final AbsolutePoint nextLocation = status.getNextLocation();
        final double distance = target.getLocation().distance(nextLocation);
        
        //System.out.println("Best bulletpower: "+bps.getBestPower(distance));
        //bulletpower = (distance > 150)?1.9:2.5;
        bulletpower = bps.getBestPower(distance, status.getEnergy(), target.getEnergy());
        bulletpower = Math.max(Math.min(bulletpower, Math.min(status.getEnergy()/10, rules.energyToBulletPower(target.getEnergy()))), 0.1);
        System.out.println("Selected bulletpower: "+bulletpower);
        //bulletpower = Math.min(3.0, status.getEnergy()); // FOR TC2K7
        //bulletpower = Math.min(0.5, status.getEnergy()); // FOR PCM
        
        predictor.run(nextLocation, target, bulletpower);
        
        final double gf;
        if (target.getEnergy() > 0 || target.getVelocity().magnitude > 0) {
            profile = predictor.getPredictedProfile();
            if (profile != null)
                gf = profile.getBestGF();
            else
                gf = 0;
        } else {
            gf = 0; // If they are disabled and unmoving, we'll just fire at GF 0.0
        }
        
        possibleWave = new SelfWave(nextLocation, target, bulletpower);
        
        double angle = possibleWave.getAngle(gf);
        possibleWave.targetedGF = gf;
        
        double turn = normalRelativeAngle(angle - status.getGunHeading());
        
        actor.setTurnGun(turn);
    }
    
    private void processWaves() {
        for (java.util.Iterator<SelfWave> iter = waves.iterator(); iter.hasNext(); ) {
            final SelfWave wave = iter.next();
            
            if (!wave.getTarget().isAlive()) {
                iter.remove();
                continue;
            }
            
            wave.move();
            wave.checkHit(wave.getTarget());
            
            if (wave.expired(wave.getTarget())) {
                iter.remove();
                predictor.waveHit(wave);
                bps.waveEnd(wave);
            }
        }
    }
    
    public List<SelfWave> getWaves() {
        return waves;
    }
    
    public java.awt.Polygon getProfileGraph(int x, int y, int width, int height) {
        if (profile == null)
            return null;
        return profile.getGraphPolygon(x, y, width, height);
    }
}
