package gh.ghgun;

import gh.ghutils.*;

/**
 * GunStat
 *
 * This class represents the statistics for a gun.
 * Credit: Idea derived from the implementation of Dookious by Voidious
 *
 * This code is open source, released under the RoboWiki Public Code License:
 * http://robowiki.net/cgi-bin/robowiki?RWPCL
 */

public class GunStat {

	private double			antiSurf;
	private double			vWaveWeight;
	private double			waveDecay;
	private int				gunFired;
	private int				gunHit;
	private double			gunAvgHit;
	private double[]		att0Buf = new double[GHUtils.GBINS];
	private double[][]		att1Buf = new double[GHUtils.GACCSEG][GHUtils.GBINS+1];		// acceleration only
//	static double[][][]		att2Buf = new double[ACCSEG][DISTSEG][BINS+1];		// + distance
	private double[][][]	att2Buf = new double[GHUtils.GACCSEG]
										[GHUtils.GVELSEG][GHUtils.GBINS+1];		// + lateral velocity
	private double[][][][]	att3Buf = new double[GHUtils.GACCSEG]
										[GHUtils.GVELSEG][GHUtils.GWALLSEG]
										[GHUtils.GBINS+1];						// + wall
	private double[][][][][] att4Buf = new double[GHUtils.GACCSEG]
										[GHUtils.GVELSEG][GHUtils.GWALLSEG]
										[GHUtils.GDL14SEG][GHUtils.GBINS+1];	// + distance last 14 ticks

	public GunStat( double vwaveweight, double decay, double antisurf) {
		vWaveWeight = vwaveweight;
		waveDecay = decay;
		antiSurf = antisurf;
		att0Buf[GHUtils.GMIDBIN] = 0.1;		// prepopulate head-on
	}

	// newRound:  new round initialisation
	void newRound() {
		gunAvgHit *= 0.9;
		return;
	}

	// getGunHit:  get number of hits for this gun
	int getGunHit() {
		return gunHit;
	}

	// getGunAvgHit:  get number of hits for this gun (normalised)
	double getGunAvgHit() {
		return gunAvgHit;
	}

	// setGunHit: increase number of hits for this gun
	void setGunHit() {
		gunHit++;
		gunAvgHit++;
	}

	// getGunFired:  get number of times this gun really fired
	int getGunFired() {
		return gunFired;
	}

	// setGunFired: increase number of real firings for this gun
	void setGunFired() {
		gunFired++;
	}

	// get segVisits:
//	public void printSegVisits() {
//		for (int i = 0; i < GHUtils.GVELSEG; i++) {
//			System.out.println("Seg "+i+" visits: "+att1Buf[i][GHUtils.GBINS]);
//			System.out.println("Seg "+i+" visits: "+att2Buf[0][i][GHUtils.GBINS]);
//			System.out.println("Seg "+i+" visits: "+att3Buf[0][0][i][GHUtils.GBINS]);
//			System.out.println("Seg "+i+" visits: "+att4Buf[0][0][0][i][GHUtils.GBINS]);
//		}
//	}

	/**
	 * getBestIndex: check which dimension to use, and then get the best index
	 */
	public int getBestIndex (int minbin, int maxbin, int aseg, int vseg, int wseg, int dseg) {
		int bestindex;
		if (att4Buf[aseg][vseg][wseg][dseg][GHUtils.GBINS] > 4) {
			bestindex = doGetBestIndex( minbin, maxbin, att4Buf[aseg][vseg][wseg][dseg]);
		}
		else if (att3Buf[aseg][vseg][wseg][GHUtils.GBINS] > 4) {
			bestindex = doGetBestIndex( minbin, maxbin, att3Buf[aseg][vseg][wseg]);
		}
		else if (att2Buf[aseg][vseg][GHUtils.GBINS] > 4) {
			bestindex = doGetBestIndex( minbin, maxbin, att2Buf[aseg][vseg]);
		}
		else if (att1Buf[aseg][GHUtils.GBINS] > 4) {
			bestindex = doGetBestIndex( minbin, maxbin, att1Buf[aseg]);
		}
		else {
			bestindex = doGetBestIndex( minbin, maxbin, att0Buf);
		}
		return bestindex;
	}

	/**
	 * doGetBestIndex: Get the best scoring index to aim
	 * minbin and maxbin represent the maximum reacheable escape angle (MEA)
	 * !??! I know this should be checked with the nr of bins of BOTWIDTH,
	 * but that would require two nested loops.
	 * I am just lazy for now and use 5 bins instead of the real nr of bins
	 */
	private int doGetBestIndex( int minbin, int maxbin, double[] testbuf) {
		int mini = Math.min( Math.max( 0, minbin - 2), GHUtils.GBINS - 5);	// set startbin
		int maxi = Math.max( Math.min( maxbin, GHUtils.GBINS - 3), 3);		// set endbin
		int bindex = mini + 2;
		double tvalue = testbuf[bindex-2] + testbuf[bindex-1] + testbuf[bindex] + testbuf[bindex] + testbuf[bindex+1] + testbuf[bindex+2];
		double bvalue = tvalue;
		for (int i = mini+3; i <= maxi; i++) {
			tvalue += (testbuf[i] + testbuf[i+2] - testbuf[i-1] - testbuf[i-3]);
			if (tvalue > bvalue) {
				bindex = i;
				bvalue = tvalue;
			}
		}
		return bindex;
	}

	// decay: let the gunstats decay
	public void decay() {
		if (waveDecay == 1.0) { return; }
		for (int i = 0; i < GHUtils.GBINS; i++) {
			att0Buf[i] *= waveDecay;
			for (int i1 = 0; i1 < GHUtils.GACCSEG; i1++) {
				att1Buf[i1][i] *= waveDecay;
				for (int i2 = 0; i2 < GHUtils.GVELSEG; i2++) {
					att2Buf[i1][i2][i] *= waveDecay;
					for (int i3 = 0; i3 < GHUtils.GWALLSEG; i3++) {
						att3Buf[i1][i2][i3][i] *= waveDecay;
						for (int i4 = 0; i4 < GHUtils.GDL14SEG; i4++) {
							att4Buf[i1][i2][i3][i4][i] *= waveDecay;
						}
					}
				}
			}
		}
	}

	// update: update the gunstats with a hit
	public void update( GunWave wave) {
		double weight = (wave.bulletWave ? 1.0 : vWaveWeight);
		if (antiSurf != 0.0 && wave.bulletWave && (wave.getminbin() <= wave.hittedBin) && (wave.getmaxbin() >= wave.hittedBin)) {
//			int difbin = (wave.getmaxbin() - wave.getminbin()) / 2;
			int minbin = Math.max(0, wave.hittedBin - 2);
			int maxbin = Math.min(wave.hittedBin + 2, GHUtils.GBINS - 1);
			for (int i = minbin; i <= maxbin; i++) {
				att0Buf[i] -= antiSurf;
				att1Buf[wave.accelerationBin][i] -= antiSurf;
				att2Buf[wave.accelerationBin][wave.vlateralBin][i] -= antiSurf;
				att3Buf[wave.accelerationBin][wave.vlateralBin][wave.wallProxBin][i] -= antiSurf;
				att4Buf[wave.accelerationBin][wave.vlateralBin][wave.wallProxBin][wave.dist14Bin][i] -= antiSurf;
			}			
//System.out.println("Anti-surf on bin "+wave.hittedBin+" ("+minbin+"-"+maxbin+")");
		}
		for (int i = wave.getminbin(); i <= wave.getmaxbin(); i++) {
			att0Buf[i] += weight;
			att1Buf[wave.accelerationBin][i] += weight;
			att2Buf[wave.accelerationBin][wave.vlateralBin][i] += weight;
			att3Buf[wave.accelerationBin][wave.vlateralBin][wave.wallProxBin][i] += weight;
			att4Buf[wave.accelerationBin][wave.vlateralBin][wave.wallProxBin][wave.dist14Bin][i] += weight;
		}
		// count the samples
		att1Buf[wave.accelerationBin][GHUtils.GBINS] += weight;
		att2Buf[wave.accelerationBin][wave.vlateralBin][GHUtils.GBINS] += weight;
		att3Buf[wave.accelerationBin][wave.vlateralBin][wave.wallProxBin][GHUtils.GBINS] += weight;
		att4Buf[wave.accelerationBin][wave.vlateralBin][wave.wallProxBin][wave.dist14Bin][GHUtils.GBINS] += weight;
	}

}
