package djc;
import robocode.*;
import java.util.Hashtable;
import java.util.Enumeration;

/**
 * TacticalStrategyManager - Based on the StrategyManager
 * class by Simon Parker in JollyNinja
 *
 *   Manages current strategy for the StrategyBot.
 */
public class TacticalStrategyManager extends Object
{
    /* ********************************************************************************** */
    /*                                   CONSTANTS                                        */
    /* ********************************************************************************** */
    public static final String DEFAULT = "DEFAULT";       // Works
    public static final String CORNERS = "CORNERS";       // Works
    public static final String TRON = "TRON";             // Works
    public static final String ORBIT = "ORBIT";           // Works
    public static final String ANTIGRAV = "ANTIGRAV";     // Works

    // Experimental
    public static final String CORNERS_AG = "CORNERS_AG"; // Initial phase
    public static final String TRON_AG = "TRON_AG";       // Initial phase
    public static final String DODGE_AG = "DODGE_AG";     // Initial phase

    public static final String STARTSTOP = "STARTSTOP";   // Does not exist
    public static final String DODGE = "DODGE";           // Initial phase

    // Special strategies
    public static final String VICTORY = "VICTORY";       // Works
    public static final String DISABLED = "DISABLED";     // Works

    /** Minimum time to keep a strategy */
    public static final double MIN_STRAT_TIME = 150;

    /* ********************************************************************************** */
    /*                                MEMBER VARIABLES                                    */
    /* ********************************************************************************** */
    /** The initial strategy to use by default. */
    //public String INITIAL = DEFAULT;
    public String INITIAL = ANTIGRAV;
    //public String INITIAL = DODGE_AG;
    //public String INITIAL = CORNERS;
    //public String INITIAL = STARTSTOP;
    //public String INITIAL = TRON;
    //public String INITIAL = ORBIT;
    public StrategyBot self = null;
    public Hashtable availableStrategies = new Hashtable();
    // Order is also order of preference
    //private String[] meleeStrategies = {TRON, CORNERS,
    //					ANTIGRAV, ORBIT};
    private String[] meleeStrategies = {ANTIGRAV, CORNERS};
    //private String[] oneVoneStrategies = {TRON, ANTIGRAV, 
    //					  ORBIT, DEFAULT};
    //private String[] oneVoneStrategies = {TRON, ORBIT, ANTIGRAV};
    //private String[] oneVoneStrategies = {ANTIGRAV, TRON};
    private String[] oneVoneStrategies = {ANTIGRAV, TRON, DEFAULT};
    //private String[] oneVoneStrategies = {ORBIT};
    //private String[] oneVoneStrategies = {ANTIGRAV};
    //private String[] oneVoneStrategies = {ANTIGRAV};
    //private String[] oneVoneStrategies = {DODGE_AG};
    /** Stores total accrued damage for each strategy */
    public static Hashtable accruedDamage = new Hashtable();
    /** Stores total accrued time for each strategy */
    public static Hashtable timeStrategyUsed = new Hashtable();
    /** Stores time last started for each strategy */
    public static Hashtable lastTimeStrategyStarted = new Hashtable();
    public static boolean bStartNewRound = true;
    public int iLastTimeWon = 0;
    public String sLastWinningStrategy = INITIAL;

    /**
     * TacticalStrategyManager constructor
     * 
     * Currently adds the following TacticalStrategies:
     * <ul>
     *   <li> DefaultTacticalStrategy </li>
     *   <li> AntiGravTacticalStrategy </li>
     * </ul>
     *
     * @param theRobot - who I work for.
     */
    public TacticalStrategyManager(StrategyBot theRobot)
    {
	self = theRobot;
	TacticalStrategy aStrategy, initStrategy;
	
	// Add all known TacticalStrategies to availableStrategies.
	aStrategy = new DefaultTacticalStrategy(self, this);
	availableStrategies.put(aStrategy.name, aStrategy);

	aStrategy = new AntiGravTacticalStrategy(self, this);
	availableStrategies.put(aStrategy.name, aStrategy);
	
	// This one was really bad - don't even bother
	//aStrategy = new DodgeAGTacticalStrategy(self, this);
	//availableStrategies.put(aStrategy.name, aStrategy);

	aStrategy = new CornersTacticalStrategy(self, this);
	availableStrategies.put(aStrategy.name, aStrategy);

	aStrategy = new TronTacticalStrategy(self, this);
	availableStrategies.put(aStrategy.name, aStrategy);

	//aStrategy = new OrbitTacticalStrategy(self, this);
	//availableStrategies.put(aStrategy.name, aStrategy);

	aStrategy = new VictoryTacticalStrategy(self, this);
	availableStrategies.put(aStrategy.name, aStrategy);

	aStrategy = new DisabledTacticalStrategy(self, this);
	availableStrategies.put(aStrategy.name, aStrategy);

	initStrategy = (TacticalStrategy)availableStrategies.get(INITIAL);
	if(initStrategy == null) initStrategy = aStrategy;
	self.currentStrategy = initStrategy;

	// Initialize the hashtables
	double a=0;
	// Start with accrued damage
	accruedDamage.put(DEFAULT, new Double(a));
	accruedDamage.put(ANTIGRAV, new Double(a));
	accruedDamage.put(DODGE_AG, new Double(a));
	accruedDamage.put(TRON, new Double(a));
	accruedDamage.put(CORNERS, new Double(a));
	accruedDamage.put(ORBIT, new Double(a));
	accruedDamage.put(STARTSTOP, new Double(a));
	accruedDamage.put(DODGE, new Double(a));

	// Next, TimeUsed
	timeStrategyUsed.put(DEFAULT, new Double(a));
	timeStrategyUsed.put(ANTIGRAV, new Double(a));
	timeStrategyUsed.put(DODGE_AG, new Double(a));
	timeStrategyUsed.put(TRON, new Double(a));
	timeStrategyUsed.put(CORNERS, new Double(a));
	timeStrategyUsed.put(ORBIT, new Double(a));
	timeStrategyUsed.put(STARTSTOP, new Double(a));
	timeStrategyUsed.put(DODGE, new Double(a));

	// Next, TimeUsed
	lastTimeStrategyStarted.put(DEFAULT, new Double(a));
	lastTimeStrategyStarted.put(ANTIGRAV, new Double(a));
	lastTimeStrategyStarted.put(DODGE_AG, new Double(a));
	lastTimeStrategyStarted.put(TRON, new Double(a));
	lastTimeStrategyStarted.put(CORNERS, new Double(a));
	lastTimeStrategyStarted.put(ORBIT, new Double(a));
	lastTimeStrategyStarted.put(STARTSTOP, new Double(a));
	lastTimeStrategyStarted.put(DODGE, new Double(a));

    }

    /**
     * Change current strategy to newStrategyName.  If this
     * is not found, go back to the INITIAL.
     *
     * @param newStrategyName
     */
    public void setStrategy(String newStrategyName)
    {
	if (newStrategyName != self.currentStrategy.name) {
	    self.out.println("Switching to strategy: " + newStrategyName);
	    TacticalStrategy newStrategy;
	    newStrategy =  (TacticalStrategy)availableStrategies.get(newStrategyName);
	    if(newStrategy == null) newStrategy =  (TacticalStrategy)availableStrategies.get(INITIAL);
	    // Update timeStrategyUsed.
	    Double d = (Double)timeStrategyUsed.remove(self.currentStrategy.name);
	    Double t = (Double)lastTimeStrategyStarted.get(self.currentStrategy.name);
	    if(d != null && t != null) {
		d = new Double(d.doubleValue() + (self.getTime() - t.doubleValue()));

		timeStrategyUsed.put(self.currentStrategy.name, d);
	    }
	    self.currentStrategy = newStrategy;
	    t = (Double)lastTimeStrategyStarted.remove(newStrategy.name);
	    lastTimeStrategyStarted.put(newStrategy.name, new Double(self.getTime()));
	    self.currentStrategy.accruedDamageThisUse = 0;
	    self.currentStrategy.reset();
	}
    }

    /**
     * Reset every strategy.
     */
    public void reset()
    {
	bStartNewRound = true;

	TacticalStrategy aStrategy;
	Enumeration e = availableStrategies.elements();
	while(e.hasMoreElements()) {
	    aStrategy = (TacticalStrategy)e.nextElement();
	    aStrategy.reset();
	}
    }

    /**
     * Reset every strategy.
     */
    public void resetStats()
    {
	if(self.bOneVOne) {
	    for(int i=0;i<oneVoneStrategies.length;i++) {
		Double d = (Double)accruedDamage.remove(oneVoneStrategies[i]);
		if(d != null) {
		    d = new Double(0);
		    accruedDamage.put(oneVoneStrategies[i], d);
		}
		d = (Double)timeStrategyUsed.remove(oneVoneStrategies[i]);
		if(d != null) {
		    d = new Double(0);
		    timeStrategyUsed.put(oneVoneStrategies[i], d);
		}
		d = (Double)lastTimeStrategyStarted.remove(oneVoneStrategies[i]);
		if(d != null) {
		    d = new Double(0);
		    lastTimeStrategyStarted.put(oneVoneStrategies[i], d);
		}
	    }
	}
    }

    /**
     * Initialize with default strategy
     */
    public void setInitialStrategy()
    {
	TacticalStrategy initStrategy;

	if(self.getRoundNum() == 0) {
	    // Set initial strategy
	    if(self.getOthers() >= 2)
		initStrategy = (TacticalStrategy)availableStrategies.get(CORNERS);
	    else
		initStrategy = (TacticalStrategy)availableStrategies.get(INITIAL);
	    if(initStrategy == null) initStrategy = (TacticalStrategy)availableStrategies.get(INITIAL);
	    self.currentStrategy = initStrategy;
	    self.out.println("Initializing first round with strategy: " + self.currentStrategy.name);
	    self.currentStrategy.accruedDamageThisUse = 0;
	} else {
	    if(self.getOthers() >= 2) {
		initStrategy = (TacticalStrategy)availableStrategies.get(CORNERS);
		self.out.println("Initializing current round melee with strategy: " + self.currentStrategy.name);
	    } else {
		if(self.currentStrategy == null || self.currentStrategy.name == VICTORY) {
		    self.currentStrategy = (TacticalStrategy)availableStrategies.get(INITIAL);
		}
		decideStrategyChange();
		self.out.println("Initializing current round 1v1 with strategy: " + self.currentStrategy.name);
	    }
	}
	bStartNewRound = false;
    }

    /**
     * getStrategyEvasiveness
     *
     * Computes damageAccrued over timeStrategyUsed
     *
     *  @return rate of damage sustained
     */
    public final double getStrategyEvasiveness(String name)
    {
	Double dTimeStrategyUsed = (Double)timeStrategyUsed.get(name);
	Double dAccruedDamage = (Double)accruedDamage.get(name);
	Double dLastTimeStrategyStarted = (Double)lastTimeStrategyStarted.get(name);

	if(dTimeStrategyUsed == null || dAccruedDamage == null ||
	   dLastTimeStrategyStarted == null) return Double.NaN;

	if((dTimeStrategyUsed.doubleValue() > MIN_STRAT_TIME && name != self.currentStrategy.name) ||
	   (name == self.currentStrategy.name &&
	    dTimeStrategyUsed.doubleValue() + (self.getTime() - dLastTimeStrategyStarted.doubleValue()) >
	    MIN_STRAT_TIME)) {
	    if(name == self.currentStrategy.name) {
		return 100 * dAccruedDamage.doubleValue() / 
		    (dTimeStrategyUsed.doubleValue() + (self.getTime() - dLastTimeStrategyStarted.doubleValue()));
	    } else {
		return 100 * dAccruedDamage.doubleValue() / dTimeStrategyUsed.doubleValue();
	    }
	}
	else
	    return 0;
    }

    /**
     * Decides when to change strategies
     */
    public void decideStrategyChange()
    {
	int othersLeft = self.getOthers();

	Double t = (Double)lastTimeStrategyStarted.get(self.currentStrategy.name);
	double lastTimeStarted = t.doubleValue();
	if((self.getTime() - lastTimeStarted < MIN_STRAT_TIME) && !bStartNewRound)
	    return;

	if(othersLeft == 0 && self.inboundBulletList.size() == 0 ) {
	    setStrategy(VICTORY);
	} else if(othersLeft == 0 && self.inboundBulletList.size() != 0) {
	    setStrategy(ANTIGRAV);
	} else { // Harder to decide.
	    if(othersLeft == 1) { // 1v1
		if(self.currentTarget == null || self.currentTarget.isAlive == false) {
		    self.setTurnRadarRightRadians(MathHelper.TWO_PI);
		    self.execute();
		}
		// Give current strategy a chance
		if(self.currentStrategy.accruedDamageThisUse < 13 &&
		   self.currentStrategy.name != CORNERS)
		    return;

		// Pick from 1v1 array
		String newStrategyName = "";
		double bestEvasiveness = Double.MAX_VALUE;
		double curEvasiveness = Double.MAX_VALUE;
		for(int i=0;i<oneVoneStrategies.length;i++) {
		    curEvasiveness = getStrategyEvasiveness(oneVoneStrategies[i]);
		    if(curEvasiveness < bestEvasiveness) {
			newStrategyName = oneVoneStrategies[i];
			bestEvasiveness = curEvasiveness;
		    }
		}
		setStrategy(newStrategyName);
	    } else {  // melee
		// Really prefer CORNERS
		if(self.currentStrategy.name == CORNERS &&
		   (self.currentStrategy.accruedDamageThisUse < 30 ||
		    self.getEnergy() > 65))
		    return;

		// Give current strategy a chance
		if(self.currentStrategy.accruedDamageThisUse < 13)
		    return;

		// Pick from melee array
		String newStrategyName = "";
		double bestEvasiveness = Double.MAX_VALUE;
		double curEvasiveness = Double.MAX_VALUE;
		for(int i=0;i<meleeStrategies.length;i++) {
		    curEvasiveness = getStrategyEvasiveness(meleeStrategies[i]);
		    /*self.out.println("Evasiveness for " + meleeStrategies[i] +
		      " is " + curEvasiveness);*/
		    if(curEvasiveness < bestEvasiveness) {
			newStrategyName = meleeStrategies[i];
			bestEvasiveness = curEvasiveness;
		    }
		}
		setStrategy(newStrategyName);
	    }
	}
    }
}
