package nat.wave;

import java.io.Serializable;
import java.util.ArrayList;

import nat.utils.M;
import nat.utils.Point;
import nat.utils.Range;

import robocode.util.Utils;

public final class Wave implements Serializable {
	private static final long serialVersionUID = 6826380621990735922L;
	public static final int INTERSECTION = 3, PASSED = 1, ACTIVE = 2;

	// Wave state
	public final double bulletVelocity;
	public final double bulletPower;
	public Point fireLocation;
	public final double headOnAngle;
	public final double lateralDirection;
	public final double fireAngle;
	public long fireTime;
	public final double maximumEscapedAngle;

	// Wave status
	public int status;
	public double distanceTraveled;
	public Range passedRange = null;
	public Point passedLocation = null;

	public boolean isHit = false;
	public boolean isHitBullet = false;
	public Point hitLocation = null;
	public boolean isFiringWave;

	// unrelated information
	public float[] paintBucket;

	// Clone constructor
	private Wave(double distanceTraveled, double bulletVelocity,
			Point fireLocation, double mea, double headOnAngle,
			double lateralDirection, double bulletPower, double fireAngle,
			long fireTime, int status, boolean isFiringWave, boolean isHit,
			boolean isHitBullet, Point hitLocation, Range passedRange, Point passedLocation,
			float[] paintBucket) {
		this.distanceTraveled = distanceTraveled;
		this.bulletVelocity = bulletVelocity;
		this.fireLocation = fireLocation;
		this.maximumEscapedAngle = mea;
		this.headOnAngle = headOnAngle;
		this.lateralDirection = lateralDirection;
		this.bulletPower = bulletPower;
		this.fireAngle = this.headOnAngle;
		this.fireTime = fireTime;
		this.status = status;
		this.isFiringWave = isFiringWave;
		this.isHit = isHit;
		this.isHitBullet = isHitBullet;
		this.passedRange = passedRange;
		this.paintBucket = paintBucket;
	}

	public Wave(Point center, double absBearing, int dir, double bulletPower, long time) {
		this.distanceTraveled = 0;
		this.bulletVelocity = 20d - 3d * bulletPower;
		this.fireLocation = center;
		this.headOnAngle = absBearing;
		this.lateralDirection = dir;
		this.bulletPower = bulletPower;
		this.fireAngle = this.headOnAngle;
		this.fireTime = time;
		this.status = ACTIVE;
		this.isFiringWave = false;
		this.isHit = false;
		this.isHitBullet = false;
		this.passedRange = null;
		this.paintBucket = null;
		this.maximumEscapedAngle = M.asin(M.MAX_VELOCITY / this.bulletVelocity);
	}

	public final Wave clone() {
		return new Wave(distanceTraveled, bulletVelocity, fireLocation, maximumEscapedAngle,
				headOnAngle, lateralDirection, bulletPower,
				fireAngle, fireTime, status, isFiringWave, isHit,
				isHitBullet, hitLocation, passedRange, passedLocation, paintBucket);
	}

	public final double getRelativeAngle(double angle) {
		return Utils.normalRelativeAngle(angle - this.headOnAngle) * lateralDirection;
	}

	public final double getGuessFactor(double angle) {
		return getRelativeAngle(angle) / maximumEscapedAngle;
	}

	public final double getAngle(double GF) {
		return M
				.normalAbsoluteAngle((GF * maximumEscapedAngle * lateralDirection)
						+ headOnAngle);
	}

	public final void advanceWave() {
		distanceTraveled += bulletVelocity;
	}

	public final void update(Point enemyLocation) {
		advanceWave();
		status = getStatus(enemyLocation);

		// get precise intersection range
		if (status == INTERSECTION) {
			Range currentIntersection = getCurrentIntersectionRange(enemyLocation);
			if (passedRange == null)
				passedRange = currentIntersection;
			else
				passedRange = passedRange.grow(currentIntersection);
		}

		// get static hit position
		if (M.sqr(distanceTraveled) > enemyLocation.distanceSq(fireLocation)
				&& hitLocation == null) {
			passedLocation = enemyLocation.clone();
		}
	}

	// /////////////////////////////////
	// Precise Wave Intersection code
	// CREDIT: Skilgannon
	private final int getStatus(Point botLocation) {
		double[] enemyCorner = new double[] {
				fireLocation.distanceSq(botLocation.getX() - 18, botLocation
						.getY() + 18),
				fireLocation.distanceSq(botLocation.getX() + 18, botLocation
						.getY() + 18),
				fireLocation.distanceSq(botLocation.getX() + 18, botLocation
						.getY() - 18),
				fireLocation.distanceSq(botLocation.getX() - 18, botLocation
						.getY() - 18) };

		// score, for speed
		int score = 0;

		// reached
		if (M.sqr(distanceTraveled) > M.min(enemyCorner[0], enemyCorner[1],
				enemyCorner[2], enemyCorner[3]))
			score++;

		// passed
		if (M.sqr(distanceTraveled - bulletVelocity) < M.max(enemyCorner[0],
				enemyCorner[1], enemyCorner[2], enemyCorner[3]))
			score += 2;

		return score;
	}

	private final Range getCurrentIntersectionRange(Point botLocation) {
		double[] robotYBounds = new double[] { botLocation.getY() - 18,
				botLocation.getY() + 18 };
		double[] robotXBounds = new double[] { botLocation.getX() - 18,
				botLocation.getX() + 18 };

		double[] radius = new double[] { distanceTraveled,
				distanceTraveled - bulletVelocity };

		ArrayList<Point> intersectPoints = new ArrayList<Point>();

		// loop through each robot bound
		for (int i = 0; i < 2; i++) {
			for (int j = 0; j < 2; j++) {

				// get the vertical intersection point
				Point[] vertIntersectPoints = getVerticalIntersectionPoints(
						fireLocation.getX(), fireLocation.getY(), radius[i],
						robotXBounds[j]);

				for (int k = 0; k < vertIntersectPoints.length; k++) {
					if (M
							.inRange(vertIntersectPoints[k].getY(),
									robotYBounds))
						intersectPoints.add(vertIntersectPoints[k]);
				}

				// horizontal intersection
				Point[] horizIntersectPoints = getHorizontalIntersectionPoints(
						fireLocation.getX(), fireLocation.getY(), radius[i],
						robotYBounds[j]);

				for (int k = 0; k < horizIntersectPoints.length; k++) {
					if (M.inRange(horizIntersectPoints[k].getX(),
							robotXBounds))
						intersectPoints.add(horizIntersectPoints[k]);
				}

				// corner
				Point testCorner = new Point(robotXBounds[i],
						robotYBounds[j]);
				double distSq = testCorner.distanceSq(fireLocation);
				if (distSq <= M.sqr(radius[0]) && distSq > M.sqr(radius[1]))
					intersectPoints.add(testCorner);
			}
		}

		double lowerGF = 1;
		double upperGF = -1;

		for (Point p : intersectPoints) {
			double currentGF = getGuessFactor(M.getAngle(fireLocation, p));
			lowerGF = M.min(lowerGF, currentGF);
			upperGF = M.max(upperGF, currentGF);
		}

		return new Range(lowerGF, upperGF);
	}

	private static Point[] getVerticalIntersectionPoints(double centerX,
			double centerY, double r, double intersectX) {
		double deltaX = centerX - intersectX;
		double sqrtVal = r * r - deltaX * deltaX;

		if (sqrtVal < 0)
			return new Point[] {};

		sqrtVal = M.sqrt(sqrtVal);

		return new Point[] {
				new Point(intersectX, centerY + sqrtVal),
				new Point(intersectX, centerY - sqrtVal) };
	}

	private final static Point[] getHorizontalIntersectionPoints(
			double centerX, double centerY, double r, double intersectY) {
		double deltaY = centerY - intersectY;
		double sqrtVal = r * r - deltaY * deltaY;

		if (sqrtVal < 0)
			return new Point[] {};

		sqrtVal = M.sqrt(sqrtVal);

		return new Point[] {
				new Point(centerX + sqrtVal, intersectY),
				new Point(centerX - sqrtVal, intersectY) };
	}
	// End Precise Wave intersection code
	// ////////////////////////////////////
}
