package sgp;
import robocode.*;
import java.util.*;
import java.awt.Color;

/**
 * Strategy - a class by Simon Parker
 */
public class Strategy
{
	private AdvancedRobot robot;
	protected StrategyManager strategyManager;
	protected DrunkenStrategyManager dStrategyManager;

	protected final int NUM_POSITIONS = 4;
	protected double[] velocity = new double[NUM_POSITIONS];
	protected double[] turnRight = new double[NUM_POSITIONS];
	protected double[] goodness = new double[NUM_POSITIONS];
	protected Coordinate[] nextPosition = new Coordinate[NUM_POSITIONS];
	protected double[] nextVelocity = new double[NUM_POSITIONS];
	protected double[] nextHeading = new double[NUM_POSITIONS];
	
	private final double MAX_VELOCITY = 8;
	private final double MAX_ANGULAR_VELOCITY_DEG_PER_FRAME = 10;


	public Strategy (StrategyManager theStrategyManager)
	{
		strategyManager = theStrategyManager;
		init();
	}

	public Strategy (DrunkenStrategyManager theStrategyManager)
	{
		dStrategyManager = theStrategyManager;
		init();
	}
	
	private void init()
	{
		robot = Environment.getRobot();
		velocity[0] = 1.0;		turnRight[0] = 1.0;
		velocity[1] = -2.0;		turnRight[1] = 1.0;
		velocity[2] = 1.0;		turnRight[2] = -1.0;
		velocity[3] = -2.0;		turnRight[3] = -1.0;
		
		for (int i = 0; i < NUM_POSITIONS; i++)
		{
			nextPosition[i] = new Coordinate();
			nextVelocity[i] = 0;
			nextHeading[i] = 0;
		}
	}

	public void startTurn()
	{

	}

	public void setScan(){} //control the radar motion
	public void setGunRotation(){} //control the gun motion and fires if neccessary
	public void endTurn(){}
	public void onScannedRobot(ScannedRobotEvent e){}
	public void onHitByBullet(HitByBulletEvent e){}
	public void onDeath(DeathEvent event){}
	public void onHitWall(HitWallEvent e){}
	public void onHitRobot(HitRobotEvent e) {}

	// Helper function
	static public double normalRelativeAngle(double angle)
	{
		if (angle > -180 && angle <= 180)
			return angle;
		double fixedAngle = angle;
		while (fixedAngle <= -180)
			fixedAngle += 360;
		while (fixedAngle > 180)
			fixedAngle -= 360;
		return fixedAngle;
	}

	public void onRobotDeath(RobotDeathEvent e)
	{
		//Victory dance
		if (robot.getOthers() == 0)
		{
			robot.setAhead(0);
			robot.turnRight(36000);
		}
	}

	public void reset()
	{
		robot = Environment.getRobot();
	}


	public void onBulletHit(BulletHitEvent event)
	{
	}

	public void onBulletMissed(BulletMissedEvent event)
	{
	}

	public void onWin(WinEvent event)
	{
	}

	public void setMovement() //sets the turning angle and the ahead/back movement
	{
		//compare the 4 positions for ahead/back left/right

		int maxIndex = getBestMovementIndex(Environment.getRobotPosition(), Environment.getHeading(), Environment.getVelocity());

		if (velocity[maxIndex] > 0)
		{
			robot.setAhead(100);
		}
		else
		{
			robot.setBack(100);
		}

		robot.setTurnRight(turnRight[maxIndex] * 360);
	}
	
	protected int getBestMovementIndex(Coordinate currentPosition, double currentHeading, double currentVelocity)
	{
		int maxIndex = 0;

		for (int i = 0; i < NUM_POSITIONS; i++)
		{
			double nextDeltaHeading = (MAX_ANGULAR_VELOCITY_DEG_PER_FRAME - 0.75 * currentVelocity) * turnRight[i];
			nextVelocity[i] = currentVelocity + velocity[i];
			nextVelocity[i] = Math.max(nextVelocity[i], -MAX_VELOCITY);
			nextVelocity[i] = Math.min(nextVelocity[i], MAX_VELOCITY);
			
			nextHeading[i] = currentHeading + nextDeltaHeading;

			nextPosition[i].set
			(
				currentPosition.getNextPosition
				(
					currentHeading,
					nextVelocity[i],
					nextDeltaHeading
				)
			);
			goodness[i] = evaluateGoodness(nextPosition[i], nextHeading[i], nextVelocity[i]);
			if (goodness[i] > goodness[maxIndex]) maxIndex = i;
		}
		
		return maxIndex;
	}

	public double evaluateGoodness(Coordinate robotPosition, double robotHeading, double robotVelocity)
	{
		return Math.random();
	}

	public void onMessageReceived(MessageEvent e)
	{
	}

	static public double evaluateWallGoodness(Coordinate robotPosition)
	{
		int numOthers = Environment.getRobot().getOthers();
		double WALL_DISTANCE_WEIGHT = numOthers * numOthers * 1000.0;
		double WALL_DEFLECTION_DISTANCE = 45.0;
		double CORNER_RADIUS = 150;

		double xDistance, yDistance, wallDistance;

		xDistance = Math.min(robotPosition.x, (Environment.battleFieldWidth - robotPosition.x));
		yDistance = Math.min(robotPosition.y, (Environment.battleFieldHeight - robotPosition.y));

		if ((xDistance < CORNER_RADIUS) && (yDistance < CORNER_RADIUS))
		{
			//do corner radius calculation of wall distance
			Coordinate cornerCentre = new Coordinate(CORNER_RADIUS, CORNER_RADIUS);
			Coordinate normalisedRobotPosition = new Coordinate(xDistance, yDistance);
			wallDistance = CORNER_RADIUS - normalisedRobotPosition.distanceFrom(cornerCentre);
		}
		else
		{
			wallDistance = Math.min(xDistance, yDistance);
		}

		double goodness = 0;

		if (wallDistance < WALL_DEFLECTION_DISTANCE)
		{
			goodness = -WALL_DISTANCE_WEIGHT * (WALL_DEFLECTION_DISTANCE - wallDistance);
		}
		return goodness;
	}

	static public Coordinate limitToBattleField(Coordinate position)
	{
		final double ROBOT_RADIUS = 20;
		Coordinate result = new Coordinate(position);

		result.x = Math.max(result.x, ROBOT_RADIUS);
		result.y = Math.max(result.y, ROBOT_RADIUS);
		result.x = Math.min(result.x, Environment.battleFieldWidth - ROBOT_RADIUS);
		result.y = Math.min(result.y, Environment.battleFieldHeight - ROBOT_RADIUS);

		return result;
	}
	
	static public double getPowerFromDamage(double damage)
	{
		if (damage <= 4.0)
		{
			return damage / 4.0;
		}
		else
		{
			return (damage + 2.0) / 6.0;
		}
	}
	
	static public double getDamageFromPower(double power)
	{
		double damage = 4.0 * power;
		if (power > 1.0) damage += 2.0 * (power - 1.0);
		return damage;
	}

}
