package staticline.whiskey.movement;

import java.awt.geom.Point2D;
import java.util.ArrayList;

import robocode.AdvancedRobot;
import robocode.Condition;
import robocode.HitByBulletEvent;
import robocode.HitWallEvent;
import robocode.ScannedRobotEvent;
import robocode.util.Utils;
import staticline.IRobotManager;
import staticline.whiskey.AbstractWhiskeyManager;

/**
 * Wave surfing movement module
 * based on the BasicWaveSurfer with code from Komarious by <b>Voidious</b>.
 * 
 * @author "Carsten Witzke"
 */
public class WaveSurfing extends AbstractWhiskeyManager implements IRobotManager{
	//wave surfing vars
    private static final int BINS = 47;
    private static final double surfStats[] = new double[BINS];
    private static Point2D.Double myLocation;     // our bot's location
    private static Point2D.Double enemyLocation;  // enemy bot's location
    private static ArrayList<Wave> waves;
    private ArrayList<Integer> surfDirections;
    private ArrayList<Double> surfAbsBearings;
    private static final double WALL_STICK = 160;
    
	
	public WaveSurfing(AdvancedRobot b) {
		super(b);
	}
	
	public void initialize(){
		// wave surfing
        this.bot.fieldRect = new java.awt.geom.Rectangle2D.Double(
        		18, 18, this.bot.getBattleFieldWidth()-36, this.bot.getBattleFieldHeight()-36);
        waves = new ArrayList<Wave>();
        this.surfDirections = new ArrayList<Integer>();
        this.surfAbsBearings = new ArrayList<Double>();
	}
	
	public void onScannedRobot(ScannedRobotEvent e) {
		//basic wave surfing - based on BasicSurfer this.bot
		myLocation = new Point2D.Double(this.bot.getX(), this.bot.getY());
        
        double lateralVelocity = this.bot.getVelocity()*Math.sin(e.getBearingRadians());
        double absBearing = e.getBearingRadians() + this.bot.getHeadingRadians();

        this.bot.setTurnRadarRightRadians(Utils.normalRelativeAngle(absBearing
            - this.bot.getRadarHeadingRadians()) * 2);
        
        this.surfDirections.add(0, new Integer((lateralVelocity >= 0) ? 1 : -1));
        this.surfAbsBearings.add(0, new Double(absBearing + Math.PI));

        double bulletPower = this.bot.t_energy_old - e.getEnergy();
        if (bulletPower < 3.01 && bulletPower > 0.09
            && this.surfDirections.size() > 2) {
            Wave ew = new Wave();
            ew.fireTime = this.bot.getTime() - 1;
            ew.bulletVelocity = bulletVelocity(bulletPower);
            ew.distanceTraveled = bulletVelocity(bulletPower);
            ew.direction = ((Integer)this.surfDirections.get(2)).intValue();
            ew.directAngle = ((Double)this.surfAbsBearings.get(2)).doubleValue();
            ew.fireLocation = (Point2D.Double)enemyLocation.clone(); // last tick
            //this.bot.addCustomEvent(ew);
            waves.add(ew);
        }

        this.bot.t_energy_old = e.getEnergy();

        // update after Wave detection, because that needs the previous
        // enemy location as the source of the wave
        enemyLocation = project(myLocation, absBearing, e.getDistance());
        
        updateWaves();
        doSurfing();
	}
	
	public void onHitWall(HitWallEvent e){
    	this.bot.move = Math.random()*200;
    	this.bot.setTurnRightRadians((this.bot.move%2 == 0)?e.getBearingRadians()+this.bot.move*0.04:e.getBearingRadians()-this.bot.move*0.04);
        if(this.bot.getDistanceRemaining() > 0){
        	this.bot.setBack(this.bot.move);
        }else{
        	this.bot.setAhead(this.bot.move);
        }
    }
	
	public void updateWaves() {
        for (int x = 0; x < waves.size(); x++) {
            Wave ew = (Wave)waves.get(x);

            ew.distanceTraveled = (this.bot.getTime() - ew.fireTime) * ew.bulletVelocity;
            if (ew.distanceTraveled > myLocation.distance(ew.fireLocation) + 50) {
            	waves.remove(x);
                x--;
            }
        }
    }
    
    public Wave getClosestSurfableWave() {
        double closestDistance = Double.MAX_VALUE;
        Wave surfWave = null;

        for (int x = 0; x < waves.size(); x++) {
            Wave ew = (Wave)waves.get(x);
            double distance = myLocation.distance(ew.fireLocation) - ew.distanceTraveled;

            if (distance > ew.bulletVelocity && distance < closestDistance) {
                surfWave = ew;
                closestDistance = distance;
            }
        }
        return surfWave;
    }
    
    // Given the Wave that the bullet was on, and the point where we
    // were hit, calculate the index into our stat array for that factor.
    public static int getFactorIndex(Wave ew, Point2D.Double targetLocation) {
        double offsetAngle = (absoluteBearing(ew.fireLocation, targetLocation) - ew.directAngle);
        double factor = Utils.normalRelativeAngle(offsetAngle) / maxEscapeAngle(ew.bulletVelocity) * ew.direction;
        
        return (int)limit(0, (factor * ((BINS - 1) / 2)) + ((BINS - 1) / 2), BINS - 1);
    }
    
    // Given the Wave that the bullet was on, and the point where we
    // were hit, update our stat array to reflect the danger in that area.
    public static void logHit(Wave w, Point2D.Double targetLocation, double rollingDepth) {
    	int index = getFactorIndex(w, targetLocation);
    	 
        for (int x = 0; x < BINS; x++) {
            // for the spot bin that we were hit on, add 1;
            // for the bins next to it, add 1 / 2;
            // the next one, add 1 / 5; and so on...
            surfStats[x] += 1.0 / (Math.pow(index - x, 2) + 1);
        }
        
    	/*
    	for (int x = BINS-1; x >= 0; x--) {
            w.waveGuessFactors[x] = ((w.waveGuessFactors[x] * rollingDepth)
                + ((1 + w.weight) / (Math.pow(x - getFactorIndex(w, targetLocation), 2) + 1)))
                / (rollingDepth + 1 + w.weight);
        }
        */
    }
    
    public void onHitByBullet(HitByBulletEvent e) {
        // If the _Waves collection is empty, we must have missed the
        // detection of this wave somehow.
        if (!waves.isEmpty()) {
            Point2D.Double hitBulletLocation = new Point2D.Double(
                e.getBullet().getX(), e.getBullet().getY());
            Wave hitWave = null;

            // look through the Waves, and find one that could've hit us.
            for (int x = 0; x < waves.size(); x++) {
                Wave ew = (Wave)waves.get(x);

                if (Math.abs(ew.distanceTraveled -
                		myLocation.distance(ew.fireLocation)) < 50
                    && Math.abs(bulletVelocity(e.getBullet().getPower()) 
                        - ew.bulletVelocity) < 0.001) {
                    hitWave = ew;
                    break;
                }
            }

            if (hitWave != null) {
                logHit(hitWave, hitBulletLocation, 0.85);

                // We can remove this wave now, of course.
                waves.remove(waves.lastIndexOf(hitWave));
            }
        }else{
        	System.out.println("Ouch, I missed a wave :´(");
        }
    }
    
    public Point2D.Double predictPosition(Wave surfWave, int direction) {
        Point2D.Double predictedPosition = (Point2D.Double)myLocation.clone();
        double predictedVelocity = this.bot.getVelocity();
        double predictedHeading = this.bot.getHeadingRadians();
        double maxTurning, moveAngle, moveDir;

        int counter = 0; // number of ticks in the future
        boolean intercepted = false;

        do {
            moveAngle =
                wallSmoothing(predictedPosition, absoluteBearing(surfWave.fireLocation,
                predictedPosition) + (direction * (Math.PI/2)), direction)
                - predictedHeading;
            moveDir = 1;

            if(Math.cos(moveAngle) < 0) {
                moveAngle += Math.PI;
                moveDir = -1;
            }

            moveAngle = Utils.normalRelativeAngle(moveAngle);

            // maxTurning is built in like this, you can't turn more then this in one tick
            maxTurning = Math.PI/720d*(40d - 3d*Math.abs(predictedVelocity));
            predictedHeading = Utils.normalRelativeAngle(predictedHeading
            	+ limit(-maxTurning, moveAngle, maxTurning));

            // this one is nice ;). if predictedVelocity and moveDir have
            // different signs you want to breack down
            // otherwise you want to accelerate (look at the factor "2")
            predictedVelocity += (predictedVelocity * moveDir < 0 ? 2*moveDir : moveDir);
            predictedVelocity = limit(-8, predictedVelocity, 8);

            // calculate the new predicted position
            predictedPosition = project(predictedPosition, predictedHeading, predictedVelocity);

            counter++;

            if (predictedPosition.distance(surfWave.fireLocation) <
                surfWave.distanceTraveled + (counter * surfWave.bulletVelocity)
                + surfWave.bulletVelocity) {
                intercepted = true;
            }
        } while(!intercepted && counter < 500);

        return predictedPosition;
    }
    
    public double checkDanger(Wave surfWave, int direction) {
        int index = getFactorIndex(surfWave, predictPosition(surfWave, direction));
        return surfStats[index];
    }

    public void doSurfing() {
        Wave surfWave = getClosestSurfableWave();

        if (surfWave == null) { return; }

        //wall smoothing
        double goAngle = absoluteBearing(surfWave.fireLocation, WaveSurfing.myLocation);
        double dangerLeft = checkDanger(surfWave, -1);
        double dangerRight = checkDanger(surfWave, 1);
        if (dangerLeft < dangerRight) {
            goAngle = wallSmoothing(myLocation, goAngle - (Math.PI/2), -1);
        } else {
            goAngle = wallSmoothing(myLocation, goAngle + (Math.PI/2), 1);
        }

        /*
        double goAngle = 0;// = absoluteBearing(surfWave.fireLocation, this.myLocation);
        try {
            goAngle = surfWave.absoluteBearing(myLocation) +
                1.25 * (this.direction = (sign(checkDanger(surfWave,-1) - checkDanger(surfWave,1))));
        } catch (Exception ex) { 
        	ex.printStackTrace();
        }
        this.bot.setTurnRightRadians(Math.tan(goAngle = 
            wallSmoothing(myLocation, goAngle, this.direction) - this.bot.getHeadingRadians()));
        */
        
        setBackAsFront(this.bot, goAngle);
    }
    
    public double wallSmoothing(Point2D.Double botLocation, double angle, int orientation) {
    	double a = angle;
        while (!this.bot.fieldRect.contains(project(botLocation, a, WALL_STICK))) {
        	a += orientation*0.05;
        }
        return a;
    }

    
    /*
     * 
     * Static helper methods
     * 
     */
    public static Point2D.Double project(Point2D.Double sourceLocation, double angle, double length) {
        return new Point2D.Double(
        		sourceLocation.x + Math.sin(angle) * length, 
        		sourceLocation.y + Math.cos(angle) * length
        		);
    }
    
    private static int sign(double d) {
        if (d < 0) { return -1; }
        return 1;
    }

    public static double absoluteBearing(Point2D.Double source, Point2D.Double target) {
        return Math.atan2(target.x - source.x, target.y - source.y);
    }

    public static double limit(double min, double value, double max) {
        return Math.max(min, Math.min(value, max));
    }

    public static double bulletVelocity(double power) {
        return (20.0 - (3.0*power));
    }

    public static double maxEscapeAngle(double velocity) {
        return Math.asin(8.0/velocity);
    }

    public static void setBackAsFront(AdvancedRobot robot, double goAngle) {
        double angle =
            Utils.normalRelativeAngle(goAngle - robot.getHeadingRadians());
        if (Math.abs(angle) > (Math.PI/2)) {
            if (angle < 0) {
                robot.setTurnRightRadians(Math.PI + angle);
            } else {
                robot.setTurnLeftRadians(Math.PI - angle);
            }
            robot.setBack(100);
        } else {
            if (angle < 0) {
                robot.setTurnLeftRadians(-1*angle);
           } else {
                robot.setTurnRightRadians(angle);
           }
            robot.setAhead(100);
        }
    }
    
    // ENEMY WAVE
    static class Wave extends Condition {
        Point2D.Double fireLocation;
        double[] waveGuessFactors;
        long fireTime;
        double bulletVelocity, directAngle, distanceTraveled;
        int direction, weight;
 
        public double distanceToPoint(Point2D.Double p) {
            return fireLocation.distance(p);
        }
 
        public double absoluteBearing(Point2D.Double target) {
            return Math.atan2(target.x - fireLocation.x, target.y - fireLocation.y);
        }
 
        public boolean test() {
            if (distanceToPoint(WaveSurfing.waves.contains(this)?
            		WaveSurfing.myLocation:WaveSurfing.enemyLocation)
                    <= (distanceTraveled+=bulletVelocity) + (2 * bulletVelocity)) {
//            logHit((Wave)_Waves.removeFirst(),Komarious._myLocation, 500);
                if (!WaveSurfing.waves.remove(this)) {
                	WaveSurfing.logHit(this, WaveSurfing.enemyLocation, 600);
                }
 
                return true;
            }
            return false;
        }
    }
}
