package SHAM;

import robocode.*;
import robocode.util.Utils;
import java.awt.geom.*; // for Point2D's
import java.lang.*; // for Double and Integer objects
import java.util.ArrayList; // for collection of waves


/**
 * SHAM.WOW created by Steven Hatton and Aditya Majumdar Period 3, JAVA
 * Programming Due: 06/05/09 Robocode, JAVA, Second Semester Final Project
 * 
 *A fairly simplistic style of wave surfing. Based off of the BasicSurfer from
 * the robowiki. Stores away information from enemy bullets into list arrays. To
 * be used and calculated later. credit to:
 * http://robowiki.net/w/index.php?title=Wave_Surfing_Tutorial
 */
public class WaveSurfing
{

    public static int BINS = 47;

    public static double _surfStats[] = new double[BINS];

    public Point2D.Double _myLocation; // our location, of course

    public Point2D.Double _enemyLocation; // our enemy's location, of course

    public ArrayList _enemyWaves; // these will be our 'surfable' waves which we
                                  // will get data from

    public ArrayList _surfDirections; // will store the direction of our bot in
                                      // relation to that

    // of the enemy - pretty simple - will be somewhere CW or CCW, will be from
    // 2 ticks before

    public ArrayList _surfAbsBearings; // holds past absolute bearings of the
                                       // enemy

    public static double _oppEnergy = 100.0; // we need to start with a
                                             // threshold enemy energy level

    // to detect later drops in this energy

    private AdvancedRobot _robot;


    public WaveSurfing( AdvancedRobot robot )
    {
        _robot = robot;
    }

    // Creates a "boundary" rectangle for the purposes of wallsmoothing
    // leaves enough buffer room so tanks should not hit walls very often
    public static Rectangle2D.Double _fieldRect = new java.awt.geom.Rectangle2D.Double( 18,
        18,
        764,
        564 );

    public static double WALL_STICK = 160;


    public void run()
    {
        _enemyWaves = new ArrayList();
        _surfDirections = new ArrayList();
        _surfAbsBearings = new ArrayList();
    }


    public void onScannedRobot( ScannedRobotEvent e )
    {
        _myLocation = new Point2D.Double( _robot.getX(), _robot.getY() );

        // robot's initial velocity
        // figure out which direction they're moving
        double lateralVelocity = _robot.getVelocity()
            * Math.sin( e.getBearingRadians() );
        // robot's initial absolute bearing
        double absBearing = e.getBearingRadians() + _robot.getHeadingRadians();

        // adding information collected about the enemy to our list arrays
        _surfDirections.add( 0, new Integer( ( lateralVelocity >= 0 ) ? 1 : -1 ) );
        _surfAbsBearings.add( 0, new Double( absBearing + Math.PI ) );

        // this figures out how powerful the bullet fired was
        double bulletPower = _oppEnergy - e.getEnergy();

        // enter code if enough information is stored and met
        if ( bulletPower < 3.01 && bulletPower > 0.09
            && _surfDirections.size() > 2 )
        {
            EnemyWave wave = new EnemyWave();
            wave.fireTime = _robot.getTime() - 1; // subtracts a tick to account
                                                  // for gun turning
            wave.bulletVelocity = bulletVelocity( bulletPower );
            wave.distanceTraveled = bulletVelocity( bulletPower );
            wave.direction = ( (Integer)_surfDirections.get( 2 ) ).intValue();
            wave.directAngle = ( (Double)_surfAbsBearings.get( 2 ) ).doubleValue();
            wave.fireLocation = (Point2D.Double)_enemyLocation.clone();
            // last tick
            // creates duplicate object to be stored in array
            _enemyWaves.add( wave );
        }

        _oppEnergy = e.getEnergy();

        // update after EnemyWave detection, because that needs the previous
        // enemy location as the source of the wave
        _enemyLocation = project( _myLocation, absBearing, e.getDistance() );

        updateWaves();
        Surf();
    }


    public void updateWaves()
    {
        for ( int x = 0; x < _enemyWaves.size(); x++ )
        {
            EnemyWave wave = (EnemyWave)_enemyWaves.get( x );

            // calculate how long the wave has been deployed
            wave.distanceTraveled = ( _robot.getTime() - wave.fireTime )
                * wave.bulletVelocity;
            // adding 50 gives some extra buffer time because the bullet may
            // still be around
            // after that, the wave is removed and we look to newer and closer
            // ones
            if ( wave.distanceTraveled > _myLocation.distance( wave.fireLocation ) + 50 )
            {
                _enemyWaves.remove( x );
                x--;
            }
        }
    }


    public EnemyWave getClosestSurfableWave()
    {
        double closestDistance = 50000; // any big number is suitable for these
                                        // purposes
        // it'll go off the screen at this size anyway
        EnemyWave surfWave = null;

        for ( int x = 0; x < _enemyWaves.size(); x++ )
        {
            // get the wave!
            EnemyWave ew = (EnemyWave)_enemyWaves.get( x );
            // how far away is it? based on time and where it was fired from
            double distance = _myLocation.distance( ew.fireLocation )
                - ew.distanceTraveled;

            if ( distance > ew.bulletVelocity && distance < closestDistance )
            {
                surfWave = ew;
                closestDistance = distance;
            }
        }
        // essentially, this will have looped thru, looking for waves that get
        // closer and closer
        // this is of course most beneficial since when a wave passes us, we
        // don't really care
        // about the bullet hit we might take from that
        // instead, we would want to focus our attention on the next wave +
        // bullet

        return surfWave;
    }


    public static int getFactorIndex(
        EnemyWave ew,
        Point2D.Double targetLocation )
    {
        // this offset angle basically gives us the current location of the wave
        // in relation to us less
        // the original angle fired at us

        double offsetAngle = ( absoluteBearing( ew.fireLocation, targetLocation ) - ew.directAngle );

        // a guessfactor of sorts, may be + or - depending on our direction
        double factor = Utils.normalRelativeAngle( offsetAngle )
            / maxEscapeAngle( ew.bulletVelocity ) * ew.direction;

        // basically find the middle bin and that would be 0
        // then you have + and - on either side
        // but when adding something into an array, it must start at 0
        // so we add half of the list on as well to get 0 as the minimum
        return (int)limit( 0, ( factor * ( ( BINS - 1 ) / 2 ) )
            + ( ( BINS - 1 ) / 2 ), BINS - 1 );
    }


    public void logHit( EnemyWave ew, Point2D.Double targetLocation )
    {
        int index = getFactorIndex( ew, targetLocation );

        for ( int x = 0; x < BINS; x++ )
        {
            _surfStats[x] += 1.0 / ( Math.pow( index - x, 2 ) + 1 );
        }
    }


    public void onHitByBullet( HitByBulletEvent e )
    {
        // If the _enemyWaves collection is empty, we must have missed the
        // detection of this wave somehow.
        if ( !_enemyWaves.isEmpty() )
        {
            Point2D.Double hitBulletLocation = new Point2D.Double( e.getBullet()
                .getX(),
                e.getBullet().getY() );
            EnemyWave hitWave = null;

            // look through the EnemyWaves array, and find one that could've hit
            // us.
            for ( int x = 0; x < _enemyWaves.size(); x++ )
            {
                EnemyWave ew = (EnemyWave)_enemyWaves.get( x );

                if ( Math.abs( ew.distanceTraveled
                    - _myLocation.distance( ew.fireLocation ) ) < 50
                    && Math.round( bulletVelocity( e.getBullet().getPower() ) * 10 ) == Math.round( ew.bulletVelocity * 10 ) )
                // checking the bullet power of the current bullet and from our
                // list to see
                // if it was the same bullet
                {
                    // setting it to the same thing we were looking for
                    hitWave = ew;
                    break;
                }
            }

            // check for waves that may be in a distance of 50 from the source
            // check for the bullet velocity as well and compare it to the
            // bullet we were hit by
            if ( hitWave != null )
            {
                // store the wave away w/ it's respective information
                logHit( hitWave, hitBulletLocation );

                // We can remove this wave now, of course.
                _enemyWaves.remove( _enemyWaves.lastIndexOf( hitWave ) );
            }
        }
    }


    // Prediction Code - Critical!
    public Point2D.Double predictPosition( EnemyWave surfWave, int direction )
    {
        Point2D.Double predictedPosition = (Point2D.Double)_myLocation.clone();
        double predictedVelocity = _robot.getVelocity();
        double predictedHeading = _robot.getHeadingRadians();
        double maxTurning, moveAngle, moveDir;

        int counter = 0;
        boolean intercepted = false;

        do
        {
            moveAngle = wallSmoothing( predictedPosition,
                absoluteBearing( surfWave.fireLocation, predictedPosition )
                    + ( direction * ( Math.PI / 2 ) ),
                direction )
                - predictedHeading;
            moveDir = 1;

            // you're getting the absolute angle of the bot
            // at the same time, we're squared off to the enemy
            // hand it off to wallSmoothing to keep the bot moving at the
            // specified angle

            if ( Math.cos( moveAngle ) < 0 )
            {
                moveAngle += Math.PI;
                // make angle positive, switch directions
                moveDir = -1;
            }

            moveAngle = Utils.normalRelativeAngle( moveAngle );

            // maxTurning is built in like this, you can't turn more than this
            // in one tick
            maxTurning = Math.PI / 720d
                * ( 40d - 3d * Math.abs( predictedVelocity ) );
            predictedHeading = Utils.normalRelativeAngle( predictedHeading
                + limit( -maxTurning, moveAngle, maxTurning ) );

            // if velocity and move direction variables have the same sign
            // you would want to accelerate
            // the 2 factor is for deceleration
            predictedVelocity += ( predictedVelocity * moveDir < 0 ? 2 * moveDir
                : moveDir );
            predictedVelocity = limit( -8, predictedVelocity, 8 );

            predictedPosition = project( predictedPosition,
                predictedHeading,
                predictedVelocity );

            counter++;

            // checking to see if we already caught the wave
            if ( predictedPosition.distance( surfWave.fireLocation ) < surfWave.distanceTraveled
                + ( counter * surfWave.bulletVelocity )
                + surfWave.bulletVelocity )
            {
                intercepted = true;
            }
        } while ( !intercepted && counter < 500 );

        return predictedPosition;
    }


    public double checkDanger( EnemyWave surfWave, int direction )
    {
        // predict position of being intercepted by wave
        int index = getFactorIndex( surfWave, predictPosition( surfWave,
            direction ) );

        return _surfStats[index];
    }


    public void Surf()
    {
        EnemyWave surfWave = getClosestSurfableWave();

        if ( surfWave == null )
        {
            return;
        }

        // choose the safer direction to orbit (left, right)
        double dangerLeft = checkDanger( surfWave, -1 );
        double dangerRight = checkDanger( surfWave, 1 );

        double goAngle = absoluteBearing( surfWave.fireLocation, _myLocation );

        // just move into the safer direction XD
        if ( dangerLeft < dangerRight )
        {
            goAngle = wallSmoothing( _myLocation, goAngle - ( Math.PI / 2 ), -1 );
        }
        else
        {
            goAngle = wallSmoothing( _myLocation, goAngle + ( Math.PI / 2 ), 1 );
        }

        setBackAsFront( this._robot, goAngle );
    }


    // inner class
    class EnemyWave
    {
        Point2D.Double fireLocation;

        long fireTime;

        double bulletVelocity, directAngle, distanceTraveled;

        int direction;


        public EnemyWave()
        {
        }
    }


    // WallSmoothing
    public double wallSmoothing(
        Point2D.Double botLocation,
        double angle,
        int orientation )
    {
        while ( !_fieldRect.contains( project( botLocation, angle, 160 ) ) )
        {
            angle += orientation * 0.05;
        }
        return angle;
    }


    // helps in figuring out future position of enemy bot
    public static Point2D.Double project(
        Point2D.Double sourceLocation,
        double angle,
        double length )
    {
        return new Point2D.Double( sourceLocation.x + Math.sin( angle )
            * length, sourceLocation.y + Math.cos( angle ) * length );
    }


    public static double absoluteBearing(
        Point2D.Double source,
        Point2D.Double target )
    {
        return Math.atan2( target.x - source.x, target.y - source.y );
    }


    public static double limit( double min, double value, double max )
    {
        return Math.max( min, Math.min( value, max ) );
    }


    public static double bulletVelocity( double power )
    {
        return ( 20D - ( 3D * power ) );
    }


    // an angle in which we ought to move away from quickest
    public static double maxEscapeAngle( double velocity )
    {
        return Math.asin( 8.0 / velocity );
    }


    // method essentially calculates shortest turning angle
    public static void setBackAsFront( AdvancedRobot robot, double goAngle )
    {
        double angle = Utils.normalRelativeAngle( goAngle
            - robot.getHeadingRadians() );
        if ( Math.abs( angle ) > ( Math.PI / 2 ) )
        {
            if ( angle < 0 )
            {
                robot.setTurnRightRadians( Math.PI + angle );
            }
            else
            {
                robot.setTurnLeftRadians( Math.PI - angle );
            }
            robot.setBack( 100 );
        }
        else
        {
            if ( angle < 0 )
            {
                robot.setTurnLeftRadians( -1 * angle );
            }
            else
            {
                robot.setTurnRightRadians( angle );
            }
            robot.setAhead( 100 );
        }
    }


    public void onPaint( java.awt.Graphics2D g )
    {
        g.setColor( java.awt.Color.ORANGE );
        for ( int i = 0; i < _enemyWaves.size(); i++ )
        {
            EnemyWave w = (EnemyWave)( _enemyWaves.get( i ) );
            Point2D.Double center = w.fireLocation;

            int radius = (int)w.distanceTraveled;

            if ( radius - 40 < center.distance( _myLocation ) )
                g.drawOval( (int)( center.x - radius ),
                    (int)( center.y - radius ),
                    radius * 2,
                    radius * 2 );
        }
    }
}