package tjk.offense;

import java.util.ArrayList;
import java.util.Random;

import org.bolson.vote.CondorcetRTB;

import robocode.util.Utils;
import tjk.deBroglie;
import tjk.universe.BracketHist;
import tjk.universe.GFBracket;
import tjk.universe.Universe;
import tjk.universe.Wave;
import tjk.universe.WaveStats;
import tjk.universe.BracketHist.HistType;
import tjk.utils.HitTracker;
import tjk.utils.tjkUtil;

/**
 * VirtualGunManager - a class by tkiesel
 * Copyright (c) 2012 Tom Kiesel (Tkiesel @ Robowiki)
 * 
 * This software is provided 'as-is', without any express or implied warranty. In no event will the authors be held liable for any damages arising from the use of this software.
 * 
 * Permission is granted to anyone to use this software for any purpose, including commercial applications, and to alter it and redistribute it freely, subject to the following restrictions:
 * 
 *     1. The origin of this software must not be misrepresented; you must not claim that you wrote the original software. 
 *        If you use this software in a product, an acknowledgment in the product documentation would be appreciated but is not required.
 * 
 *     2. Altered source versions must be plainly marked as such, and must not be misrepresented as being the original software.
 * 
 *     3. This notice may not be removed or altered from any source distribution.
 * 
 */

public class VirtualGunManager {
	
	private static final int NUM_GUNS = 8;
	
	public static final double MAIN_REAL_WAVE_WEIGHT = 9.5;
	
	public static final double ADAPT_REAL_WAVE_WEIGHT = 20.0;
	
	private static final double SHORT_ADAPT_GUN_HALFLIFE = 2.0;
	
	private static final double LONG_ADAPT_GUN_HALFLIFE = 14.0;
	
	//private static final double SIMPLE_GUN_PENALTY = 0.65;
	
	//private static double[] STATIC_VOTE = {1.0, 0.75, 0.0, 0.0};
	//private static double[] STATIC_VOTE = {1.0, 0.5, 0.25};
	
	//private static double[] ANTI_SIMPLE_VOTE = {1.0, 0.99999, 0.05, 0.0};
	
	private WaveStats stats = null;
	
	private CircularGun circleGun = null;
	
	private HitTracker[] gunStats = new HitTracker[NUM_GUNS];
	
	private double[] SHORT_HL_ARRAY = null;
	
	private double[] LONG_HL_ARRAY = null;
	
	private Random randGen = null;
	
	private int[] gunUses = new int[NUM_GUNS];
	
	private int lastUsedGun = 0;
	
	public VirtualGunManager(Universe u, WaveStats s)
	{
		this.circleGun = new CircularGun(u);
		this.stats = s;
		this.SHORT_HL_ARRAY = tjkUtil.halfLifeArray(SHORT_ADAPT_GUN_HALFLIFE);
		this.LONG_HL_ARRAY = tjkUtil.halfLifeArray(LONG_ADAPT_GUN_HALFLIFE);
		randGen = new Random();
		for ( int i = 0; i < NUM_GUNS; i++)
		{
			gunStats[i] = new HitTracker();
		}
	}
	
	/**
	 *  Workhorse Virtual gun method.
	 *  
	 *  Evaluate all virtual guns for this wave.
	 *  Tag wave with all fire angles. 
	 *  Choose best fire angle.
	 *  
	 *  Return absolute fire angle.
	 *  
	 */
	public double tagAndShoot(Wave w)
	{
		// Get all gun firing angles.	
		ArrayList<ArrayList<GFBracket>> kNN = stats.getNearestGFBrackets(w, closestK(), true);
		
		ArrayList<GFBracket> noCullKNN = kNN.get(0);
		ArrayList<GFBracket> culledKNN = kNN.get(1);
		
		BracketHist mainBH = new BracketHist(HistType.TARGETING, noCullKNN, MAIN_REAL_WAVE_WEIGHT, 1.0, false, BracketHist.nullB);
		BracketHist mainCullBH = new BracketHist(HistType.TARGETING, culledKNN, MAIN_REAL_WAVE_WEIGHT, 1.0, false, BracketHist.nullB);
		
		BracketHist mainRollBH = new BracketHist(HistType.TARGETING, noCullKNN, MAIN_REAL_WAVE_WEIGHT, 1.0, true, LONG_HL_ARRAY );
		BracketHist mainRollCullBH = new BracketHist(HistType.TARGETING, culledKNN, MAIN_REAL_WAVE_WEIGHT, 1.0, true, LONG_HL_ARRAY );
		
		BracketHist adaptBH = new BracketHist(HistType.TARGETING, noCullKNN, ADAPT_REAL_WAVE_WEIGHT, 1.0, true, SHORT_HL_ARRAY );
		BracketHist adaptCullBH = new BracketHist(HistType.TARGETING, culledKNN, ADAPT_REAL_WAVE_WEIGHT, 1.0, true, SHORT_HL_ARRAY );
		
		BracketHist virtualBH = new BracketHist(HistType.TARGETING, noCullKNN, 1.0, 1.0, false, BracketHist.nullB);
		BracketHist virtualCullBH = new BracketHist(HistType.TARGETING, culledKNN, 1.0, 1.0, true, LONG_HL_ARRAY );
		
		// Main Gun
		double gun0 = Utils.normalAbsoluteAngle( w.GFToAbs(mainBH.maxAngle()) );
		// Culled Main Gun
		double gun1 = Utils.normalAbsoluteAngle( w.GFToAbs(mainCullBH.maxAngle()) );
		// Rolling Main Gun
		double gun2 = Utils.normalAbsoluteAngle( w.GFToAbs(mainRollBH.maxAngle()) );
		// Culled Rolling Main Gun
		double gun3 = Utils.normalAbsoluteAngle( w.GFToAbs(mainRollCullBH.maxAngle()) );
		// Adaptive Gun
		double gun4 = Utils.normalAbsoluteAngle( w.GFToAbs(adaptBH.maxAngle()) );
		// Culled Adaptive Gun
		double gun5 = Utils.normalAbsoluteAngle( w.GFToAbs(adaptCullBH.maxAngle()) );
		// Virtual waves equal to Real
		double gun6 = Utils.normalAbsoluteAngle( w.GFToAbs(virtualBH.maxAngle()) );
		// Culled Virtual waves equal to Real
		double gun7 = Utils.normalAbsoluteAngle( w.GFToAbs(virtualCullBH.maxAngle()) );
		// HOT
		//double gun4 = Utils.normalAbsoluteAngle( w.GFToAbs(0.0) );
		// Random Targeting
		//double gun5 = randomShot(w);
		// Circular Targeting
		//double gun6 = circleShot(w.getPower());
				
		double[] gunGuesses = {gun0,gun1,gun2,gun3,gun4,gun5,gun6,gun7};
		
		// Load firing angles to wave.
		w.loadGunGuesses(gunGuesses);
			
		int gunToUse = 0;
		
		if ( deBroglie.isASTC )
		{
			gunToUse = 1;
		}
		else if ( Universe.getRoundNum() >= 2 )
		{
			
			// Compile the gun vote for this kNN
			boolean useNowVote = false;
			double[] rightNowGunVotesShort = new double[NUM_GUNS];
			double[] rightNowGunVotesLong = new double[NUM_GUNS];
			int timeNow = Universe.getRoundNum();
			for (int i = 0; i < noCullKNN.size(); i++)
			{
				GFBracket b = noCullKNN.get(i);
				if ( b.real )
				{
					if ( b.gunHits.length == gunGuesses.length )
					{
						for (int j = 0; j < gunGuesses.length; j++ )
						{
							rightNowGunVotesShort[j] += SHORT_HL_ARRAY[timeNow - b.round] * b.gunHits[j];
							rightNowGunVotesLong[j] += LONG_HL_ARRAY[timeNow - b.round] * b.gunHits[j];
							if ( !useNowVote && b.gunHits[j] > 0.0 )
							{
								useNowVote = true;
							}
						}
					}
				}
			}
			
			
			
			//Get votes from HitTrackers.
			double[] shortVote = getShortVote();
			double[] mediumVote = getMediumVote();
			double[] longVote = getLongVote();
			double[] totalVote = getTotalVote();
			
			// Make vote object.
			CondorcetRTB vote = new CondorcetRTB(NUM_GUNS);
			
			//Load votes.
			
			if ( useNowVote )
			{
				//Slight penalty to simple guns.
				//rightNowGunVotesShort[4] = SIMPLE_GUN_PENALTY*rightNowGunVotesShort[4];
				//rightNowGunVotesLong[4] = SIMPLE_GUN_PENALTY*rightNowGunVotesLong[4];
				//rightNowGunVotesShort[5] = SIMPLE_GUN_PENALTY*rightNowGunVotesShort[5];
				//rightNowGunVotesLong[5] = SIMPLE_GUN_PENALTY*rightNowGunVotesLong[5];
				//rightNowGunVotesShort[6] = 0.9*SIMPLE_GUN_PENALTY*rightNowGunVotesShort[6];
				//rightNowGunVotesLong[6] = 0.9*SIMPLE_GUN_PENALTY*rightNowGunVotesLong[6];
								
				//Normalize.
				rightNowGunVotesShort = normalizeVote(rightNowGunVotesShort);
				rightNowGunVotesLong = normalizeVote(rightNowGunVotesLong);
				
				//Load.
				vote.voteRating(rightNowGunVotesShort);
				vote.voteRating(rightNowGunVotesLong);
			}
			
			
			vote.voteRating(shortVote);
			vote.voteRating(mediumVote);
			vote.voteRating(mediumVote);
			vote.voteRating(longVote);
			vote.voteRating(totalVote);
			//vote.voteRating(STATIC_VOTE);
			
			// Decide winner.
			gunToUse = vote.getWinners()[0];
			
			//System.out.println("Using gun " + gunToUse);
		}
		
		lastUsedGun = gunToUse;
		
		// return best angle.
		return gunGuesses[gunToUse];
	}
	
	public void gunWasUsed()
	{
		gunUses[lastUsedGun]++;
	}
	
	@SuppressWarnings("unused")
	private double randomShot(Wave w) {
		double angle = 2.0*randGen.nextDouble() - 1.0;
		return Utils.normalAbsoluteAngle( w.GFToAbs(angle) );
	}

	private int closestK()
	{
		return (int)Math.round(Math.min(Math.max(2,stats.getSize()/20),150));
	}
	
	public double circleShot(double firepower)
	{
		return circleGun.shoot(firepower);
	}
	
	public double[] getShortVote()
	{
		double[] answer = new double[NUM_GUNS];
		for ( int i = 0; i < NUM_GUNS; i++)
		{
			answer[i] = gunStats[i].rollingHitRateShort();
		}
		return normalizeVote(answer);
	}
	
	public double[] getMediumVote()
	{
		double[] answer = new double[NUM_GUNS];
		for ( int i = 0; i < NUM_GUNS; i++)
		{
			answer[i] = gunStats[i].rollingHitRateMedium();
		}
		return normalizeVote(answer);
	}
	
	public double[] getLongVote()
	{
		double[] answer = new double[NUM_GUNS];
		for ( int i = 0; i < NUM_GUNS; i++)
		{
			answer[i] = gunStats[i].rollingHitRateLong();
		}
		return normalizeVote(answer);
	}
	
	public double[] getTotalVote()
	{
		double[] answer = new double[NUM_GUNS];
		for ( int i = 0; i < NUM_GUNS; i++)
		{
			answer[i] = gunStats[i].totalHitRate();
		}
		return normalizeVote(answer);
	}
	
	/**
	 * Normalize votes so the total vote is 1.0
	 */
	public double[] normalizeVote(double[] in)
	{
		
		double[] answer = new double[in.length];
		double TOTAL = 0.0;
		for ( int i = 0; i < in.length; i++)
		{
			TOTAL += in[i];
		}
		
		for ( int i = 0; i < in.length; i++)
		{
			answer[i] = in[i] / TOTAL;
		}
		
		return answer;
		
	}
	
	public void loadWave(Wave w)
	{
		if ( w.getState() == Wave.DEAD )
		{
			double[] gunHits = w.gunHits();
			if ( gunHits.length == NUM_GUNS )
			{
				for ( int i = 0; i < NUM_GUNS; i++ )
				{
					this.gunStats[i].addHit(gunHits[i]);
				}
			}
		}
	}
	
	public void onRoundEnded()
	{
		System.out.println("Overall gun averages: ");
		for ( int i = 0; i < NUM_GUNS; i++ )
		{
			System.out.println("Gun " + i + ": Used: " + gunUses[i] + ", Rating: " + gunStats[i].totalHitRate() );
		}
	}

}
