/*
 * Copyright (c) 2004 Marcin Fuszara
 */  
package fushi.PvP1;
import robocode.*;
import java.util.LinkedList;

/* This class is responsible for storing data about a single enemy
 * for a single instance of this class;
 */
public class Enemy
{
    
   /* a single record of data stored in the log
    */
   public class Data
   {
      public long time = -1;     //< frame in which the data was gathered (-1 means that data is not valid)
      public Vector2d position;  //< robot position
      public Vector2d velocity;  //< robot velocity vector
      public double speed;       //< robot velocity scalar
      public double heading;     //< robot heading
      public double energy;      //< robot energy
   }
   
   // length of the history log for robot
   static final int maxLogLength = 8;
   // the robot history log
   LinkedList log;
   
   // the robot from which time is get
   AdvancedRobot robot;
   
   // a pattern matcher for this enemy
   public PatternMatcher patternMatcher;
   
   /* for the sake of construction an Enemy instance, provide
    * the robot class which tracks this enemy
    * and desired number of data elements stored in robot history log;
    */
   public Enemy( AdvancedRobot robot )
   {
      this.robot = robot;
      // create a log for linear prediction
      log = new LinkedList();
      // create a pattern matcher for this robot
      patternMatcher = new PatternMatcher( this );
   }
   
   /* run this method at the beginning of a new round;
    * the log will be cleared, but the pattern matcher
    * will leave its data collected unchanged;
    */
   public void newRound()
   {
      log = new LinkedList();
   }
   
   /* this method should be called for each event concerning
    * this enemy;
    * the event is used to update the robot history log
    */
   public void onScannedRobot( ScannedRobotEvent e ) 
   {
      // calculate enemy relative angle from current robot position
      double enemyRelativeAngle = robot.getHeadingRadians() + e.getBearingRadians();
      // calculate enemy position
      Vector2d enemyPosition = new Vector2d();
      enemyPosition.setFromAngleDistance( enemyRelativeAngle, e.getDistance() );
      Vector2d robotPosition = new Vector2d( robot.getX(), robot.getY() );
      enemyPosition.add( robotPosition );
      
      // calculate enemy velocity vector
      Vector2d enemyVelocity = new Vector2d();
      enemyVelocity.setFromAngleDistance( e.getHeadingRadians(), e.getVelocity() );
      
      // create a single data element
      Data data = new Data();
      data.energy = e.getEnergy();
      data.heading = e.getHeadingRadians();
      data.position = enemyPosition;
      data.speed = e.getVelocity();
      data.time = e.getTime();
      data.velocity = enemyVelocity;
      
      // add data to log and assure that it is not too long
      log.add( data );
      if ( log.size() > maxLogLength )
         log.removeFirst();
         
      // base on the data collected update the pattern log in the matcher
      patternMatcher.updatePatternLog();
   }
   
   /* returns the current length of the robot history log
    */
   public int getLogLength()
   {
      return log.size();
   }
   
   /* returns the maximum length of the robot history log
    * (set at construction)
    */
   public int getMaxLogLength()
   {
      return maxLogLength;
   }
   
   /* get a single piece of data logged for this enemy;
    * \param index 0 is the newest data, (getLogLength()-1) is the oldest data;
    * \return if no data with specified index exists returns null;
    */
   public Data getData( int index )
   {
      if ( index >= log.size() )
         return null;
      return (Data)log.get( log.size() - 1 - index );
   }
   
   /* returns true if the last data collected is valid;
    * it means that with getData( 0 ) you get the most recent data
    * possible
    */
   public boolean isUpToDate()
   {
      if ( log.size() == 0 )
         return false;
      if ( ((Data)log.getLast()).time == robot.getTime() )
         return true;
      return false;
   }
   
   /* returns true if the robot data is up-to-date and all the 
    * data collected in the log comes from consecutive frames;
    * so if the robot was lost during the last maxLogLength
    * frames, the enemy is not locked;
    */
   public boolean isLocked()
   {
      // ensure that the last data is up-to-date
      if ( !isUpToDate() )
         return false;
         
      // check if there were no interruptions in robot data
      for ( int i = log.size() - 2; i >= 0; --i )
         if ( ((Data)log.get( i+1 )).time - ((Data)log.get( i )).time != 1 )
            return false;
      
      return true;
   }
   
   /* returns true if the robot data is up-to-date and given number of 
    * data records collected in the log comes from consecutive frames;
    * \param numFrames is clipped to range 1..maxLogLength;
    */
   public boolean isLocked( int numFrames )
   {
      // ensure that the last data is up-to-date
      if ( !isUpToDate() )
         return false;
      
      if ( numFrames < 1 )
         numFrames = 1;
      else if ( numFrames > maxLogLength )
         numFrames = maxLogLength;
         
      if ( numFrames > log.size() )
         return false;       // not enough data in the log
         
      // check if there were no interruptions in robot data
      for ( int i = numFrames - 2; i >= 0; --i )
         if ( ((Data)log.get( i+1 )).time - ((Data)log.get( i )).time != 1 )
            return false;
      
      return true;
   }
   
}